/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "globus_rls_client.h"
#include "version.h"
#include <stdio.h>
#include <string.h>

static globus_module_descriptor_t	*modules[] = {
  GLOBUS_COMMON_MODULE,
  GLOBUS_IO_MODULE,
  GLOBUS_RLS_CLIENT_MODULE,
};
#define NMODS	(sizeof(modules) / sizeof(globus_module_descriptor_t *))

static void			cleanexit(int s);
static void			usage(char *prog);
static globus_rls_handle_t	*h = NULL;
static char			*dn = "RLS=stats, Mds-Vo-name=local, o=grid";

int
main(int argc, char **argv)

{
  int			rc;
  globus_result_t	r;
  char			*server;
  char			errmsg[MAXERRMSG];
  int			i;
  globus_rls_stats_t	rlsstats;
  char			*certfile = NULL;
  char			*keyfile = NULL;
  extern char		*optarg;
  extern int		optind;
  globus_list_t		*list;
  globus_list_t		*p;
  globus_rls_rli_info_t	*rliinfo;
  globus_rls_lrc_info_t	*senderinfo;
  
  while ((i = getopt(argc, argv, "c:d:k:v")) != -1)
    switch (i) {
      case 'c':	certfile = optarg;
		break;
      case 'd': dn = optarg;
		break;
      case 'k':	keyfile = optarg;
		break;
      case 'v':	printf("Version: %d.%d\n", MAJOR, MINOR);
		exit(0);
      default:	usage(argv[0]);
    }
  if (optind >= argc)
    server = "rls://localhost";
  else
    server = argv[optind];

  for (i = 0; i < NMODS; i++)
    if ((rc = globus_module_activate(modules[i])) != GLOBUS_SUCCESS) {
      fprintf(stderr, "Unable to activate module %d", i);
      cleanexit(1);
    }

  if (certfile || keyfile)
    globus_rls_client_certificate(certfile, keyfile);
  if ((r = globus_rls_client_connect(server, &h)) != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(r, NULL, errmsg, MAXERRMSG, GLOBUS_FALSE);
    fprintf(stderr, "%s: %s\n", server, errmsg);
    cleanexit(1);
  }

  if ((r = globus_rls_client_stats(h, &rlsstats)) != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(r, NULL, errmsg, MAXERRMSG, GLOBUS_FALSE);
    fprintf(stderr, "%s: %s\n", server, errmsg);
    cleanexit(1);
  }

  printf("dn: %s\n", dn);
  printf("objectclass: RLS\n");
  printf("RLS-URL: %s\n", server);
  printf("RLS-Uptime: %d\n", (int) rlsstats.uptime);
  printf("RLS-LRC-Server: %d\n", rlsstats.flags & RLS_LRCSERVER ? 1 : 0);
  printf("RLS-RLI-Server: %d\n", rlsstats.flags & RLS_RLISERVER ? 1 : 0);
  if (rlsstats.flags & RLS_LRCSERVER) {
    if (rlsstats.flags & RLS_SNDLFNLIST)
      printf("RLS-LRC-UpdateMethod: LFNList\n");
    if (rlsstats.flags & RLS_SNDBLOOMFILTER)
      printf("RLS-LRC-UpdateMethod: Bloomfilter\n");
    printf("RLS-LRC-BloomFilterUpdateInterval: %d\n",
	   rlsstats.lrc_bloomfilterui);
    printf("RLS-LRC-LFNListUpdateInterval: %d\n", rlsstats.lrc_lfnlistui);
    if (globus_rls_client_lrc_rli_list(h, &list) == GLOBUS_SUCCESS) {
      for (p = list; p; p = globus_list_rest(p)) {
	rliinfo = globus_list_first(p);
	if (rliinfo->flags & FRLI_BLOOMFILTER)
	  printf("RLS-LRC-UpdatesBloomFilter: %s\n", rliinfo->url);
	else
	  printf("RLS-LRC-UpdatesLFNList: %s\n", rliinfo->url);
      }
    }
    printf("RLS-LRC-NumLFN: %d\n", rlsstats.lrc_numlfn);
    printf("RLS-LRC-NumPFN: %d\n", rlsstats.lrc_numpfn);
    printf("RLS-LRC-NumMAP: %d\n", rlsstats.lrc_nummap);
  }
  if (rlsstats.flags & RLS_RLISERVER) {
    if (globus_rls_client_rli_sender_list(h, &list) == GLOBUS_SUCCESS) {
      for (p = list; p; p = globus_list_rest(p)) {
	senderinfo = globus_list_first(p);
	printf("RLS-RLI-UpdatedBy: %s\n", senderinfo->url);
      }
    }
    if (rlsstats.flags & RLS_RCVBLOOMFILTER)
      printf("RLS-RLI-UpdateMethod: BloomFilter\n");
    else if (rlsstats.flags & RLS_RCVLFNLIST) {
      printf("RLS-RLI-UpdateMethod: LFNList\n");
      printf("RLS-RLI-NumLFN: %d\n", rlsstats.rli_numlfn);
      printf("RLS-RLI-NumLRC: %d\n", rlsstats.rli_numlrc);
      printf("RLS-RLI-NumSENDER: %d\n", rlsstats.rli_numsender);
      printf("RLS-RLI-NumMAP: %d\n", rlsstats.rli_nummap);
    }
  }

  cleanexit(0);
  exit(0);		/* Stop compiler warning	*/
}

static void
cleanexit(int s)

{
  int	i;

  if (h)
    globus_rls_client_close(h);
  for (i = NMODS - 1; i >= 0; i--)
    globus_module_deactivate(modules[i]);
  exit(s);
}

static void
usage(char *prog)

{
  fprintf(stderr, "Usage: %s [-c certfile] [-d distinguished-name] [-k keyfile] [rls-url] [-v]\n", prog);
  exit(1);
}
