# Copyright © 2007-2009 Raphaël Hertzog <hertzog@debian.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.

=encoding utf8

=head1 NAME

Dpkg::Control - parse and manipulate official control-like information

=head1 DESCRIPTION

The Dpkg::Control object is a smart version of L<Dpkg::Control::Hash>.
It associates a type to the control information. That type can be
used to know what fields are allowed and in what order they must be
output.

The types are constants that are exported by default. Here's the full
list:

=over 4

=item CTRL_UNKNOWN

This type is the default type, it indicates that the type of control
information is not yet known.

=item CTRL_TMPL_SRC

Corresponds to the first source package stanza in a F<debian/control> file in
a Debian source package.

=item CTRL_TMPL_PKG

Corresponds to subsequent binary package stanza in a F<debian/control> file
in a Debian source package.

=item CTRL_REPO_RELEASE

Corresponds to a F<Release> file in a repository.

=item CTRL_REPO_SRC

Corresponds to a stanza in a F<Sources> file of a source package
repository.

=item CTRL_REPO_PKG

Corresponds to a stanza in a F<Packages> file of a binary package
repository.

=item CTRL_DSC

Corresponds to a .dsc file of a Debian source package.

=item CTRL_DEB

Corresponds to the F<control> file generated by dpkg-gencontrol
(F<DEBIAN/control>) and to the same file inside .deb packages.

=item CTRL_FILE_BUILDINFO

Corresponds to a .buildinfo file.

=item CTRL_FILE_CHANGES

Corresponds to a .changes file.

=item CTRL_FILE_VENDOR

Corresponds to a vendor file in $Dpkg::CONFDIR/origins/.

=item CTRL_FILE_STATUS

Corresponds to a stanza in dpkg's F<status> file ($Dpkg::ADMINDIR/status).

=item CTRL_CHANGELOG

Corresponds to the output of dpkg-parsechangelog.

=item CTRL_COPYRIGHT_HEADER

Corresponds to the header stanza in a F<debian/copyright> file in
machine readable format.

=item CTRL_COPYRIGHT_FILES

Corresponds to a files stanza in a F<debian/copyright> file in
machine readable format.

=item CTRL_COPYRIGHT_LICENSE

Corresponds to a license stanza in a F<debian/copyright> file in
machine readable format.

=item CTRL_TESTS

Corresponds to a source package tests control file in F<debian/tests/control>.

=item CTRL_INFO_SRC

Alias for B<CTRL_TMPL_SRC>.

=item CTRL_INFO_PKG

Alias for B<CTRL_TMPL_PKG>.

=item CTRL_PKG_SRC

Alias for B<CTRL_DSC>.

=item CTRL_PKG_DEB

Alias for B<CTRL_DEB>.

=item CTRL_INDEX_SRC

Alias for B<CTRL_REPO_SRC>.

=item CTRL_INDEX_PKG

Alias for B<CTRL_REPO_PKG>.

=back

=cut

package Dpkg::Control 1.04;

use strict;
use warnings;

our @EXPORT = qw(
    CTRL_UNKNOWN
    CTRL_TMPL_SRC
    CTRL_TMPL_PKG
    CTRL_REPO_RELEASE
    CTRL_REPO_SRC
    CTRL_REPO_PKG
    CTRL_DSC
    CTRL_DEB
    CTRL_FILE_BUILDINFO
    CTRL_FILE_CHANGES
    CTRL_FILE_VENDOR
    CTRL_FILE_STATUS
    CTRL_CHANGELOG
    CTRL_COPYRIGHT_HEADER
    CTRL_COPYRIGHT_FILES
    CTRL_COPYRIGHT_LICENSE
    CTRL_TESTS

    CTRL_INFO_SRC
    CTRL_INFO_PKG
    CTRL_PKG_SRC
    CTRL_PKG_DEB
    CTRL_INDEX_SRC
    CTRL_INDEX_PKG
);

use Exporter qw(import);

use Dpkg::Gettext;
use Dpkg::ErrorHandling;
use Dpkg::Control::Types;
use Dpkg::Control::Hash;
use Dpkg::Control::Fields;

use parent qw(Dpkg::Control::Hash);

=head1 METHODS

All the methods of L<Dpkg::Control::Hash> are available. Those listed below
are either new or overridden with a different behavior.

=over 4

=item $c = Dpkg::Control->new(%opts)

If the "type" option is given, it's used to setup default values
for other options. See set_options() for more details.

=cut

sub new {
    my ($this, %opts) = @_;
    my $class = ref($this) || $this;

    my $self = Dpkg::Control::Hash->new();
    bless $self, $class;
    $self->set_options(%opts);

    return $self;
}

=item $c->set_options(%opts)

Changes the value of one or more options. If the "type" option is changed,
it is used first to define default values for others options. The option
"allow_pgp" is set to 1 for CTRL_DSC, CTRL_FILE_CHANGES and
CTRL_REPO_RELEASE and to 0 otherwise. The option "drop_empty" is set to 0
for CTRL_TMPL_SRC and CTRL_TMPL_PKG and to 1 otherwise. The option "name"
is set to a textual description of the type of control information.

The output order is also set to match the ordered list returned by
Dpkg::Control::Fields::field_ordered_list($type).

=cut

sub set_options {
    my ($self, %opts) = @_;
    if (exists $opts{type}) {
        my $t = $opts{type};
        $$self->{allow_pgp} = ($t & (CTRL_DSC | CTRL_FILE_CHANGES | CTRL_REPO_RELEASE)) ? 1 : 0;
        $$self->{drop_empty} = ($t & (CTRL_TMPL_SRC | CTRL_TMPL_PKG)) ?  0 : 1;
        if ($t == CTRL_TMPL_SRC) {
            $$self->{name} = g_('source package stanza of template control file');
        } elsif ($t == CTRL_TMPL_PKG) {
            $$self->{name} = g_('binary package stanza of template control file');
        } elsif ($t == CTRL_CHANGELOG) {
            $$self->{name} = g_('parsed version of changelog');
        } elsif ($t == CTRL_COPYRIGHT_HEADER) {
            $$self->{name} = g_('header stanza of copyright file');
        } elsif ($t == CTRL_COPYRIGHT_FILES) {
            $$self->{name} = g_('files stanza of copyright file');
        } elsif ($t == CTRL_COPYRIGHT_HEADER) {
            $$self->{name} = g_('license stanza of copyright file');
        } elsif ($t == CTRL_TESTS) {
            $$self->{name} = g_('source package tests control file');
        } elsif ($t == CTRL_REPO_RELEASE) {
            $$self->{name} = sprintf(g_("repository's %s file"), 'Release');
        } elsif ($t == CTRL_REPO_SRC) {
            $$self->{name} = sprintf(g_("stanza in repository's %s file"), 'Sources');
        } elsif ($t == CTRL_REPO_PKG) {
            $$self->{name} = sprintf(g_("stanza in repository's %s file"), 'Packages');
        } elsif ($t == CTRL_DSC) {
            $$self->{name} = g_('source package control file');
        } elsif ($t == CTRL_DEB) {
            $$self->{name} = g_('binary package control file');
        } elsif ($t == CTRL_FILE_BUILDINFO) {
            $$self->{name} = g_('build information file');
        } elsif ($t == CTRL_FILE_CHANGES) {
            $$self->{name} = g_('upload changes control file');
        } elsif ($t == CTRL_FILE_VENDOR) {
            $$self->{name} = g_('vendor file');
        } elsif ($t == CTRL_FILE_STATUS) {
            $$self->{name} = g_("stanza in dpkg's status file");
        }
        $self->set_output_order(field_ordered_list($opts{type}));
    }

    # Options set by the user override default values
    $$self->{$_} = $opts{$_} foreach keys %opts;
}

=item $c->get_type()

Returns the type of control information stored. See the type parameter
set during new().

=cut

sub get_type {
    my $self = shift;
    return $$self->{type};
}

=back

=head1 CHANGES

=head2 Version 1.04 (dpkg 1.22.2)

New types: CTRL_TMPL_SRC, CTRL_TMPL_PKG, CTRL_REPO_SRC, CTRL_REPO_PKG,
CTRL_DSC, CTRL_DEB.

=head2 Version 1.03 (dpkg 1.18.11)

New type: CTRL_FILE_BUILDINFO.

=head2 Version 1.02 (dpkg 1.18.8)

New type: CTRL_TESTS.

=head2 Version 1.01 (dpkg 1.18.5)

New types: CTRL_REPO_RELEASE, CTRL_COPYRIGHT_HEADER, CTRL_COPYRIGHT_FILES,
CTRL_COPYRIGHT_LICENSE.

=head2 Version 1.00 (dpkg 1.15.6)

Mark the module as public.

=cut

1;
