# Copyright 2015 Google Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

require 'date'
require 'google/apis/core/base_service'
require 'google/apis/core/json_representation'
require 'google/apis/core/hashable'
require 'google/apis/errors'

module Google
  module Apis
    module CloudbuildV1
      
      # A build resource in the Container Builder API.
      # At a high level, a Build describes where to find source code, how to build
      # it (for example, the builder image to run on the source), and what tag to
      # apply to the built image when it is pushed to Google Container Registry.
      # Fields can include the following variables which will be expanded when the
      # build is created:
      # - $PROJECT_ID: the project ID of the build.
      # - $BUILD_ID: the autogenerated ID of the build.
      # - $REPO_NAME: the source repository name specified by RepoSource.
      # - $BRANCH_NAME: the branch name specified by RepoSource.
      # - $TAG_NAME: the tag name specified by RepoSource.
      # - $REVISION_ID or $COMMIT_SHA: the commit SHA specified by RepoSource or
      # resolved from the specified branch or tag.
      # - $SHORT_SHA: first 7 characters of $REVISION_ID or $COMMIT_SHA.
      class Build
        include Google::Apis::Core::Hashable
      
        # The ID of the BuildTrigger that triggered this build, if it was
        # triggered automatically.
        # @OutputOnly
        # Corresponds to the JSON property `buildTriggerId`
        # @return [String]
        attr_accessor :build_trigger_id
      
        # Time at which the request to create the build was received.
        # @OutputOnly
        # Corresponds to the JSON property `createTime`
        # @return [String]
        attr_accessor :create_time
      
        # Time at which execution of the build was finished.
        # The difference between finish_time and start_time is the duration of the
        # build's execution.
        # @OutputOnly
        # Corresponds to the JSON property `finishTime`
        # @return [String]
        attr_accessor :finish_time
      
        # Unique identifier of the build.
        # @OutputOnly
        # Corresponds to the JSON property `id`
        # @return [String]
        attr_accessor :id
      
        # A list of images to be pushed upon the successful completion of all build
        # steps.
        # The images will be pushed using the builder service account's credentials.
        # The digests of the pushed images will be stored in the Build resource's
        # results field.
        # If any of the images fail to be pushed, the build is marked FAILURE.
        # Corresponds to the JSON property `images`
        # @return [Array<String>]
        attr_accessor :images
      
        # URL to logs for this build in Google Cloud Console.
        # @OutputOnly
        # Corresponds to the JSON property `logUrl`
        # @return [String]
        attr_accessor :log_url
      
        # Google Cloud Storage bucket where logs should be written (see
        # [Bucket Name
        # Requirements](https://cloud.google.com/storage/docs/bucket-naming#requirements)
        # ).
        # Logs file names will be of the format `$`logs_bucket`/log-$`build_id`.txt`.
        # Corresponds to the JSON property `logsBucket`
        # @return [String]
        attr_accessor :logs_bucket
      
        # Optional arguments to enable specific features of builds.
        # Corresponds to the JSON property `options`
        # @return [Google::Apis::CloudbuildV1::BuildOptions]
        attr_accessor :options
      
        # ID of the project.
        # @OutputOnly.
        # Corresponds to the JSON property `projectId`
        # @return [String]
        attr_accessor :project_id
      
        # Results describes the artifacts created by the build pipeline.
        # Corresponds to the JSON property `results`
        # @return [Google::Apis::CloudbuildV1::Results]
        attr_accessor :results
      
        # Secrets to decrypt using Cloud KMS.
        # Corresponds to the JSON property `secrets`
        # @return [Array<Google::Apis::CloudbuildV1::Secret>]
        attr_accessor :secrets
      
        # Source describes the location of the source in a supported storage
        # service.
        # Corresponds to the JSON property `source`
        # @return [Google::Apis::CloudbuildV1::Source]
        attr_accessor :source
      
        # Provenance of the source. Ways to find the original source, or verify that
        # some source was used for this build.
        # Corresponds to the JSON property `sourceProvenance`
        # @return [Google::Apis::CloudbuildV1::SourceProvenance]
        attr_accessor :source_provenance
      
        # Time at which execution of the build was started.
        # @OutputOnly
        # Corresponds to the JSON property `startTime`
        # @return [String]
        attr_accessor :start_time
      
        # Status of the build.
        # @OutputOnly
        # Corresponds to the JSON property `status`
        # @return [String]
        attr_accessor :status
      
        # Customer-readable message about the current status.
        # @OutputOnly
        # Corresponds to the JSON property `statusDetail`
        # @return [String]
        attr_accessor :status_detail
      
        # Describes the operations to be performed on the workspace.
        # Corresponds to the JSON property `steps`
        # @return [Array<Google::Apis::CloudbuildV1::BuildStep>]
        attr_accessor :steps
      
        # Substitutions data for Build resource.
        # Corresponds to the JSON property `substitutions`
        # @return [Hash<String,String>]
        attr_accessor :substitutions
      
        # Tags for annotation of a Build. These are not docker tags.
        # Corresponds to the JSON property `tags`
        # @return [Array<String>]
        attr_accessor :tags
      
        # Amount of time that this build should be allowed to run, to second
        # granularity. If this amount of time elapses, work on the build will cease
        # and the build status will be TIMEOUT.
        # Default time is ten minutes.
        # Corresponds to the JSON property `timeout`
        # @return [String]
        attr_accessor :timeout
      
        # Stores timing information for phases of the build. Valid keys are:
        # * BUILD: time to execute all build steps
        # * PUSH: time to push all specified images.
        # * FETCHSOURCE: time to fetch source.
        # If the build does not specify source, or does not specify images,
        # these keys will not be included.
        # @OutputOnly
        # Corresponds to the JSON property `timing`
        # @return [Hash<String,Google::Apis::CloudbuildV1::TimeSpan>]
        attr_accessor :timing
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @build_trigger_id = args[:build_trigger_id] if args.key?(:build_trigger_id)
          @create_time = args[:create_time] if args.key?(:create_time)
          @finish_time = args[:finish_time] if args.key?(:finish_time)
          @id = args[:id] if args.key?(:id)
          @images = args[:images] if args.key?(:images)
          @log_url = args[:log_url] if args.key?(:log_url)
          @logs_bucket = args[:logs_bucket] if args.key?(:logs_bucket)
          @options = args[:options] if args.key?(:options)
          @project_id = args[:project_id] if args.key?(:project_id)
          @results = args[:results] if args.key?(:results)
          @secrets = args[:secrets] if args.key?(:secrets)
          @source = args[:source] if args.key?(:source)
          @source_provenance = args[:source_provenance] if args.key?(:source_provenance)
          @start_time = args[:start_time] if args.key?(:start_time)
          @status = args[:status] if args.key?(:status)
          @status_detail = args[:status_detail] if args.key?(:status_detail)
          @steps = args[:steps] if args.key?(:steps)
          @substitutions = args[:substitutions] if args.key?(:substitutions)
          @tags = args[:tags] if args.key?(:tags)
          @timeout = args[:timeout] if args.key?(:timeout)
          @timing = args[:timing] if args.key?(:timing)
        end
      end
      
      # Metadata for build operations.
      class BuildOperationMetadata
        include Google::Apis::Core::Hashable
      
        # A build resource in the Container Builder API.
        # At a high level, a Build describes where to find source code, how to build
        # it (for example, the builder image to run on the source), and what tag to
        # apply to the built image when it is pushed to Google Container Registry.
        # Fields can include the following variables which will be expanded when the
        # build is created:
        # - $PROJECT_ID: the project ID of the build.
        # - $BUILD_ID: the autogenerated ID of the build.
        # - $REPO_NAME: the source repository name specified by RepoSource.
        # - $BRANCH_NAME: the branch name specified by RepoSource.
        # - $TAG_NAME: the tag name specified by RepoSource.
        # - $REVISION_ID or $COMMIT_SHA: the commit SHA specified by RepoSource or
        # resolved from the specified branch or tag.
        # - $SHORT_SHA: first 7 characters of $REVISION_ID or $COMMIT_SHA.
        # Corresponds to the JSON property `build`
        # @return [Google::Apis::CloudbuildV1::Build]
        attr_accessor :build
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @build = args[:build] if args.key?(:build)
        end
      end
      
      # Optional arguments to enable specific features of builds.
      class BuildOptions
        include Google::Apis::Core::Hashable
      
        # Requested disk size for the VM that runs the build. Note that this is *NOT*
        # "disk free"; some of the space will be used by the operating system and
        # build utilities. Also note that this is the minimum disk size that will be
        # allocated for the build -- the build may run with a larger disk than
        # requested. At present, the maximum disk size is 1000GB; builds that request
        # more than the maximum are rejected with an error.
        # Corresponds to the JSON property `diskSizeGb`
        # @return [Fixnum]
        attr_accessor :disk_size_gb
      
        # LogStreamingOption to define build log streaming behavior to Google Cloud
        # Storage.
        # Corresponds to the JSON property `logStreamingOption`
        # @return [String]
        attr_accessor :log_streaming_option
      
        # Compute Engine machine type on which to run the build.
        # Corresponds to the JSON property `machineType`
        # @return [String]
        attr_accessor :machine_type
      
        # Requested verifiability options.
        # Corresponds to the JSON property `requestedVerifyOption`
        # @return [String]
        attr_accessor :requested_verify_option
      
        # Requested hash for SourceProvenance.
        # Corresponds to the JSON property `sourceProvenanceHash`
        # @return [Array<String>]
        attr_accessor :source_provenance_hash
      
        # SubstitutionOption to allow unmatch substitutions.
        # Corresponds to the JSON property `substitutionOption`
        # @return [String]
        attr_accessor :substitution_option
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @disk_size_gb = args[:disk_size_gb] if args.key?(:disk_size_gb)
          @log_streaming_option = args[:log_streaming_option] if args.key?(:log_streaming_option)
          @machine_type = args[:machine_type] if args.key?(:machine_type)
          @requested_verify_option = args[:requested_verify_option] if args.key?(:requested_verify_option)
          @source_provenance_hash = args[:source_provenance_hash] if args.key?(:source_provenance_hash)
          @substitution_option = args[:substitution_option] if args.key?(:substitution_option)
        end
      end
      
      # BuildStep describes a step to perform in the build pipeline.
      class BuildStep
        include Google::Apis::Core::Hashable
      
        # A list of arguments that will be presented to the step when it is started.
        # If the image used to run the step's container has an entrypoint, these args
        # will be used as arguments to that entrypoint. If the image does not define
        # an entrypoint, the first element in args will be used as the entrypoint,
        # and the remainder will be used as arguments.
        # Corresponds to the JSON property `args`
        # @return [Array<String>]
        attr_accessor :args
      
        # Working directory to use when running this step's container.
        # If this value is a relative path, it is relative to the build's working
        # directory. If this value is absolute, it may be outside the build's working
        # directory, in which case the contents of the path may not be persisted
        # across build step executions, unless a volume for that path is specified.
        # If the build specifies a RepoSource with dir and a step with a dir which
        # specifies an absolute path, the RepoSource dir is ignored for the step's
        # execution.
        # Corresponds to the JSON property `dir`
        # @return [String]
        attr_accessor :dir
      
        # Optional entrypoint to be used instead of the build step image's default
        # If unset, the image's default will be used.
        # Corresponds to the JSON property `entrypoint`
        # @return [String]
        attr_accessor :entrypoint
      
        # A list of environment variable definitions to be used when running a step.
        # The elements are of the form "KEY=VALUE" for the environment variable "KEY"
        # being given the value "VALUE".
        # Corresponds to the JSON property `env`
        # @return [Array<String>]
        attr_accessor :env
      
        # Optional unique identifier for this build step, used in wait_for to
        # reference this build step as a dependency.
        # Corresponds to the JSON property `id`
        # @return [String]
        attr_accessor :id
      
        # The name of the container image that will run this particular build step.
        # If the image is already available in the host's Docker daemon's cache, it
        # will be run directly. If not, the host will attempt to pull the image
        # first, using the builder service account's credentials if necessary.
        # The Docker daemon's cache will already have the latest versions of all of
        # the officially supported build steps
        # ([https://github.com/GoogleCloudPlatform/cloud-builders](https://github.com/
        # GoogleCloudPlatform/cloud-builders)).
        # The Docker daemon will also have cached many of the layers for some popular
        # images, like "ubuntu", "debian", but they will be refreshed at the time you
        # attempt to use them.
        # If you built an image in a previous build step, it will be stored in the
        # host's Docker daemon's cache and is available to use as the name for a
        # later build step.
        # Corresponds to the JSON property `name`
        # @return [String]
        attr_accessor :name
      
        # A list of environment variables which are encrypted using a Cloud KMS
        # crypto key. These values must be specified in the build's secrets.
        # Corresponds to the JSON property `secretEnv`
        # @return [Array<String>]
        attr_accessor :secret_env
      
        # Stores start and end times for a build execution phase.
        # Corresponds to the JSON property `timing`
        # @return [Google::Apis::CloudbuildV1::TimeSpan]
        attr_accessor :timing
      
        # List of volumes to mount into the build step.
        # Each volume will be created as an empty volume prior to execution of the
        # build step. Upon completion of the build, volumes and their contents will
        # be discarded.
        # Using a named volume in only one step is not valid as it is indicative
        # of a mis-configured build request.
        # Corresponds to the JSON property `volumes`
        # @return [Array<Google::Apis::CloudbuildV1::Volume>]
        attr_accessor :volumes
      
        # The ID(s) of the step(s) that this build step depends on.
        # This build step will not start until all the build steps in wait_for
        # have completed successfully. If wait_for is empty, this build step will
        # start when all previous build steps in the Build.Steps list have completed
        # successfully.
        # Corresponds to the JSON property `waitFor`
        # @return [Array<String>]
        attr_accessor :wait_for
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @args = args[:args] if args.key?(:args)
          @dir = args[:dir] if args.key?(:dir)
          @entrypoint = args[:entrypoint] if args.key?(:entrypoint)
          @env = args[:env] if args.key?(:env)
          @id = args[:id] if args.key?(:id)
          @name = args[:name] if args.key?(:name)
          @secret_env = args[:secret_env] if args.key?(:secret_env)
          @timing = args[:timing] if args.key?(:timing)
          @volumes = args[:volumes] if args.key?(:volumes)
          @wait_for = args[:wait_for] if args.key?(:wait_for)
        end
      end
      
      # Configuration for an automated build in response to source repository
      # changes.
      class BuildTrigger
        include Google::Apis::Core::Hashable
      
        # A build resource in the Container Builder API.
        # At a high level, a Build describes where to find source code, how to build
        # it (for example, the builder image to run on the source), and what tag to
        # apply to the built image when it is pushed to Google Container Registry.
        # Fields can include the following variables which will be expanded when the
        # build is created:
        # - $PROJECT_ID: the project ID of the build.
        # - $BUILD_ID: the autogenerated ID of the build.
        # - $REPO_NAME: the source repository name specified by RepoSource.
        # - $BRANCH_NAME: the branch name specified by RepoSource.
        # - $TAG_NAME: the tag name specified by RepoSource.
        # - $REVISION_ID or $COMMIT_SHA: the commit SHA specified by RepoSource or
        # resolved from the specified branch or tag.
        # - $SHORT_SHA: first 7 characters of $REVISION_ID or $COMMIT_SHA.
        # Corresponds to the JSON property `build`
        # @return [Google::Apis::CloudbuildV1::Build]
        attr_accessor :build
      
        # Time when the trigger was created.
        # @OutputOnly
        # Corresponds to the JSON property `createTime`
        # @return [String]
        attr_accessor :create_time
      
        # Human-readable description of this trigger.
        # Corresponds to the JSON property `description`
        # @return [String]
        attr_accessor :description
      
        # If true, the trigger will never result in a build.
        # Corresponds to the JSON property `disabled`
        # @return [Boolean]
        attr_accessor :disabled
        alias_method :disabled?, :disabled
      
        # Path, from the source root, to a file whose contents is used for the
        # template.
        # Corresponds to the JSON property `filename`
        # @return [String]
        attr_accessor :filename
      
        # Unique identifier of the trigger.
        # @OutputOnly
        # Corresponds to the JSON property `id`
        # @return [String]
        attr_accessor :id
      
        # Substitutions data for Build resource.
        # Corresponds to the JSON property `substitutions`
        # @return [Hash<String,String>]
        attr_accessor :substitutions
      
        # RepoSource describes the location of the source in a Google Cloud Source
        # Repository.
        # Corresponds to the JSON property `triggerTemplate`
        # @return [Google::Apis::CloudbuildV1::RepoSource]
        attr_accessor :trigger_template
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @build = args[:build] if args.key?(:build)
          @create_time = args[:create_time] if args.key?(:create_time)
          @description = args[:description] if args.key?(:description)
          @disabled = args[:disabled] if args.key?(:disabled)
          @filename = args[:filename] if args.key?(:filename)
          @id = args[:id] if args.key?(:id)
          @substitutions = args[:substitutions] if args.key?(:substitutions)
          @trigger_template = args[:trigger_template] if args.key?(:trigger_template)
        end
      end
      
      # BuiltImage describes an image built by the pipeline.
      class BuiltImage
        include Google::Apis::Core::Hashable
      
        # Docker Registry 2.0 digest.
        # Corresponds to the JSON property `digest`
        # @return [String]
        attr_accessor :digest
      
        # Name used to push the container image to Google Container Registry, as
        # presented to `docker push`.
        # Corresponds to the JSON property `name`
        # @return [String]
        attr_accessor :name
      
        # Stores start and end times for a build execution phase.
        # Corresponds to the JSON property `pushTiming`
        # @return [Google::Apis::CloudbuildV1::TimeSpan]
        attr_accessor :push_timing
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @digest = args[:digest] if args.key?(:digest)
          @name = args[:name] if args.key?(:name)
          @push_timing = args[:push_timing] if args.key?(:push_timing)
        end
      end
      
      # Request to cancel an ongoing build.
      class CancelBuildRequest
        include Google::Apis::Core::Hashable
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
        end
      end
      
      # The request message for Operations.CancelOperation.
      class CancelOperationRequest
        include Google::Apis::Core::Hashable
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
        end
      end
      
      # A generic empty message that you can re-use to avoid defining duplicated
      # empty messages in your APIs. A typical example is to use it as the request
      # or the response type of an API method. For instance:
      # service Foo `
      # rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty);
      # `
      # The JSON representation for `Empty` is empty JSON object ````.
      class Empty
        include Google::Apis::Core::Hashable
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
        end
      end
      
      # Container message for hashes of byte content of files, used in
      # SourceProvenance messages to verify integrity of source input to the build.
      class FileHashes
        include Google::Apis::Core::Hashable
      
        # Collection of file hashes.
        # Corresponds to the JSON property `fileHash`
        # @return [Array<Google::Apis::CloudbuildV1::HashProp>]
        attr_accessor :file_hash
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @file_hash = args[:file_hash] if args.key?(:file_hash)
        end
      end
      
      # Container message for hash values.
      class HashProp
        include Google::Apis::Core::Hashable
      
        # The type of hash that was performed.
        # Corresponds to the JSON property `type`
        # @return [String]
        attr_accessor :type
      
        # The hash value.
        # Corresponds to the JSON property `value`
        # NOTE: Values are automatically base64 encoded/decoded in the client library.
        # @return [String]
        attr_accessor :value
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @type = args[:type] if args.key?(:type)
          @value = args[:value] if args.key?(:value)
        end
      end
      
      # Response containing existing BuildTriggers.
      class ListBuildTriggersResponse
        include Google::Apis::Core::Hashable
      
        # BuildTriggers for the project, sorted by create_time descending.
        # Corresponds to the JSON property `triggers`
        # @return [Array<Google::Apis::CloudbuildV1::BuildTrigger>]
        attr_accessor :triggers
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @triggers = args[:triggers] if args.key?(:triggers)
        end
      end
      
      # Response including listed builds.
      class ListBuildsResponse
        include Google::Apis::Core::Hashable
      
        # Builds will be sorted by create_time, descending.
        # Corresponds to the JSON property `builds`
        # @return [Array<Google::Apis::CloudbuildV1::Build>]
        attr_accessor :builds
      
        # Token to receive the next page of results.
        # Corresponds to the JSON property `nextPageToken`
        # @return [String]
        attr_accessor :next_page_token
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @builds = args[:builds] if args.key?(:builds)
          @next_page_token = args[:next_page_token] if args.key?(:next_page_token)
        end
      end
      
      # The response message for Operations.ListOperations.
      class ListOperationsResponse
        include Google::Apis::Core::Hashable
      
        # The standard List next-page token.
        # Corresponds to the JSON property `nextPageToken`
        # @return [String]
        attr_accessor :next_page_token
      
        # A list of operations that matches the specified filter in the request.
        # Corresponds to the JSON property `operations`
        # @return [Array<Google::Apis::CloudbuildV1::Operation>]
        attr_accessor :operations
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @next_page_token = args[:next_page_token] if args.key?(:next_page_token)
          @operations = args[:operations] if args.key?(:operations)
        end
      end
      
      # This resource represents a long-running operation that is the result of a
      # network API call.
      class Operation
        include Google::Apis::Core::Hashable
      
        # If the value is `false`, it means the operation is still in progress.
        # If `true`, the operation is completed, and either `error` or `response` is
        # available.
        # Corresponds to the JSON property `done`
        # @return [Boolean]
        attr_accessor :done
        alias_method :done?, :done
      
        # The `Status` type defines a logical error model that is suitable for different
        # programming environments, including REST APIs and RPC APIs. It is used by
        # [gRPC](https://github.com/grpc). The error model is designed to be:
        # - Simple to use and understand for most users
        # - Flexible enough to meet unexpected needs
        # # Overview
        # The `Status` message contains three pieces of data: error code, error message,
        # and error details. The error code should be an enum value of
        # google.rpc.Code, but it may accept additional error codes if needed.  The
        # error message should be a developer-facing English message that helps
        # developers *understand* and *resolve* the error. If a localized user-facing
        # error message is needed, put the localized message in the error details or
        # localize it in the client. The optional error details may contain arbitrary
        # information about the error. There is a predefined set of error detail types
        # in the package `google.rpc` that can be used for common error conditions.
        # # Language mapping
        # The `Status` message is the logical representation of the error model, but it
        # is not necessarily the actual wire format. When the `Status` message is
        # exposed in different client libraries and different wire protocols, it can be
        # mapped differently. For example, it will likely be mapped to some exceptions
        # in Java, but more likely mapped to some error codes in C.
        # # Other uses
        # The error model and the `Status` message can be used in a variety of
        # environments, either with or without APIs, to provide a
        # consistent developer experience across different environments.
        # Example uses of this error model include:
        # - Partial errors. If a service needs to return partial errors to the client,
        # it may embed the `Status` in the normal response to indicate the partial
        # errors.
        # - Workflow errors. A typical workflow has multiple steps. Each step may
        # have a `Status` message for error reporting.
        # - Batch operations. If a client uses batch request and batch response, the
        # `Status` message should be used directly inside batch response, one for
        # each error sub-response.
        # - Asynchronous operations. If an API call embeds asynchronous operation
        # results in its response, the status of those operations should be
        # represented directly using the `Status` message.
        # - Logging. If some API errors are stored in logs, the message `Status` could
        # be used directly after any stripping needed for security/privacy reasons.
        # Corresponds to the JSON property `error`
        # @return [Google::Apis::CloudbuildV1::Status]
        attr_accessor :error
      
        # Service-specific metadata associated with the operation.  It typically
        # contains progress information and common metadata such as create time.
        # Some services might not provide such metadata.  Any method that returns a
        # long-running operation should document the metadata type, if any.
        # Corresponds to the JSON property `metadata`
        # @return [Hash<String,Object>]
        attr_accessor :metadata
      
        # The server-assigned name, which is only unique within the same service that
        # originally returns it. If you use the default HTTP mapping, the
        # `name` should have the format of `operations/some/unique/name`.
        # Corresponds to the JSON property `name`
        # @return [String]
        attr_accessor :name
      
        # The normal response of the operation in case of success.  If the original
        # method returns no data on success, such as `Delete`, the response is
        # `google.protobuf.Empty`.  If the original method is standard
        # `Get`/`Create`/`Update`, the response should be the resource.  For other
        # methods, the response should have the type `XxxResponse`, where `Xxx`
        # is the original method name.  For example, if the original method name
        # is `TakeSnapshot()`, the inferred response type is
        # `TakeSnapshotResponse`.
        # Corresponds to the JSON property `response`
        # @return [Hash<String,Object>]
        attr_accessor :response
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @done = args[:done] if args.key?(:done)
          @error = args[:error] if args.key?(:error)
          @metadata = args[:metadata] if args.key?(:metadata)
          @name = args[:name] if args.key?(:name)
          @response = args[:response] if args.key?(:response)
        end
      end
      
      # RepoSource describes the location of the source in a Google Cloud Source
      # Repository.
      class RepoSource
        include Google::Apis::Core::Hashable
      
        # Name of the branch to build.
        # Corresponds to the JSON property `branchName`
        # @return [String]
        attr_accessor :branch_name
      
        # Explicit commit SHA to build.
        # Corresponds to the JSON property `commitSha`
        # @return [String]
        attr_accessor :commit_sha
      
        # Directory, relative to the source root, in which to run the build.
        # This must be a relative path. If a step's dir is specified and is an
        # absolute path, this value is ignored for that step's execution.
        # Corresponds to the JSON property `dir`
        # @return [String]
        attr_accessor :dir
      
        # ID of the project that owns the repo. If omitted, the project ID requesting
        # the build is assumed.
        # Corresponds to the JSON property `projectId`
        # @return [String]
        attr_accessor :project_id
      
        # Name of the repo. If omitted, the name "default" is assumed.
        # Corresponds to the JSON property `repoName`
        # @return [String]
        attr_accessor :repo_name
      
        # Name of the tag to build.
        # Corresponds to the JSON property `tagName`
        # @return [String]
        attr_accessor :tag_name
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @branch_name = args[:branch_name] if args.key?(:branch_name)
          @commit_sha = args[:commit_sha] if args.key?(:commit_sha)
          @dir = args[:dir] if args.key?(:dir)
          @project_id = args[:project_id] if args.key?(:project_id)
          @repo_name = args[:repo_name] if args.key?(:repo_name)
          @tag_name = args[:tag_name] if args.key?(:tag_name)
        end
      end
      
      # Results describes the artifacts created by the build pipeline.
      class Results
        include Google::Apis::Core::Hashable
      
        # List of build step digests, in order corresponding to build step indices.
        # Corresponds to the JSON property `buildStepImages`
        # @return [Array<String>]
        attr_accessor :build_step_images
      
        # Images that were built as a part of the build.
        # Corresponds to the JSON property `images`
        # @return [Array<Google::Apis::CloudbuildV1::BuiltImage>]
        attr_accessor :images
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @build_step_images = args[:build_step_images] if args.key?(:build_step_images)
          @images = args[:images] if args.key?(:images)
        end
      end
      
      # RetryBuildRequest specifies a build to retry.
      class RetryBuildRequest
        include Google::Apis::Core::Hashable
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
        end
      end
      
      # Secret pairs a set of secret environment variables containing encrypted
      # values with the Cloud KMS key to use to decrypt the value.
      class Secret
        include Google::Apis::Core::Hashable
      
        # Cloud KMS key name to use to decrypt these envs.
        # Corresponds to the JSON property `kmsKeyName`
        # @return [String]
        attr_accessor :kms_key_name
      
        # Map of environment variable name to its encrypted value.
        # Secret environment variables must be unique across all of a build's
        # secrets, and must be used by at least one build step. Values can be at most
        # 1 KB in size. There can be at most ten secret values across all of a
        # build's secrets.
        # Corresponds to the JSON property `secretEnv`
        # @return [Hash<String,String>]
        attr_accessor :secret_env
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @kms_key_name = args[:kms_key_name] if args.key?(:kms_key_name)
          @secret_env = args[:secret_env] if args.key?(:secret_env)
        end
      end
      
      # Source describes the location of the source in a supported storage
      # service.
      class Source
        include Google::Apis::Core::Hashable
      
        # RepoSource describes the location of the source in a Google Cloud Source
        # Repository.
        # Corresponds to the JSON property `repoSource`
        # @return [Google::Apis::CloudbuildV1::RepoSource]
        attr_accessor :repo_source
      
        # StorageSource describes the location of the source in an archive file in
        # Google Cloud Storage.
        # Corresponds to the JSON property `storageSource`
        # @return [Google::Apis::CloudbuildV1::StorageSource]
        attr_accessor :storage_source
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @repo_source = args[:repo_source] if args.key?(:repo_source)
          @storage_source = args[:storage_source] if args.key?(:storage_source)
        end
      end
      
      # Provenance of the source. Ways to find the original source, or verify that
      # some source was used for this build.
      class SourceProvenance
        include Google::Apis::Core::Hashable
      
        # Hash(es) of the build source, which can be used to verify that the original
        # source integrity was maintained in the build. Note that FileHashes will
        # only be populated if BuildOptions has requested a SourceProvenanceHash.
        # The keys to this map are file paths used as build source and the values
        # contain the hash values for those files.
        # If the build source came in a single package such as a gzipped tarfile
        # (.tar.gz), the FileHash will be for the single path to that file.
        # @OutputOnly
        # Corresponds to the JSON property `fileHashes`
        # @return [Hash<String,Google::Apis::CloudbuildV1::FileHashes>]
        attr_accessor :file_hashes
      
        # RepoSource describes the location of the source in a Google Cloud Source
        # Repository.
        # Corresponds to the JSON property `resolvedRepoSource`
        # @return [Google::Apis::CloudbuildV1::RepoSource]
        attr_accessor :resolved_repo_source
      
        # StorageSource describes the location of the source in an archive file in
        # Google Cloud Storage.
        # Corresponds to the JSON property `resolvedStorageSource`
        # @return [Google::Apis::CloudbuildV1::StorageSource]
        attr_accessor :resolved_storage_source
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @file_hashes = args[:file_hashes] if args.key?(:file_hashes)
          @resolved_repo_source = args[:resolved_repo_source] if args.key?(:resolved_repo_source)
          @resolved_storage_source = args[:resolved_storage_source] if args.key?(:resolved_storage_source)
        end
      end
      
      # The `Status` type defines a logical error model that is suitable for different
      # programming environments, including REST APIs and RPC APIs. It is used by
      # [gRPC](https://github.com/grpc). The error model is designed to be:
      # - Simple to use and understand for most users
      # - Flexible enough to meet unexpected needs
      # # Overview
      # The `Status` message contains three pieces of data: error code, error message,
      # and error details. The error code should be an enum value of
      # google.rpc.Code, but it may accept additional error codes if needed.  The
      # error message should be a developer-facing English message that helps
      # developers *understand* and *resolve* the error. If a localized user-facing
      # error message is needed, put the localized message in the error details or
      # localize it in the client. The optional error details may contain arbitrary
      # information about the error. There is a predefined set of error detail types
      # in the package `google.rpc` that can be used for common error conditions.
      # # Language mapping
      # The `Status` message is the logical representation of the error model, but it
      # is not necessarily the actual wire format. When the `Status` message is
      # exposed in different client libraries and different wire protocols, it can be
      # mapped differently. For example, it will likely be mapped to some exceptions
      # in Java, but more likely mapped to some error codes in C.
      # # Other uses
      # The error model and the `Status` message can be used in a variety of
      # environments, either with or without APIs, to provide a
      # consistent developer experience across different environments.
      # Example uses of this error model include:
      # - Partial errors. If a service needs to return partial errors to the client,
      # it may embed the `Status` in the normal response to indicate the partial
      # errors.
      # - Workflow errors. A typical workflow has multiple steps. Each step may
      # have a `Status` message for error reporting.
      # - Batch operations. If a client uses batch request and batch response, the
      # `Status` message should be used directly inside batch response, one for
      # each error sub-response.
      # - Asynchronous operations. If an API call embeds asynchronous operation
      # results in its response, the status of those operations should be
      # represented directly using the `Status` message.
      # - Logging. If some API errors are stored in logs, the message `Status` could
      # be used directly after any stripping needed for security/privacy reasons.
      class Status
        include Google::Apis::Core::Hashable
      
        # The status code, which should be an enum value of google.rpc.Code.
        # Corresponds to the JSON property `code`
        # @return [Fixnum]
        attr_accessor :code
      
        # A list of messages that carry the error details.  There is a common set of
        # message types for APIs to use.
        # Corresponds to the JSON property `details`
        # @return [Array<Hash<String,Object>>]
        attr_accessor :details
      
        # A developer-facing error message, which should be in English. Any
        # user-facing error message should be localized and sent in the
        # google.rpc.Status.details field, or localized by the client.
        # Corresponds to the JSON property `message`
        # @return [String]
        attr_accessor :message
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @code = args[:code] if args.key?(:code)
          @details = args[:details] if args.key?(:details)
          @message = args[:message] if args.key?(:message)
        end
      end
      
      # StorageSource describes the location of the source in an archive file in
      # Google Cloud Storage.
      class StorageSource
        include Google::Apis::Core::Hashable
      
        # Google Cloud Storage bucket containing source (see
        # [Bucket Name
        # Requirements](https://cloud.google.com/storage/docs/bucket-naming#requirements)
        # ).
        # Corresponds to the JSON property `bucket`
        # @return [String]
        attr_accessor :bucket
      
        # Google Cloud Storage generation for the object. If the generation is
        # omitted, the latest generation will be used.
        # Corresponds to the JSON property `generation`
        # @return [Fixnum]
        attr_accessor :generation
      
        # Google Cloud Storage object containing source.
        # This object must be a gzipped archive file (.tar.gz) containing source to
        # build.
        # Corresponds to the JSON property `object`
        # @return [String]
        attr_accessor :object
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @bucket = args[:bucket] if args.key?(:bucket)
          @generation = args[:generation] if args.key?(:generation)
          @object = args[:object] if args.key?(:object)
        end
      end
      
      # Stores start and end times for a build execution phase.
      class TimeSpan
        include Google::Apis::Core::Hashable
      
        # End of time span.
        # Corresponds to the JSON property `endTime`
        # @return [String]
        attr_accessor :end_time
      
        # Start of time span.
        # Corresponds to the JSON property `startTime`
        # @return [String]
        attr_accessor :start_time
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @end_time = args[:end_time] if args.key?(:end_time)
          @start_time = args[:start_time] if args.key?(:start_time)
        end
      end
      
      # Volume describes a Docker container volume which is mounted into build steps
      # in order to persist files across build step execution.
      class Volume
        include Google::Apis::Core::Hashable
      
        # Name of the volume to mount.
        # Volume names must be unique per build step and must be valid names for
        # Docker volumes. Each named volume must be used by at least two build steps.
        # Corresponds to the JSON property `name`
        # @return [String]
        attr_accessor :name
      
        # Path at which to mount the volume.
        # Paths must be absolute and cannot conflict with other volume paths on the
        # same build step or with certain reserved volume paths.
        # Corresponds to the JSON property `path`
        # @return [String]
        attr_accessor :path
      
        def initialize(**args)
           update!(**args)
        end
      
        # Update properties of this object
        def update!(**args)
          @name = args[:name] if args.key?(:name)
          @path = args[:path] if args.key?(:path)
        end
      end
    end
  end
end
