/*  RipOff - Plugin based CD Ripper
 *  Copyright (C) 2006 Bobby Ryan Newberry
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "RipOffPreferences.h"
#include "lib/RipOffConfigFile.h"
#include "lib/ripoff.xpm"
#include <ltdl.h>
enum
{
	COL_PLUGIN_NAME=0,
	COL_ABOUT,
	COL_PREFERENCES,
	COL_PLUGIN_POINTER,
	NUM_COLS_PREF_VIEW
};

/* convenience struct for communicating data using gtk+ */
struct pref_window_struct_
{
	RipOffPreferences pref_object;
	GtkWidget *preferences_window;
	GtkWidget *cd_path_entry;
	GtkWidget *format_entry;
	GtkWidget *output_directory_label;
	GtkWidget *about_button;
	GtkWidget *prefs_button;
};
typedef struct pref_window_struct_ * pref_window_struct;

/* internal helper functions */

/* creates a RipOffPluginList based on the directories that RipOff searches for Plugins in */
RipOffPluginList create_plugin_list();

/* loops through the contents of the directory and creates RipOffPlugins from the 
   RawRipOffPlugins that can be sucessfully linked */
void search_directory_for_plugins(gchar *directory_path, RipOffPluginList list);

/* used by the output path widget for opening a GTK directory selection dialog */
void directory_select(GtkWidget *button, gpointer data);

/* creates the GtkView that users select the RipOffPlugin they wish to use */
GtkWidget *create_plugin_view(RipOffPluginList plugins);

/* creates the GtkTreeModel that will attach to the GtkView create_plugin_view uses 
   The model has 4 columns
   COL_PLUGIN_NAME
   COL_ABOUT which tells whether the plugin has an about dialog true so does false if not
   COL_PREFS which tells whether the plugin has a preferences dialog true if so false if not
   COL_PLUGIN_POINTER which is a pointer to the RipOffPlugin object associated with the plugin

   Only COL_PLUGIN_NAME is displayed to the user. The other three columns are used by
   the RipOffPreferences object and are not shown to the user. */ 
GtkTreeModel *create_plugin_model(RipOffPluginList plugins);

void select_correct_plugin(RipOffPlugin selected, GtkTreeView *view);

gboolean row_selection	 (	GtkTreeSelection *selection,
                     	  	GtkTreeModel     *model,
                       		GtkTreePath      *path,
                    		gboolean          path_currently_selected,
                     		gpointer          data);

void about_the_plugin(GtkWidget *widget, gpointer data);

void prefs_of_plugin(GtkWidget *widget, gpointer data);

void close_preferences_window(GtkWidget *widget, gpointer data);

void close_preferences_event(GtkWidget *window, GdkEvent *event, gpointer data);

RipOffPreferences ripoff_preferences_new()
{
	RipOffPreferences prefs = g_new(struct RipOffPreferences_, 1);
	gchar *home_dir = getenv("HOME");
	gchar *plugin_label;
	prefs->selected = NULL;

	prefs->cd_path = ripoff_config_file_retrieve_string("core", "cd_path", "/dev/cdrom");
	prefs->file_pattern = ripoff_config_file_retrieve_string("core", "file_pattern", "%N - %t");

	if(home_dir == NULL)
		prefs->output_folder = ripoff_config_file_retrieve_string("core", "output_folder", "/");
	else
		prefs->output_folder = ripoff_config_file_retrieve_string("core", "output_folder", home_dir);

	prefs->plugins = create_plugin_list();
	plugin_label = ripoff_config_file_retrieve_string("core", "label_of_selected_plugin", NULL);

	if(plugin_label == NULL)
		prefs->selected = ripoff_plugin_list_start(prefs->plugins);
	else
	{
		/* Cycle through the plugin list and try to find one that matches the stored label */
		if(!ripoff_plugin_list_is_empty(prefs->plugins))
		{
			prefs->selected = ripoff_plugin_list_start(prefs->plugins);

			do {
				if(!strcmp(plugin_label, ripoff_plugin_get_label(prefs->selected)))
					break;
				else
					prefs->selected = ripoff_plugin_list_next_plugin(prefs->plugins);

			} while(prefs->selected != NULL);
		} 
	}
	

	return prefs;
}

void ripoff_preferences_draw_window(RipOffPreferences pref, GtkWidget *main_window)
{
	/* Preferences window widgets */
	GtkWidget *pref_window;
	GtkWidget *page1;
	GtkWidget *page2;
	GtkWidget *notebook;
	GtkWidget *close_button;
	GtkWidget *table;
        GdkPixbuf *icon;

	/* page1 widgets */
	GtkWidget *device_label;
	GtkWidget *format_label;
	GtkWidget *output_directory_display_label;
	GtkWidget *output_directory_label;
	GtkWidget *device_text_entry;
	GtkWidget *format_text_entry;
	GtkWidget *output_button;
	GtkTooltips *entry_tips;

	/* page2 widgets */
	GtkWidget *view;
	GtkWidget *prefs_button;
	GtkWidget *about_button;
	GtkWidget *scroll;

	/* structure used to get around GTK limitation of only one data pointer per signal */
	pref_window_struct shared_data = g_new(struct pref_window_struct_, 1);

	/* Notebook setup code */
	notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);

	/* Window setup code */
	pref_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_size_request (GTK_WIDGET (pref_window), 450, 275);
	gtk_window_set_title (GTK_WINDOW (pref_window), NAME" Preferences");
	gtk_container_set_border_width(GTK_CONTAINER (pref_window), 6);

	/* Table Setup Code */
	table = gtk_table_new(2, 2, FALSE);

	/* Beginning of first notebook page code*/
	page1 = gtk_table_new(4, 9, FALSE);

	/* beginning of page1 widget creation code */

	/*Creates label and text entry field for path to CD-ROM device to be used by RipOff*/
	device_label = gtk_label_new(NULL);

	gtk_label_set_markup(	GTK_LABEL(device_label), 
	g_markup_printf_escaped (	"<span weight=\"bold\" style=\"italic\">%s</span>", 
					"Path to CD-ROM device node"));

	gtk_label_set_justify(GTK_LABEL(device_label), GTK_JUSTIFY_LEFT);

	
	device_text_entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(device_text_entry), pref->cd_path);

	/* Creates label and text entry field for the track output format string */
	format_label = gtk_label_new(NULL);

	gtk_label_set_markup(	GTK_LABEL(format_label), 
	g_markup_printf_escaped (	"<span weight=\"bold\" style=\"italic\">%s</span>", 
					"Track Output Format"));

	gtk_label_set_justify(GTK_LABEL(format_label), GTK_JUSTIFY_LEFT);


	format_text_entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(format_text_entry), pref->file_pattern);


	/* Creates label and box for output path */
	output_directory_label = gtk_label_new(NULL);

	gtk_label_set_markup(	GTK_LABEL(output_directory_label), 
	g_markup_printf_escaped ("<span weight=\"bold\" style=\"italic\">%s</span>", 
				 "Ripoff Output Folder"));

	output_directory_display_label = gtk_label_new(pref->output_folder);
	gtk_label_set_text(GTK_LABEL(output_directory_display_label), pref->output_folder);
	gtk_label_set_line_wrap(GTK_LABEL(output_directory_display_label), TRUE);
	gtk_label_set_justify(GTK_LABEL(output_directory_display_label), GTK_JUSTIFY_LEFT);
	/* sets the selected plugin in the view according to whether a saved selected 
		plugin exists or not */

	/* creates a data structure that contains all the data the preference signals need
	   the about_button and prefs_button pointers will be set when the buttons are created
	   for page 2 of the notebook */

	shared_data->pref_object = pref;
	shared_data->preferences_window = pref_window;
	shared_data->cd_path_entry = device_text_entry;
	shared_data->format_entry = format_text_entry;
	shared_data->output_directory_label = output_directory_display_label;

	output_button = gtk_button_new_from_stock(GTK_STOCK_SAVE);

	/* end of page1 widget creation code */

	/* Signal connections for page 1*/
	g_signal_connect (	G_OBJECT (output_button), 
				"clicked",
                      		G_CALLBACK (directory_select), 
				shared_data);

	/* table attachments for page1 */
	gtk_table_attach (	GTK_TABLE(page1), 
				device_label, 0, 1, 0, 1, 
				GTK_SHRINK, 
				GTK_SHRINK, 
				6, 2);

	gtk_table_attach (	GTK_TABLE(page1), 
				device_text_entry, 0, 3, 1, 2, 
				GTK_FILL, 
				GTK_SHRINK, 
				6, 2);

	gtk_table_attach (	GTK_TABLE(page1), 
				format_label, 0, 1, 2, 3, 
				GTK_SHRINK, 
				GTK_SHRINK, 
				6, 2);

	gtk_table_attach (	GTK_TABLE(page1), 
				format_text_entry, 0, 3, 3, 4, 
				GTK_FILL, 
				GTK_SHRINK, 
				6, 2);

	gtk_table_attach (	GTK_TABLE(page1), 
				output_directory_label, 0, 1, 5, 6, 
				GTK_SHRINK, 
				GTK_SHRINK, 
				0, 5);

	gtk_table_attach (	GTK_TABLE(page1), 
				output_directory_display_label, 0, 2, 6, 7, 
				GTK_SHRINK, 
				GTK_SHRINK, 
				0, 5);

	gtk_table_attach (	GTK_TABLE(page1), 
				output_button, 2, 3, 6, 7, 
				GTK_SHRINK, 
				GTK_SHRINK, 
				0, 5);
	
	entry_tips = gtk_tooltips_new();
	gtk_tooltips_set_tip(entry_tips, format_text_entry, "This field determines the way your ripped\nsongs are outputted.\n\nFor example:\n%l/%N - %t\n\nwould output your songs in the following format:\n\n(CD Title)/(Track Number) - (Track Name)\n\n%N=Track Number with leading zero\n%n=Track Number without leading zero\n%t=Track Title\n%a=Artist Name\n%g=Genre\n%y=year\n%l=Album Title", NULL);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page1, gtk_label_new("Input/Output"));
	/* End of page1 code */

	/* Beginning of second notebook page code 	*/
	/* Needs to have three things. Name display window, about and preferences buttons
	   Each time a new name is clicked the about and preferences buttons must be 
	   enabled/disabled depending on whether the plugin supports those options.
	   Also pref->selected must be changed to reflect the new selected plugin. 
	   In the case of a new user, the first plugin in the list should be selected
	   In the case of an old user, the selected plugin should be whatever they
	   selected last time they used the program. The list will already be created
	   before this function is called.*/

	page2 = gtk_table_new(2, 2, FALSE);

	/* widget creation for page1*/
	view = create_plugin_view(pref->plugins);

	scroll = gtk_scrolled_window_new(NULL, NULL);
	
	gtk_scrolled_window_set_policy(		GTK_SCROLLED_WINDOW(scroll), 
				       		GTK_POLICY_AUTOMATIC, 
				       		GTK_POLICY_AUTOMATIC);

	gtk_container_add(GTK_CONTAINER(scroll), view);

        #if defined(GTK_MINOR_VERSION) && GTK_MINOR_VERSION > 6
        about_button = gtk_button_new_from_stock(GTK_STOCK_ABOUT);
        #else
        about_button = gtk_button_new_with_label("About");
        #endif

	prefs_button = gtk_button_new_from_stock(GTK_STOCK_PREFERENCES);

	shared_data->about_button = about_button;
	shared_data->prefs_button = prefs_button;

	/* sets the selected plugin in the view according to whether a saved selected 
	plugin exists or not */
	if(pref->selected != NULL)
	{	
		select_correct_plugin(pref->selected, GTK_TREE_VIEW(view));
		
		if(!ripoff_plugin_has_prefs(pref->selected))
			gtk_widget_set_sensitive(prefs_button, FALSE);

		if(!ripoff_plugin_has_about(pref->selected))
			gtk_widget_set_sensitive(about_button, FALSE);

	}
	else
	{
		gtk_widget_set_sensitive(about_button, FALSE);
		gtk_widget_set_sensitive(prefs_button, FALSE);
	}

	/* sets the function for handling row selections to row_selection */
    	gtk_tree_selection_set_select_function(  
						gtk_tree_view_get_selection(GTK_TREE_VIEW(view)), 
							row_selection, 
							shared_data, 
							NULL);

	g_signal_connect(	G_OBJECT(about_button),
				"clicked",
				G_CALLBACK(about_the_plugin),
				shared_data);

	g_signal_connect(	G_OBJECT(prefs_button),
				"clicked",
				G_CALLBACK(prefs_of_plugin),
				shared_data);

	/* table attachments for page2*/

	gtk_table_attach(	GTK_TABLE(page2),
				scroll, 0, 2, 0, 1,
				GTK_FILL | GTK_EXPAND,
				GTK_FILL | GTK_EXPAND,
				0, 5);

	gtk_table_attach(	GTK_TABLE(page2),
				prefs_button, 0, 1, 1, 2,
				GTK_FILL | GTK_EXPAND,
				GTK_SHRINK,
				7, 5);

	gtk_table_attach(	GTK_TABLE(page2),
				about_button, 1, 2, 1, 2,
				GTK_FILL | GTK_EXPAND,
				GTK_SHRINK,
				7, 5);




	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page2, gtk_label_new("Plugins"));
	/* End of second notebook page code */

	/* close button code, needs to be changed to call preferences closed function */
	close_button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
	gtk_widget_set_size_request(close_button, 15, 25);

	g_signal_connect (		G_OBJECT (close_button), 
					"clicked",
                             		G_CALLBACK (close_preferences_window), 
                              		shared_data);

	g_signal_connect(	pref_window,
				"delete_event",
				G_CALLBACK(close_preferences_event),
				shared_data);

	gtk_table_attach (	GTK_TABLE(table), 
				notebook, 0, 2, 0, 1, 
				GTK_FILL | GTK_EXPAND, 
				GTK_FILL | GTK_EXPAND, 
				0, 5);

	gtk_table_attach (	GTK_TABLE(table), 
				close_button, 1, 2, 1, 2, 
				GTK_FILL, 
				GTK_SHRINK, 
				0, 5);

	

	gtk_container_add(GTK_CONTAINER(pref_window), table);

	gtk_window_set_transient_for(GTK_WINDOW(pref_window), GTK_WINDOW(main_window));
	gtk_window_set_modal(GTK_WINDOW(pref_window), TRUE);
	gtk_window_set_type_hint(GTK_WINDOW(pref_window), GDK_WINDOW_TYPE_HINT_DIALOG);
	gtk_window_set_decorated(GTK_WINDOW(pref_window), TRUE);
	gtk_window_set_skip_taskbar_hint(GTK_WINDOW(pref_window), TRUE);

        icon = gdk_pixbuf_new_from_xpm_data((const gchar **) ripoff_xpm);
        gtk_window_set_icon(GTK_WINDOW(pref_window), icon);
        g_object_unref(icon);

	gtk_widget_show_all(pref_window);
}

gboolean ripoff_preferences_save(RipOffPreferences pref)
{
	RipOffPlugin plugin;
	ripoff_config_file_save_string("core", "cd_path", pref->cd_path);
	ripoff_config_file_save_string("core", "file_pattern", pref->file_pattern);
	ripoff_config_file_save_string("core", "output_folder", pref->output_folder);

	/* Cycle through the plugin list and inform each of them that they need to save
	   their settings */
	if(!ripoff_plugin_list_is_empty(pref->plugins))
	{
		/* if the list were empty, there would be no need to attempt this save */

		if(pref->selected != NULL)
			ripoff_config_file_save_string(	"core", 
							"label_of_selected_plugin" , 
							ripoff_plugin_get_label(pref->selected));

		plugin = ripoff_plugin_list_start(pref->plugins);

		do {
			ripoff_plugin_close(plugin);
			plugin = ripoff_plugin_list_next_plugin(pref->plugins);

		} while(plugin != NULL);
	}
	return ripoff_config_file_write_config_file_to_disc();
}

const gchar *ripoff_preferences_get_cd_device_path(RipOffPreferences pref)
{
	return pref->cd_path;
}

const gchar *ripoff_preferences_get_file_pattern(RipOffPreferences pref)
{
	int size = strlen(pref->file_pattern)+1;
	gchar *string_copy = g_new(gchar, size);
	strncpy(string_copy, pref->file_pattern, size);
	return pref->file_pattern;
}

RipOffPlugin ripoff_preferences_get_plugin(RipOffPreferences pref)
{
	return pref->selected;
}

const gchar *ripoff_preferences_get_output_folder(RipOffPreferences pref)
{
	return pref->output_folder;
}

RipOffPluginList create_plugin_list()
{
	RipOffPluginList list;
	gchar *home_dir;
	gchar *local_plugin_dir;
	
	list = ripoff_plugin_list_new();

	#ifdef DEBUG
	g_print("Global Plugin Directory: %s\n", PLUGIN_DIR);
	#endif 

	search_directory_for_plugins(PLUGIN_DIR, list);

	home_dir = getenv("HOME");

	if(home_dir == NULL)
		return list;

	local_plugin_dir = g_strconcat(home_dir, "/", RIPOFF_RCPATH, "/plugins", NULL);
	search_directory_for_plugins(local_plugin_dir, list);
	
	g_free(local_plugin_dir);
	return list;
}

void search_directory_for_plugins(gchar *directory_path, RipOffPluginList list)
{
	gpointer lib_handle;
	gpointer (*get_plugin)(xmlDocPtr);
	RipOffPlugin plugin;
	gchar *full_path;
	const gchar *name;
	gchar *error_msg;

	GDir *directory = g_dir_open(directory_path, 0, NULL); 

	if(directory == NULL)
		return;

	name = g_dir_read_name(directory);

	while(name != NULL)
	{
		full_path = g_strconcat(directory_path, "/", name, NULL);

		#ifdef DEBUG
                g_print("FULL PATH: %s\n", full_path);
		#endif

		lib_handle = dlopen(full_path, RTLD_LAZY);
		g_free(full_path);

		if(lib_handle != NULL)
		{
			get_plugin = dlsym(lib_handle, "ripoff_plugin_raw_new");
			error_msg = dlerror();

			if(error_msg != NULL)
			{
				#ifdef DEBUG
				g_fprintf(stderr, "RipOffPreferences: Error locating 'function' - %s\n", error_msg);
				#endif
			}

			else
			{
				#ifdef DEBUG
				g_print("RipOff successfully loaded a plugin\n");
				#endif 

				plugin = ripoff_plugin_new(get_plugin(ripoff_config_file_get_config_file_pointer()), lib_handle);
				ripoff_plugin_list_add_plugin(list, plugin);
			}
		}

		name = g_dir_read_name(directory);
	}

	g_dir_close(directory);
}

void directory_select(GtkWidget *button, gpointer data)
{
	GtkWidget *dialog;
	pref_window_struct shared_data = (pref_window_struct)data;

	/*directory choose dialog setup*/
	dialog = gtk_file_chooser_dialog_new ("Choose output directory",
				      GTK_WINDOW(shared_data->preferences_window),
				      GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
				      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
				      NULL);

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
  	{
		gchar *filename;

    		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
   		gtk_label_set_text(	GTK_LABEL(shared_data->output_directory_label), 
					filename);
		/*shared_data->output_directory_label = filename;*/
  	}

	gtk_widget_destroy(dialog);
}

GtkWidget *create_plugin_view(RipOffPluginList plugins)
{
	GtkWidget *view;
	GtkTreeModel *model;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;

	view = gtk_tree_view_new();
	model = create_plugin_model(plugins);

	/* Visible Column */
	renderer = gtk_cell_renderer_text_new();
	
	col = gtk_tree_view_column_new_with_attributes(		"Plugins", 
						  		renderer, 
						  		"text", COL_PLUGIN_NAME,
						 	 	NULL);


  	gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  	g_object_set_data(	G_OBJECT (renderer), 
				"column", (gint*)  COL_PLUGIN_NAME);

  	gtk_tree_view_set_model(GTK_TREE_VIEW(view), model); /*Connect tree and model here*/
  	g_object_unref(model); /* destroy model automatically with view */
  	gtk_tree_selection_set_mode(	gtk_tree_view_get_selection(GTK_TREE_VIEW(view)),
                             		GTK_SELECTION_BROWSE);

	return view;

}

GtkTreeModel *create_plugin_model(RipOffPluginList plugins)
{
	GtkTreeStore *store;
	GtkTreeIter iter;
	RipOffPlugin plugin;

	store = gtk_tree_store_new(	NUM_COLS_PREF_VIEW,
					G_TYPE_STRING,
					G_TYPE_BOOLEAN,
					G_TYPE_BOOLEAN,
					G_TYPE_POINTER);

	/* cycles through the RipOffPluginList and adds each plugin to the model 
	   The model stores the plugin name, whether the plugin has an about and preferences 
	   dialog, and a plugin to the plugin itself */
	if(!ripoff_plugin_list_is_empty(plugins))
	{
		plugin = ripoff_plugin_list_start(plugins);

		do{
			gtk_tree_store_append(store, &iter, NULL);

 			gtk_tree_store_set(	store, &iter,
		     				COL_PLUGIN_NAME, ripoff_plugin_get_name(plugin),
                    				COL_ABOUT, ripoff_plugin_has_about(plugin),
                     				COL_PREFERENCES, ripoff_plugin_has_prefs(plugin),
                     				COL_PLUGIN_POINTER, plugin,
		     				-1);

			plugin = ripoff_plugin_list_next_plugin(plugins);
		}while(plugin != NULL);
	}

	return GTK_TREE_MODEL(store);
}

void select_correct_plugin(RipOffPlugin selected, GtkTreeView *view)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	RipOffPlugin plugin_of_the_row;
	GtkTreePath *path;

	model = gtk_tree_view_get_model(view);

	/* checks that a plugin has indeed been selected earlier, and that the plugin model
	   is not empty */
	if(	selected != NULL && 
		gtk_tree_model_get_iter_first(model, &iter))
	{
		

		/* cycles through the model trying to find a match for the selected plugin */
		do{
			gtk_tree_model_get (	GTK_TREE_MODEL (model), 
						&iter, 
						COL_PLUGIN_POINTER, 
						&plugin_of_the_row, 
						-1);

			if(selected == plugin_of_the_row)
			{
				/* creates path string based on the Iter */
				path = gtk_tree_model_get_path(model, &iter); 

				/* activates row corresponding to the plugin that has been
				  determined to be the selected one */
				/*gtk_tree_view_row_activated(	view,
								path,
				gtk_tree_view_get_column(view, COL_PLUGIN_NAME));*/

				gtk_tree_selection_select_path(
					gtk_tree_view_get_selection(view), path);

				gtk_tree_path_free(path);

				break;
			}

		}while(gtk_tree_model_iter_next(GTK_TREE_MODEL (model), &iter));
	}
}

/*void row_selection(GtkTreeView *view, GtkTreePath *path, GtkTreeViewColumn *col, gpointer data)
{
	GtkTreeIter iter;
	RipOffPreferences prefs;

	prefs = (RipOffPreferences)data;

	g_print("row_selection\n") ;
	gtk_tree_model_get_iter(gtk_tree_view_get_model(view), &iter, path);
	
	gtk_tree_model_get (	gtk_tree_view_get_model(view), 
				&iter, 
				COL_PLUGIN_POINTER, 
				&(prefs->selected), 
				-1);
}*/

gboolean row_selection	 (	GtkTreeSelection *selection,
                     	  		GtkTreeModel     *model,
                       			GtkTreePath      *path,
                    			gboolean          path_currently_selected,
                     			gpointer          data)
{
	RipOffPreferences prefs;
	GtkTreeIter iter;
	GtkWidget *about_button;
	GtkWidget *prefs_button;
	gboolean has_about;
	gboolean has_prefs;
	pref_window_struct shared_data = (pref_window_struct)data;

	prefs = shared_data->pref_object;
	about_button = shared_data->about_button;
	prefs_button = shared_data->prefs_button;

	if(!path_currently_selected && prefs->selected != NULL)
	{
		gtk_tree_model_get_iter(model, &iter, path);

		gtk_tree_model_get (	model, 
					&iter, 
					COL_PLUGIN_POINTER, 
					&(prefs->selected),
					COL_ABOUT,
					&has_about,
					COL_PREFERENCES,
					&has_prefs,
					-1);

		if(!has_prefs)
			gtk_widget_set_sensitive(prefs_button, FALSE);
		else
			gtk_widget_set_sensitive(prefs_button, TRUE);

		if(!has_about)
			gtk_widget_set_sensitive(about_button, FALSE);
		else
			gtk_widget_set_sensitive(about_button, TRUE);
	
		#ifdef DEBUG
		g_print("RipOffPreferences: %s selected\n", ripoff_plugin_get_name(prefs->selected));
		#endif
	}

	return TRUE;
}

void about_the_plugin(GtkWidget *widget, gpointer data)
{
	GtkWidget *plugin_about_window;
	pref_window_struct shared_data = (pref_window_struct)data;
	GtkWidget *window = shared_data->preferences_window;
	RipOffPlugin plugin = shared_data->pref_object->selected;


	plugin_about_window = ripoff_plugin_about(plugin);

	if(plugin_about_window != NULL)
	{
		gtk_window_set_modal(GTK_WINDOW(plugin_about_window), TRUE);
		gtk_window_set_transient_for(GTK_WINDOW(plugin_about_window), GTK_WINDOW(window));
		gtk_window_set_type_hint(GTK_WINDOW(plugin_about_window), GDK_WINDOW_TYPE_HINT_DIALOG);
		gtk_window_set_decorated(GTK_WINDOW(plugin_about_window), TRUE);
		gtk_window_set_skip_taskbar_hint(GTK_WINDOW(plugin_about_window), TRUE);

		gtk_widget_show_all(plugin_about_window);
	}
}

void prefs_of_plugin(GtkWidget *widget, gpointer data)
{
	GtkWidget *plugin_pref_window;
	pref_window_struct shared_data = (pref_window_struct)data;
	GtkWidget *window = shared_data->preferences_window;
	RipOffPlugin plugin = shared_data->pref_object->selected;


	plugin_pref_window = ripoff_plugin_prefs(plugin);
	gtk_window_set_modal(GTK_WINDOW(plugin_pref_window), TRUE);
	gtk_window_set_transient_for(GTK_WINDOW(plugin_pref_window), GTK_WINDOW(window));
	gtk_window_set_type_hint(GTK_WINDOW(plugin_pref_window), GDK_WINDOW_TYPE_HINT_DIALOG);
	gtk_window_set_decorated(GTK_WINDOW(plugin_pref_window), TRUE);
	gtk_window_set_skip_taskbar_hint(GTK_WINDOW(plugin_pref_window), TRUE);

	gtk_widget_show_all(plugin_pref_window);
}

void close_preferences_window(GtkWidget *widget, gpointer data)
{
	const gchar *temp_buffer;
	pref_window_struct shared_data = (pref_window_struct)data;
	RipOffPreferences prefs = shared_data->pref_object;

	g_free(prefs->cd_path);
	g_free(prefs->file_pattern);
	g_free(prefs->output_folder);

	temp_buffer = gtk_entry_get_text(GTK_ENTRY(shared_data->cd_path_entry));
	prefs->cd_path = g_malloc(strlen(temp_buffer) + 1);
	strcpy(prefs->cd_path, temp_buffer);

	temp_buffer = gtk_entry_get_text(GTK_ENTRY(shared_data->format_entry));
	prefs->file_pattern= g_malloc(strlen(temp_buffer) + 1);
	strcpy(prefs->file_pattern, temp_buffer);

	temp_buffer = gtk_label_get_text(GTK_LABEL(shared_data->output_directory_label));
	prefs->output_folder= g_malloc(strlen(temp_buffer) + 1);
	strcpy(prefs->output_folder, temp_buffer);

	gtk_widget_destroy(shared_data->preferences_window);
	g_free(shared_data);
}

void close_preferences_event(GtkWidget *window, GdkEvent *event, gpointer data)
{
	close_preferences_window(window, data);
}
