# ifndef _RHEO_CGUP_ABTBC_H
# define _RHEO_CGUP_ABTBC_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:cgup_abtbc
NAME: @code{cgup_abtbc} -- Stokes solver
@findex cgup\_abtbc
@cindex Stokes problem
@cindex conjugate gradient algorithm
@cindex stabilized finite element method
DESCRIPTION: 
  @noindent
  Conjugate gradient applied to the stabilized Stokes system
  with the bock diagonal preconditionner: (inv(A),I)
EXAMPLE: 
SEE ALSO: cg, cg_abtbc 
DATE:   2 nov 1997
METHODS: @cgup_abtbc
End:
*/

namespace rheolef { 

//<cgup_abtbc:  
template<
    class MatrixSolver, 
    class MatrixPreconditioner, 
    class Matrix, class Vector, 
    class StokesPreconditioner, 
    class Real>
int
cgup_abtbc (
    const MatrixSolver&         m_solver,
    const Matrix&               A, 
    const MatrixPreconditioner& M_A,  
    const Matrix&               B,
    const Matrix&               C,
    Vector&                     U,
    Vector&                     P,
    const Vector&               F,
    const Vector&               G,
    const StokesPreconditioner& M,
    int&                        max_iter,
    Real&                       tol,
    ostream*                    p_cres = 0)

//>cgup_abtbc:  

{

  Real residu ;
  Vector R_U, R_P, Q_U, Q_P, Y_U, Y_P, Z_U=U, Z_P=P ;  
  Real alpha, beta, rho, rho_1=0 ;
  
  int status_ms = 0 ;
  
  R_U = F - (A*U + B.trans_mult(P)) ;
  R_P = G - (B*U - C*P) ;

  Real normFtilde = norm(F)+norm(G) ;

  if (normFtilde == Float(0.0)) normFtilde = Float(1.) ;

  if ( (residu = (norm(R_U)+norm(R_P))/normFtilde ) <= tol )
    {
      tol = residu ;
      max_iter = 0 ;
      return 0     ;
    }
  
  for ( int k=1 ; k <= max_iter ; k++ )
    {

      status_ms = m_solver( A, M_A, Z_U, R_U ) ;
      Z_P = R_P ;
      rho = dot(R_U,Z_U) + dot(R_P,Z_P) ;

      if (k==1) 
	{ 
	  Q_U = Z_U ;
	  Q_P = Z_P ;
	}
      else
	{
	  
	  beta = rho/rho_1 ;
	  Q_U  = Z_U + beta*Q_U ; 
	  Q_P  = Z_P + beta*Q_P ; 
	}

      Y_U = A*Q_U + B.trans_mult(Q_P) ;
      Y_P = B*Q_U - C*Q_P ;
      alpha = rho/( dot(Q_U,Y_U) + dot(Q_P,Y_P) ) ;
      
      U += alpha*Q_U ;
      P += alpha*Q_P ;

      R_U -= alpha*Y_U ;
      R_P -= alpha*Y_P ;

      if ( ( residu = (norm(R_U)+norm(R_P))/normFtilde ) <= tol )
	{
	  tol = residu ;
	  max_iter = k ;
	  return 0     ;
	}
      
      if (p_cres) *p_cres << k << " " << residu << "\n" ;
      
      rho_1 = rho ;
    }
  tol = residu ;
  return 1 ;

}
template<
    class MatrixPreconditionner, 
    class Matrix, 
    class Vector, 
    class Real>
int
cgup_abtbc(
    const Matrix&                a,
    const MatrixPreconditionner& ap,
    const Matrix&                b,
    const Matrix&                c,
    Vector&                      u,
    Vector&                      p,
    const Vector&                f,
    const Vector&                g,
    int&                         max_iter,
    Real&                        tol,
    ostream*                     p_cres = &std::cerr)
//>urm_abtb:
{
    return cgup_abtbc (ldlt_solver<MatrixPreconditionner, Matrix, Vector, Vector>(),
        a, ap, b, c, u, p, f, g, EYE, max_iter, tol, p_cres);
}

}// namespace rheolef
# endif // _RHEO_CGUP_ABTBC_H 
