/*	png.c
	Copyright (C) 2004-2007 Mark Tyler and Dmitry Groshev

	This file is part of rgbPaint.

	rgbPaint is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	rgbPaint is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with rgbPaint in the file COPYING.
*/


#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <gtk/gtk.h>

#include "global.h"

#include "memory.h"
#include "png.h"
#include "canvas.h"
#include "otherwindow.h"
#include "mainwindow.h"
#include "mygtk.h"


int silence_limit=18;


fformat file_formats[NUM_FTYPES2] = {
	{ "", "", "", 0},
	{ "PNG", "png", "", FF_IDX | FF_RGB | FF_ALPHA | FF_MULTI
		| FF_TRANS },
	{ "JPEG", "jpg", "jpeg", FF_RGB | FF_COMPR },
	{ "ICO", "ico", "icon", FF_RGB },
	{ "", "", "", 0}
};

int file_type_by_ext(char *name, guint32 mask)
{
	int i;
	char *ext = strrchr(name, '.');

	if (!ext || !ext[0]) return (FT_NONE);
	for (i = 0; i < NUM_FTYPES; i++)
	{
		if (!(file_formats[i].flags & mask)) continue;
		if (!strncasecmp(ext + 1, file_formats[i].ext, LONGEST_EXT))
			return (i);
		if (!file_formats[i].ext2[0]) continue;
		if (!strncasecmp(ext + 1, file_formats[i].ext2, LONGEST_EXT))
			return (i);
	}

	/* Special case for Gifsicle's victims */
	if ((mask & FF_IDX) && (ext - name > 4) &&
		!strncasecmp(ext - 4, ".gif", 4)) return (FT_GIF);

	return (FT_NONE);
}

/* Receives struct with image parameters, and channel flags;
 * returns 0 for success, or an error code;
 * success doesn't mean that anything was allocated, loader must check that;
 * loader may call this multiple times - say, for each channel */

static int allocate_image(ls_settings *settings, int cmask)
{
	int i, j;

	if ((settings->width < 1) || (settings->height < 1)) return (-1);

	if ((settings->width > MAX_WIDTH) || (settings->height > MAX_HEIGHT))
		return (TOO_BIG);

	// Don't show progress bar where there's no need
	if (settings->width * settings->height <= (1<<silence_limit))
		settings->silent = TRUE;

// !!! Currently allocations are created committed, have to rollback on error

	for (i = 0; i < NUM_CHANNELS; i++)
	{
		if (!(cmask & CMASK_FOR(i))) continue;

		// Overwriting is allowed
		if (settings->img[i]) continue;
		// No utility channels without image
		if ((i != CHN_IMAGE) && !settings->img[CHN_IMAGE]) return (-1);

		switch (settings->mode)
		{
		case FS_PNG_LOAD: // Regular image
			// Allocate the entire batch at once
			j = mem_new(settings->width, settings->height,
				settings->bpp, cmask);
			if (j) return (FILE_MEM_ERROR);
			memcpy(settings->img, mem_img, sizeof(chanlist));
			break;
		default: // Something has gone mad
			return (-1);
		}
	}
	return (0);
}


static int save_pbuf(char *file_name, ls_settings *settings, char *ftype, char *arg1, char *arg2)
{
	GError *error=NULL;
	gboolean res;
	GdkPixbuf *pbuf;

	if ( settings->bpp == 3 )
	{
		pbuf = gdk_pixbuf_new_from_data( settings->img[CHN_IMAGE], GDK_COLORSPACE_RGB, FALSE,
			8, settings->width, settings->height,
			settings->width * settings->bpp, NULL, NULL );

		if ( !pbuf ) return -1;		// Failed

		set_cursor_type(GDK_WATCH);
		res = gdk_pixbuf_save (pbuf, file_name, ftype, &error, arg1, arg2, NULL);
		set_cursor_type(GDK_LEFT_PTR);
		set_cursor();

		g_clear_error(&error);
		g_object_unref(pbuf);		// Remove structure

		if (res) return 0;		// Everything is OK
		else return -1;
	}
	else	return -1;			// Should never happen in rgbPaint
}



int save_image(char *file_name, ls_settings *settings)
{
	int res;
	char txt[20];

	switch (settings->ftype)
	{
	default:
	case FT_PNG:	res = save_pbuf(file_name, settings, "png", "compression", "9");
			break;
	case FT_JPEG:	snprintf(txt, 30, "%i", settings->jpeg_quality);
			res = save_pbuf(file_name, settings, "jpeg", "quality", txt);
			break;
	case FT_ICO:	res = save_pbuf(file_name, settings, "ico", NULL, NULL);
			break;
	}

	return res;
}

int load_clipboard(char *file_name)
{
	guchar *pixels, *dest_i, *dest_a, *src;
	gint i, j, w, h, rowstride, channels;
	GdkPixbuf *pbuf;

	pbuf = gdk_pixbuf_new_from_file( file_name, NULL );
	if ( !pbuf ) return -1;				// Error reading file

	w = gdk_pixbuf_get_width(pbuf);
	h = gdk_pixbuf_get_height(pbuf);

	pixels = gdk_pixbuf_get_pixels(pbuf);
	rowstride = gdk_pixbuf_get_rowstride(pbuf);
	channels = gdk_pixbuf_get_n_channels(pbuf);

	free(mem_clipboard);
	mem_clip_mask_clear();				// Lose current clipboard

	mem_clipboard = malloc(w * h * 3);		// Allocate memory for clipboard
	if ( !mem_clipboard )
	{
		free(mem_clipboard);
		g_object_unref(pbuf);			// Release pixbuf
		return -1;
	}

	if ( channels == 4 )
	{
		mem_clip_mask = calloc(w * h, 1);
		if ( !mem_clipboard || !mem_clip_mask )
		{
			free(mem_clipboard);
			g_object_unref(pbuf);		// Release pixbuf
			return -1;
		}
	}


	dest_i = mem_clipboard;
	dest_a = mem_clip_mask;
	for ( j=0; j<h; j++ )
	{
		src = pixels + rowstride*j;
		for ( i=0; i<w; i++ )
		{
			dest_i[0] = src[0];
			dest_i[1] = src[1];
			dest_i[2] = src[2];
			src += channels;
			dest_i += 3;
		}
		src = pixels + rowstride*j + 3;
		if ( channels>3 )		// Alpha data in file
		{
			for ( i=0; i<w; i++ )
			{
				dest_a[0] = src[0];
				src += channels;
				dest_a++;
			}
		}
	}
	mem_clip_w = w;
	mem_clip_h = h;
	mem_clip_bpp = 3;

	g_object_unref(pbuf);			// Release pixbuf

	return 1;		// Success
}

int load_image(char *file_name)
{
	png_color pal[256];
	ls_settings settings;
	GdkPixbuf *pbuf;
	guchar *pixels, *dest, *src;
	gint i, j, w, h, rowstride, channels;


	pbuf = gdk_pixbuf_new_from_file( file_name, NULL );
	if ( !pbuf ) return -1;						// Error reading file

	w = gdk_pixbuf_get_width(pbuf);
	h = gdk_pixbuf_get_height(pbuf);

	if ((w > MAX_WIDTH) || (h > MAX_HEIGHT))
	{
		g_object_unref(pbuf);					// Release pixbuf
		return (TOO_BIG);					// Image too big
	}

	init_ls_settings(&settings, NULL);
	settings.ftype = FT_PNG;
	settings.pal = pal;
	mem_pal_copy(pal, mem_pal_def);
	settings.colors = mem_pal_def_i;

	settings.width = w;
	settings.height = h;
	settings.bpp = 3;
	settings.mode = FS_PNG_LOAD;

	if ( allocate_image(&settings, CMASK_IMAGE) )
	{
		g_object_unref(pbuf);					// Release pixbuf
		return FILE_MEM_ERROR;
	}

	pixels = gdk_pixbuf_get_pixels(pbuf);
	rowstride = gdk_pixbuf_get_rowstride(pbuf);
	channels = gdk_pixbuf_get_n_channels(pbuf);
	dest = settings.img[CHN_IMAGE];
	for ( j=0; j<h; j++ )
	{
		src = pixels + rowstride*j;
		for ( i=0; i<w; i++ )
		{
			dest[0] = src[0];
			dest[1] = src[1];
			dest[2] = src[2];
			src+=channels;
			dest+=3;
		}
	}

	g_object_unref(pbuf);						// Release pixbuf

	return 1;		// Success
}
