% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/to_any_case.R
\name{to_any_case}
\alias{to_any_case}
\title{General case conversion}
\usage{
to_any_case(string, case = c("snake", "small_camel", "big_camel",
  "screaming_snake", "parsed", "mixed", "lower_upper", "upper_lower",
  "swap", "all_caps", "lower_camel", "upper_camel", "internal_parsing",
  "none", "flip", "sentence", "random", "title"), abbreviations = NULL,
  sep_in = "[^[:alnum:]]", parsing_option = 1,
  transliterations = NULL, numerals = c("middle", "left", "right",
  "asis", "tight"), sep_out = NULL, unique_sep = NULL,
  empty_fill = NULL, prefix = "", postfix = "")
}
\arguments{
\item{string}{A string (for example names of a data frame).}

\item{case}{The desired target case, provided as one of the following:
\itemize{
 \item{snake_case: \code{"snake"}}
 \item{lowerCamel: \code{"lower_camel"} or \code{"small_camel"}}
 \item{UpperCamel: \code{"upper_camel"} or \code{"big_camel"}}
 \item{ALL_CAPS: \code{"all_caps"} or \code{"screaming_snake"}}
 \item{lowerUPPER: \code{"lower_upper"}}
 \item{UPPERlower: \code{"upper_lower"}}
 \item{Sentence case: \code{"sentence"}}
 \item{Title Case: \code{"title"} - This one is basically the same as sentence case, but in addition it is wrapped into \code{tools::toTitleCase} and any \code{abbreviations} are always turned into upper case.}
}

 There are five "special" cases available:
\itemize{
 \item{\code{"parsed"}: This case is underlying all other cases. 
 Every substring a string consists
 of becomes surrounded by an underscore (depending on the \code{parsing_option}).
  Underscores at the start and end are trimmed. No lower or 
 upper case pattern from the input string are changed.}
 \item{\code{"mixed"}: Almost the same as \code{case = "parsed"}. Every letter which is not at the start
 or behind an underscore is turned into lowercase. If a substring is set as an abbreviation, it will be turned into upper case.}
 \item{\code{"swap"}: Upper case letters will be turned into lower case and vice versa. Also \code{case = "flip"} will work.
 Doesn't work with any of the other arguments except \code{unique_sep}, \code{empty_fill}, \code{prefix} and \code{postfix}.}
 \item{\code{"random"}: Each letter will be randomly turned into lower or upper case. Doesn't work with any of the other arguments except \code{unique_sep}, \code{empty_fill}, \code{prefix} and \code{postfix}.}
 \item{\code{"none"}: Neither parsing nor case conversion occur. This case might be helpful, when
 one wants to call the function for the quick usage of the other parameters.
 To suppress replacement of spaces to underscores set \code{sep_in = NULL}.
 Works with \code{sep_in}, \code{transliterations}, \code{sep_out}, \code{prefix},
  \code{postfix},
  \code{empty_fill} and \code{unique_sep}.}
 \item{\code{"internal_parsing"}: This case is returning the internal parsing
 (suppressing the internal protection mechanism), which means that alphanumeric characters will be surrounded by underscores.
 It should only be used in very rare use cases and is mainly implemented to showcase the internal workings of \code{to_any_case()}}
 }}

\item{abbreviations}{character. (Case insensitive) matched abbreviations are surrounded by underscores. In this way, they can get recognized by the parser. This is useful when e.g. \code{parsing_option} 1 is needed for the use case, but some abbreviations but some substrings would require \code{parsing_option} 2. Furthermore, this argument also specifies the formatting of abbreviations in the output for the cases title, mixed, lower and upper camel. E.g. for upper camel the first letter is always in upper case, but when the abbreviation is supplied in upper case, this will also be visible in the output.

Use this feature with care: One letter abbreviations and abbreviations next to each other are hard to read and also not easy to parse for further processing.}

\item{sep_in}{(short for separator input) if character, is interpreted as a
regular expression (wrapped internally into \code{stringr::regex()}). 
The default value is a regular expression that matches any sequence of
non-alphanumeric values. All matches will be replaced by underscores 
(additionally to \code{"_"} and \code{" "}, for which this is always true, even
if \code{NULL} is supplied). These underscores are used internally to split
the strings into substrings and specify the word boundaries.}

\item{parsing_option}{An integer that will determine the parsing_option.
\itemize{
 \item{1: \code{"RRRStudio" -> "RRR_Studio"}}
 \item{2: \code{"RRRStudio" -> "RRRS_tudio"}}
 \item{3: \code{"RRRStudio" -> "RRRSStudio"}. This will become for example \code{"Rrrstudio"} when we convert to lower camel case.}
 \item{-1, -2, -3: These \code{parsing_options}'s will suppress the conversion after non-alphanumeric values.}
 \item{0: no parsing}
 }}

\item{transliterations}{A character vector (if not \code{NULL}). The entries of this argument
need to be elements of \code{stringi::stri_trans_list()} (like "Latin-ASCII", which is often useful) or names of lookup tables (currently only "german" is supported). In the order of the entries the letters of the input
 string will be transliterated via \code{stringi::stri_trans_general()} or replaced via the 
 matches of the lookup table. When named character elements are supplied as part of `transliterations`, anything that matches the names is replaced by the corresponding value.
You should use this feature with care in case of \code{case = "parsed"}, \code{case = "internal_parsing"} and 
\code{case = "none"}, since for upper case letters, which have transliterations/replacements
 of length 2, the second letter will be transliterated to lowercase, for example Oe, Ae, Ss, which
 might not always be what is intended. In this case you can make usage of the option to supply named elements and specify the transliterations yourself.}

\item{numerals}{A character specifying the alignment of numerals (\code{"middle"}, \code{left}, \code{right}, \code{asis} or \code{tight}). I.e. \code{numerals = "left"} ensures that no output separator is in front of a digit.}

\item{sep_out}{(short for separator output) String that will be used as separator. The defaults are \code{"_"} 
and \code{""}, regarding the specified \code{case}. When \code{length(sep_out) > 1}, the last element of \code{sep_out} gets recycled and separators are incorporated per string according to their order.}

\item{unique_sep}{A string. If not \code{NULL}, then duplicated names will get 
a suffix integer
in the order of their appearance. The suffix is separated by the supplied string
 to this argument.}

\item{empty_fill}{A string. If it is supplied, then each entry that matches "" will be replaced
by the supplied string to this argument.}

\item{prefix}{prefix (string).}

\item{postfix}{postfix (string).}
}
\value{
A character vector according the specified parameters above.
}
\description{
Function to convert strings to any case
}
\note{
\code{to_any_case()} is vectorised over \code{string}, \code{sep_in}, \code{sep_out},
 \code{empty_fill}, \code{prefix} and \code{postfix}.
}
\examples{
### abbreviations
to_snake_case(c("HHcity", "newUSElections"), abbreviations = c("HH", "US"))
to_upper_camel_case("succesfullGMBH", abbreviations = "GmbH")
to_title_case("succesfullGMBH", abbreviations = "GmbH")

### sep_in (input separator)
string <- "R.St\\u00FCdio: v.1.0.143"
to_any_case(string)
to_any_case(string, sep_in = ":|\\\\.")
to_any_case(string, sep_in = ":|(?<!\\\\d)\\\\.")
            
### parsing_option
# the default option makes no sense in this setting
to_parsed_case("HAMBURGcity", parsing_option = 1)
# so the second parsing option is the way to address this example
to_parsed_case("HAMBURGcity", parsing_option = 2)
# By default (option 1) characters are converted after non alpha numeric characters.
# To suppress this behaviour add a minus to the parsing_option
to_upper_camel_case("lookBehindThe.dot", parsing_option = -1)
# For some exotic cases parsing option 3 might be of interest
to_parsed_case("PARSingOption3", parsing_option = 3)
# There may be reasons to suppress the parsing
to_any_case("HAMBURGcity", parsing_option = 0)

### transliterations
to_any_case("\\u00E4ngstlicher Has\\u00EA", transliterations = c("german", "Latin-ASCII"))

### case
strings <- c("this Is a Strange_string", "AND THIS ANOTHER_One")
to_any_case(strings, case = "snake")
to_any_case(strings, case = "lower_camel") # same as "small_camel"
to_any_case(strings, case = "upper_camel") # same as "big_camel"
to_any_case(strings, case = "all_caps") # same as "screaming_snake"
to_any_case(strings, case = "lower_upper")
to_any_case(strings, case = "upper_lower")
to_any_case(strings, case = "sentence")
to_any_case(strings, case = "title")
to_any_case(strings, case = "parsed")
to_any_case(strings, case = "mixed")
to_any_case(strings, case = "swap")
to_any_case(strings, case = "random")
to_any_case(strings, case = "none")
to_any_case(strings, case = "internal_parsing")

### numerals
to_snake_case("species42value 23month 7-8", numerals = "asis")
to_snake_case("species42value 23month 7-8", numerals = "left")
to_snake_case("species42value 23month 7-8", numerals = "right")
to_snake_case("species42value 23month 7-8", numerals = "middle")
to_snake_case("species42value 23month 7-8", numerals = "tight")

### sep_out (output separator)
string <- c("lowerCamelCase", "ALL_CAPS", "I-DontKNOWWhat_thisCASE_is")
to_snake_case(string, sep_out = ".")
to_mixed_case(string, sep_out = " ")
to_screaming_snake_case(string, sep_out = "=")

### empty_fill
to_any_case(c("","",""), empty_fill = c("empty", "empty", "also empty"))

### unique_sep
to_any_case(c("same", "same", "same", "other"), unique_sep = c(">"))

### prefix and postfix
to_upper_camel_case("some_path", sep_out = "//", 
  prefix = "USER://", postfix = ".exe")

}
\seealso{
\href{https://github.com/Tazinho/snakecase}{snakecase on github} or 
\code{\link{caseconverter}} for some handy shortcuts.
}
\author{
Malte Grosser, \email{malte.grosser@gmail.com}
}
\keyword{utilities}
