% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/src_dbi.R
\name{src_dbi}
\alias{src_dbi}
\alias{tbl.src_dbi}
\alias{tbl_dbi}
\title{dplyr backend for any DBI-compatible database}
\usage{
src_dbi(con, auto_disconnect = FALSE)

\method{tbl}{src_dbi}(src, from, ...)
}
\arguments{
\item{con}{An object that inherits from \link[DBI:DBIConnection-class]{DBI::DBIConnection},
typically generated by \link[DBI:dbConnect]{DBI::dbConnect}}

\item{auto_disconnect}{Should the connection be automatically closed when
the src is deleted? Set to \code{TRUE} if you initialize the connection
the call to \code{src_dbi()}. Pass \code{NA} to auto-disconnect but print a message
when this happens.}

\item{src}{Either a \code{src_dbi} or \code{DBIConnection}}

\item{from}{Either a string (giving a table name) or literal \code{\link[=sql]{sql()}}.}

\item{...}{Needed for compatibility with generic; currently ignored.}
}
\value{
An S3 object with class \code{src_dbi}, \code{src_sql}, \code{src}.
}
\description{
\code{src_dbi()} is a general dplyr backend that connects to any
DBI driver. \code{src_memdb()} connects to a temporary in-memory SQLite
database, that's useful for testing and experimenting.

You can generate a \code{tbl()} directly from the DBI connection, or
go via \code{src_dbi()}.
}
\details{
All data manipulation on SQL tbls are lazy: they will not actually
run the query or retrieve the data unless you ask for it: they all return
a new \code{tbl_dbi} object. Use \code{\link[=compute]{compute()}} to run the query and save the
results in a temporary in the database, or use \code{\link[=collect]{collect()}} to retrieve the
results to R. You can see the query with \code{\link[=show_query]{show_query()}}.

For best performance, the database should have an index on the variables
that you are grouping by. Use \code{\link[=explain]{explain()}} to check that the database is using
the indexes that you expect.

There is one excpetion: \code{\link[=do]{do()}} is not lazy since it must pull the data
into R.
}
\examples{
# Basic connection using DBI -------------------------------------------
library(dplyr)

con <- DBI::dbConnect(RSQLite::SQLite(), ":memory:")
src <- src_dbi(con, auto_disconnect = TRUE)

# Add some data
copy_to(src, mtcars)
src
DBI::dbListTables(con)

# To retrieve a single table from a source, use `tbl()`
src \%>\% tbl("mtcars")

# You can also use pass raw SQL if you want a more sophisticated query
src \%>\% tbl(sql("SELECT * FROM mtcars WHERE cyl == 8"))

# Alternatively, you can use the `src_sqlite()` helper
src2 <- src_sqlite(":memory:", create = TRUE)

# If you just want a temporary in-memory database, use src_memdb()
src3 <- src_memdb()

# To show off the full features of dplyr's database integration,
# we'll use the Lahman database. lahman_sqlite() takes care of
# creating the database.

if (has_lahman("sqlite")) {
lahman_p <- lahman_sqlite()
batting <- lahman_p \%>\% tbl("Batting")
batting

# Basic data manipulation verbs work in the same way as with a tibble
batting \%>\% filter(yearID > 2005, G > 130)
batting \%>\% select(playerID:lgID)
batting \%>\% arrange(playerID, desc(yearID))
batting \%>\% summarise(G = mean(G), n = n())

# There are a few exceptions. For example, databases give integer results
# when dividing one integer by another. Multiply by 1 to fix the problem
batting \%>\%
  select(playerID:lgID, AB, R, G) \%>\%
  mutate(
   R_per_game1 = R / G,
   R_per_game2 = R * 1.0 / G
 )

# All operations are lazy: they don't do anything until you request the
# data, either by `print()`ing it (which shows the first ten rows),
# or by `collect()`ing the results locally.
system.time(recent <- filter(batting, yearID > 2010))
system.time(collect(recent))

# You can see the query that dplyr creates with show_query()
batting \%>\%
  filter(G > 0) \%>\%
  group_by(playerID) \%>\%
  summarise(n = n()) \%>\%
  show_query()
}
}
