///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/ViewportGrid.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportConfiguration.h>
#include <core/actions/ActionManager.h>
#include <base/linalg/Plane.h>

namespace Core {

/******************************************************************************
* The constructor of the viewport grid class.
******************************************************************************/
ViewportGrid::ViewportGrid() : viewport(NULL),
	_gridMatrix(IDENTITY), _inverseGridMatrix(IDENTITY), _gridSpacing(1.0)
{
}

/******************************************************************************
* Returns whether the grid is currently enabled.
******************************************************************************/
bool ViewportGrid::isVisible() const
{
	return viewport->settings()->isGridShown();
}

/******************************************************************************
* Turns the grid on or off.
******************************************************************************/
void ViewportGrid::setVisible(bool visible)
{
	viewport->settings()->setGridShown(visible);
}

/******************************************************************************
* Sets the Orientation of the grid plane.
******************************************************************************/
void ViewportGrid::setGridMatrix(const AffineTransformation& gridmat)
{
	_gridMatrix = gridmat;
	_inverseGridMatrix = gridmat.inverse();
	viewport->updateViewport(true);
}

/******************************************************************************
* Renders the grid into the viewport.
******************************************************************************/
void ViewportGrid::renderGrid()
{
	CHECK_POINTER(viewport);
	if(!isVisible()) return;
	if(visibleGridRect().isEmpty() || gridSpacing() == 0) return;	// No grid to render.

	viewport->setWorldMatrix(gridMatrix());
	viewport->setDepthTest(false);
	viewport->setBackfaceCulling(true);
	viewport->setLightingEnabled(false);

    int xstart = (int)floor(visibleGridRect().minc.X / (gridSpacing() * 10)) * 10;
	int numLinesX = (int)ceil(visibleGridRect().maxc.X / (gridSpacing() * 10)) * 10 - xstart + 1;
	int ystart = (int)floor(visibleGridRect().minc.Y / (gridSpacing() * 10)) * 10;
	int numLinesY = (int)ceil(visibleGridRect().maxc.Y / (gridSpacing() * 10)) * 10 - ystart + 1;
	OVITO_ASSERT(numLinesX >= 0 && numLinesY >= 0);
	OVITO_ASSERT((xstart % 10) == 0);
	OVITO_ASSERT((ystart % 10) == 0);
	OVITO_ASSERT(((numLinesX-1) % 10) == 0);
	OVITO_ASSERT(((numLinesY-1) % 10) == 0);

	FloatType xstartF = (FloatType)xstart * gridSpacing();
	FloatType ystartF = (FloatType)ystart * gridSpacing();
	FloatType xendF = (FloatType)(xstart + numLinesX - 1) * gridSpacing();
	FloatType yendF = (FloatType)(ystart + numLinesY - 1) * gridSpacing();

	// Allocate vertex buffer.
	int numVertices = 2 * (numLinesX + numLinesY);
	Point3* verts = new Point3[numVertices];

	// Build lines array.
	Color color = Viewport::getVPColor(Viewport::COLOR_GRID);
	Color majorColor = Viewport::getVPColor(Viewport::COLOR_GRID_INTENS);
	Color majorMajorColor = Viewport::getVPColor(Viewport::COLOR_GRID_AXIS);

	Point3* v = verts;
	FloatType x = xstartF;
	for(int i=xstart; i<xstart + numLinesX; i++, x += gridSpacing()) {
		if((i % 10) != 0) {
			*v++ = Point3(x, ystartF, 0);
			*v++ = Point3(x, yendF, 0);
		}
	}
	FloatType y = ystartF;
	for(int i=ystart; i<ystart + numLinesY; i++, y += gridSpacing()) {
		if((i % 10) != 0) {
			*v++ = Point3(xstartF, y, 0);
			*v++ = Point3(xendF, y, 0);
		}
	}

	Box3 boundingBox(Point3(xstartF, ystartF, 0), Point3(xendF, yendF, 0));

	OVITO_ASSERT(v - verts <= numVertices);
	viewport->setRenderingColor(Viewport::getVPColor(Viewport::COLOR_GRID));
	viewport->renderLines(v - verts, boundingBox, verts);

	v = verts;
	x = xstartF;
	for(int i=xstart; i<xstart + numLinesX; i++, x += gridSpacing()) {
		if((i % 10) == 0 && i != 0) {
			*v++ = Point3(x, ystartF, 0);
			*v++ = Point3(x, yendF, 0);
		}
	}
	y = ystartF;
	for(int i=ystart; i<ystart + numLinesY; i++, y += gridSpacing()) {
		if((i % 10) == 0 && i != 0) {
			*v++ = Point3(xstartF, y, 0);
			*v++ = Point3(xendF, y, 0);
		}
	}
	OVITO_ASSERT(v - verts <= numVertices);
	viewport->setRenderingColor(Viewport::getVPColor(Viewport::COLOR_GRID_INTENS));
	viewport->renderLines(v - verts, boundingBox, verts);

	v = verts;
	x = xstartF;
	if(xstartF <= 0.0 && xendF >= 0.0) {
		*v++ = Point3(0, ystartF, 0);
		*v++ = Point3(0, yendF, 0);
	}
	if(ystartF <= 0.0 && yendF >= 0.0) {
		*v++ = Point3(xstartF, 0, 0);
		*v++ = Point3(xendF, 0, 0);
	}
	OVITO_ASSERT(v - verts <= numVertices);
	viewport->setRenderingColor(Viewport::getVPColor(Viewport::COLOR_GRID_AXIS));
	viewport->renderLines(v - verts, boundingBox, verts);

	delete[] verts;
}

/******************************************************************************
* Calculates the visible grid area.
******************************************************************************/
void ViewportGrid::updateGrid()
{
	static const Point2 testPoints[] = {
		Point2(-1,-1), Point2( 1,-1), Point2( 1, 1), Point2(-1, 1),
		Point2(0,1), Point2(0,-1), Point2(1,0), Point2(-1,0),
		Point2(0,1), Point2(0,-1), Point2(1,0), Point2(-1,0),
		Point2(-1, 0.5), Point2(-1,-0.5), Point2(1,-0.5), Point2(1,0.5),
		Point2(0,0)
	};

	// Compute intersection points of test rays with grid plane.
	_visibleGridRect.setEmpty();
	size_t numberOfIntersections = 0;
	for(size_t i = 0; i < sizeof(testPoints)/sizeof(testPoints[0]); i++) {
		Point3 p;
		if(viewportComputePlaneIntersection(testPoints[i], p, 0.2f)) {
			numberOfIntersections++;
			_visibleGridRect.addPoint(p.X, p.Y);
		}
	}

	if(numberOfIntersections < 2) {
		// Cannot determine visible parts of the grid.
		_visibleGridRect.setEmpty();
		_gridSpacing = 0;
	}
	else {
		// Use adaptive grid spacing.
		Point3 gridCenter(_visibleGridRect.center().X, _visibleGridRect.center().Y, 0.0);
		_gridSpacing = viewport->nonScalingSize(gridMatrix() * gridCenter) * 2.0;
		// Round to nearest power of 10.
		_gridSpacing = pow((FloatType)10.0, floor(log10(_gridSpacing)));
	}
}

/******************************************************************************
* Computes the intersection of a ray going through a point on in the
* viewport projection plane and the grid plane.
*
* Returns true if an intersection has been found.
******************************************************************************/
bool ViewportGrid::viewportComputePlaneIntersection(const Point2& viewportPosition, Point3& intersectionPoint, FloatType epsilon)
{
	// The construction plane in grid coordinates.
	Plane3 gridPlane = Plane3(Vector3(0,0,1), 0);

	// Compute the ray and transform it to the grid coordinate system.
	Ray3 ray = inverseGridMatrix() * viewport->viewportRay(viewportPosition);

	// Compute intersection point.
	FloatType t = gridPlane.intersectionT(ray, epsilon);
    if(t == numeric_limits<FloatType>::max()) return false;
	if(viewport->isPerspectiveProjection() && t <= 0.0) return false;

	intersectionPoint = ray.point(t);
	intersectionPoint.Z = 0.0;

	return true;
}

/******************************************************************************
* Computes the intersection of a ray going through a point on in the
* viewport projection plane and the grid plane.
*
* Returns true if an intersection has been found.
******************************************************************************/
bool ViewportGrid::screenComputePlaneIntersection(const Point2I& screenPosition, Point3& intersectionPoint, FloatType epsilon)
{
	// Just convert input window point to viewport coordinates.
	return viewportComputePlaneIntersection(viewport->screenToViewport(screenPosition), intersectionPoint, epsilon);
}

/******************************************************************************
* Computes a length.
******************************************************************************/
FloatType ViewportGrid::computeConstructionLength(const Ray3& ray, const Point2I& p1, const Point2I& p2)
{
	if(p2 == p1) return 0;
	FloatType length = 0;

	// Compute the two rays and transform them to the grid coordinate system.
	Ray3 ray1 = inverseGridMatrix() * viewport->screenRay(p1);
	Ray3 ray2 = inverseGridMatrix() * viewport->screenRay(p2);

	Vector3 c = CrossProduct(ray.dir, ray1.dir);
	Vector3 dir = ray.dir;
	if(LengthSquared(c) < 1e-4) {
		dir = Normalize(inverseGridMatrix() * viewport->inverseViewMatrix() * Vector3(0,1,0));
		c = CrossProduct(dir, ray1.dir);
	}
	if(LengthSquared(c) < 1e-4) {
		dir = (ray2.base + ray2.dir) - (ray1.base + ray1.dir);
		c = CrossProduct(dir, ray1.dir);
	}

	Plane3 plane(ray.base, dir, c);
	FloatType t = plane.intersectionT(ray2, FLOATTYPE_EPSILON);
	if(t != FLOATTYPE_MAX) {
		Point3 isec = ray2.point(t);
		length = DotProduct(isec - ray.base, dir);
	}

    // Snap length.
	if(ACTION_MANAGER.findActionProxy(ACTION_SNAPPING_OBJECT)->isChecked())
		length = floor(length / gridSpacing() + 0.5) * gridSpacing();

	return length;
}

};
