///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/objects/CreationMode.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/input/ViewportInputManager.h>
#include <core/scene/animation/AnimManager.h>
#include <core/scene/ObjectNode.h>
#include <core/scene/SceneRoot.h>
#include <core/scene/objects/SceneObject.h>
#include <core/data/DataSetManager.h>
#include <core/gui/properties/PropertiesPanel.h>

namespace Core {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(CreationMode, SimpleInputHandler)
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(SimpleCreationMode, CreationMode)

/******************************************************************************
* Default constructor.
******************************************************************************/
SimpleCreationMode::SimpleCreationMode(const QString& _undoDisplayName, const QString& _objectBaseName) : CreationMode(),
	createOperation(NULL), objNode(NULL), obj(NULL), _clickCount(0), undoDisplayName(_undoDisplayName), objectBaseName(_objectBaseName)
{
}

/******************************************************************************
* Is called when the user aborts the operation by clicking the right
* mouse button or activating another input mode. 
******************************************************************************/
void SimpleCreationMode::onAbort()
{
	SimpleInputHandler::onAbort();

	if(createOperation == NULL) return;
	OVITO_ASSERT(!adjustOperations.empty());
	OVITO_ASSERT(obj != NULL);

	// Close properties editor
	if(propertiesPanel() != NULL)
		propertiesPanel()->setEditObject(NULL);

	while(!adjustOperations.empty())
		abortAdjustOperation();
	createOperation->clear();
	UNDO_MANAGER.endCompoundOperation();	
	createOperation = NULL;
	objNode = NULL;
	obj = NULL;
	_clickCount = 0;
}

/******************************************************************************
* Is called when the user finishes the operation. 
******************************************************************************/
void SimpleCreationMode::onFinish()
{
	SimpleInputHandler::onFinish();

	if(createOperation == NULL) return;
	OVITO_ASSERT(!adjustOperations.empty());
	OVITO_ASSERT(obj != NULL);

	while(!adjustOperations.empty())
		commitAdjustOperation();
	UNDO_MANAGER.endCompoundOperation();
	createOperation = NULL;
	objNode = NULL;
	obj = NULL;
	_clickCount = 0;
}

/******************************************************************************
* Will be called when the user presses the left mouse button.
******************************************************************************/
void SimpleCreationMode::onMousePressed(QMouseEvent* event)
{
	if(clickCount() != 0) return;
	try {
		OVITO_ASSERT(createOperation == NULL);
		OVITO_ASSERT(adjustOperations.empty());

		// Begin recording operation.
		createOperation = UNDO_MANAGER.beginCompoundOperation(undoDisplayName);
		
		// Do not create any animation keys.
		AnimationSuspender animSuspender;

		// Create object.
		obj = createObject();
		CHECK_OBJECT_POINTER(obj);

		SceneRoot* scene = DATASET_MANAGER.currentSet()->sceneRoot();
		CHECK_OBJECT_POINTER(scene);

		{ 
			UndoSuspender unsoSuspender;	// Do not create undo records for this.
			
			// Create scene node for object
			objNode = createNode(obj.get());
			CHECK_OBJECT_POINTER(objNode);

			// Give the new node a name.
			objNode->setName(scene->makeNameUnique(objectBaseName));
		}

		// Insert node into scene.
		scene->addChild(objNode);

		// Select new node.
		DATASET_MANAGER.currentSet()->selection()->clear();
		DATASET_MANAGER.currentSet()->selection()->add(objNode);

		// Show properties of new object.
		if(propertiesPanel() != NULL)
			propertiesPanel()->setEditObject(obj);

		// Begin recording adjusting object operation.
		beginAdjustOperation();

		// This is the first click.
		_clickCount = 1;
	}
	catch(const Exception& ex) {
		onAbort();
		ex.showError();
	}
}

/******************************************************************************
* Will be called when the user releases the left mouse button.
******************************************************************************/
void SimpleCreationMode::onMouseReleased(QMouseEvent* event)
{	
	_clickCount++;
}

/******************************************************************************
* Will be called when the user moves the mouse while 
* the operation is active.
******************************************************************************/
void SimpleCreationMode::onMouseDrag(QMouseEvent* event)
{	
}

/******************************************************************************
* Will be called when the user moves the mouse while
* the operation is not active.
******************************************************************************/
void SimpleCreationMode::onMouseFreeMove(Viewport& vp, QMouseEvent* event)
{
	snapPreview(vp, event);
}

/******************************************************************************
* Starts a new adjust operation.
******************************************************************************/
void SimpleCreationMode::beginAdjustOperation()
{
	CompoundOperation* adjustOperation = UNDO_MANAGER.beginCompoundOperation("Create Object Adjust Operation");
	adjustOperations.push_back(adjustOperation);
}

/******************************************************************************
* Undoes the current adjust operation and removes it from the undo stack.
******************************************************************************/
void SimpleCreationMode::abortAdjustOperation()
{
	OVITO_ASSERT(!adjustOperations.empty());
	CompoundOperation* adjustOperation = adjustOperations.back();
	adjustOperations.pop_back();
	adjustOperation->clear();
	UNDO_MANAGER.endCompoundOperation();
}

/******************************************************************************
* Finishes an adjust operation and puts it on the undo stack.
******************************************************************************/
void SimpleCreationMode::commitAdjustOperation()
{
	OVITO_ASSERT(!adjustOperations.empty());
	adjustOperations.pop_back();
	UNDO_MANAGER.endCompoundOperation();
}

/******************************************************************************
* This creates the scene node for the object.
******************************************************************************/
ObjectNode::SmartPtr SimpleCreationMode::createNode(SceneObject* obj)
{
	ObjectNode::SmartPtr node = new ObjectNode();
	node->setSceneObject(obj);
	return node;
}


};
