///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/data/units/ParameterUnit.h>

namespace Core {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(ParameterUnit, PluginClass)
IMPLEMENT_PLUGIN_CLASS(FloatParameterUnit, ParameterUnit)
IMPLEMENT_PLUGIN_CLASS(IntegerParameterUnit, ParameterUnit)
IMPLEMENT_PLUGIN_CLASS(WorldParameterUnit, FloatParameterUnit)
IMPLEMENT_PLUGIN_CLASS(AngleParameterUnit, FloatParameterUnit)
IMPLEMENT_PLUGIN_CLASS(PercentParameterUnit, FloatParameterUnit)
IMPLEMENT_PLUGIN_CLASS(TimeParameterUnit, IntegerParameterUnit)

/// The singleton instance of this class.
UnitsManager* UnitsManager::_singletonInstance = NULL;

/******************************************************************************
* Initializes the units manager.
******************************************************************************/
UnitsManager::UnitsManager()
{
	// Create a shortcut to the identity unit.
	floatIdentityUnit = new FloatParameterUnit();
	integerIdentityUnit = new IntegerParameterUnit();
	units[PLUGINCLASSINFO(FloatParameterUnit)] = floatIdentityUnit;
	units[PLUGINCLASSINFO(IntegerParameterUnit)] = integerIdentityUnit;
}

/******************************************************************************
* Returns the global instance of the given parameter unit service. 
******************************************************************************/
ParameterUnit* UnitsManager::getUnit(PluginClassDescriptor* parameterUnitClass)
{
	CHECK_POINTER(parameterUnitClass);
	OVITO_ASSERT_MSG(parameterUnitClass->isKindOf(PLUGINCLASSINFO(ParameterUnit)), "UnitsManager::getUnit()", "A ParameterUnit derived must be specified.");

	map<PluginClassDescriptor*, ParameterUnit::SmartPtr>::iterator iter = units.find(parameterUnitClass);
	if(iter != units.end())
		return iter->second.get();

	// Create an instance of this class.
	ParameterUnit::SmartPtr unit = static_object_cast<ParameterUnit>(parameterUnitClass->createInstance());
	units.insert(make_pair(parameterUnitClass, unit));

	return unit.get();
}

};
