%feature("docstring") OT::QuantileConfidence
R"RAW(Estimate confidence intervals of a quantile.

Refer to :ref:`quantile_confidence_estimation`.

.. warning::
    This class is experimental and likely to be modified in future releases.
    To use it, import the ``openturns.experimental`` submodule.

Parameters
----------
alpha : float
    Quantile level
beta : float, optional
    Confidence level. Default value is 0.95

Notes
-----
This class estimates bounds of the quantile of level :math:`\alpha \in [0,1]` of the random variable :math:`X`
with a confidence greater than :math:`\beta` using order statistics.

Let :math:`x_{\alpha}` be the unknown quantile of level :math:`\alpha` of the random variable :math:`X` of dimension 1.
Let :math:`(X_0, \dots, X_{\sampleSize-1})` be a sample of independent and identically distributed variables according to :math:`X`.

The bounds of the interval are computed from order statistics that we now introduce.
Let :math:`X_{(k)}` be the :math:`k+1` -th order statistics of :math:`(X_0, \dots, X_{\sampleSize-1})` for :math:`0 \leq k \leq \sampleSize -1`.
For example, :math:`X_{(0)} = \min (X_0, \dots, X_{\sampleSize-1})`
and :math:`X_{(\sampleSize -1)} = \max (X_0, \dots, X_{\sampleSize-1})`.

We have:

.. math::

    X_{(0)} \leq X_{(1)} \leq \dots \leq X_{(\sampleSize -1)}

Care: Python enumeration begins at 0 so that the mathematical :math:`X_{(k)}` order statistics of the theoretical file
:ref:`quantile_confidence_estimation` where :math:`1 \leq k \leq n` corresponds to the order statistics :math:`X_{(k-1)}` given by the library
with this class `QuantileConfidence`.


Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> sample = ot.Gumbel().getSample(100)
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> ci = algo.computeUnilateralConfidenceInterval(sample)
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeUnilateralRank
R"RAW(Evaluate the rank of a quantile lower bound.

The lower rank :math:`k_{low}` is the largest rank :math:`k`  with :math:`0 \leq k \leq \sampleSize -1` such that:

.. math::

    \Prob{X_{(k)} \leq x_{\alpha}} \geq \beta.

In other words, the interval :math:`\left[ X_{(k_{low})}, +\infty\right[` is a unilateral confidence
interval for the quantile :math:`x_\alpha` with confidence :math:`\beta`.

The upper rank :math:`k_{up}` is the smallest rank :math:`k` with :math:`0 \leq k \leq \sampleSize -1` such that:

.. math::

    \Prob{x_{\alpha} \leq X_{(k)}} \geq \beta.

In other words, the interval :math:`\left]-\infty, X_{(k_{up})}\right]` is a unilateral confidence
interval for the quantile :math:`x_\alpha` with confidence :math:`\beta`.

See  :ref:`quantile_confidence_estimation`
to get the conditions of the existence of a solution and the solution for each case.

Parameters
----------
size : int
    Sample size
lower_bounded : bool, optional
    True indicates the interval is bounded by a lower value.

    False indicates the interval is bounded by an upper value.

    Default value is False.

Returns
-------
rank : int
    Rank :math:`k`.

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> rank = algo.computeUnilateralRank(100)
>>> print(rank)
9
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeBilateralRank
R"RAW(Evaluate the ranks of a quantile bilateral bound.

The ranks :math:`k_1, k_2` with :math:`0 \leq k_1, k_2 \leq \sampleSize -1` are defined by:

.. math::

    \begin{array}{ll}
    (k_1, k_2) = & \argmin \Prob{X_{(k_1)} \leq x_{\alpha} \leq X_{(k_2)}}\\
                 & \mbox{s.t.} \Prob{X_{(k_1)} \leq x_{\alpha} \leq X_{(k_2)}} \geq \beta
    \end{array}

In other words, the interval :math:`\left[ X_{(k_1)}, X_{(k_2)} \right[` is a bilateral confidence
interval for the quantile :math:`x_\alpha` with confidence :math:`\beta`.

See  :ref:`quantile_confidence_estimation`
to get the conditions of the existence of a solution and the solution.
 

Parameters
----------
size : int
    Sample size :math:`\sampleSize`.

Returns
-------
ranks : :class:`~openturns.Indices` of size 2
    Pair of lower and upper ranks :math:`(k_1, k_2)`.

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> k1, k2 = algo.computeBilateralRank(100)
>>> print((k1, k2))
(1, 10)
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeUnilateralConfidenceInterval
R"RAW(Evaluate an unilateral confidence interval of a quantile.

The lower bounded confidence interval  is given by the order statistics:

.. math::

    [X_{(k_{low})}, +\infty[

where :math:`k_{low}` is the largest rank with :math:`0 \leq k_{low} \leq \sampleSize - 1` so that:

.. math::

    \Prob{X_{(k_{low})} \leq x_{\alpha}} \geq \beta.

The upper bounded confidence interval :

.. math::

    ]-\infty, X_{(k_{up})}]

where :math:`k_{up}` is the smallest rank with :math:`0 \leq k_{up} \leq \sampleSize - 1` so that:

.. math::

    \Prob{x_{\alpha} \leq X_{(k_{up})}} \geq \beta

The lower or upper ranks :math:`k_{low}` and :math:`k_{up}` are given by :meth:`computeUnilateralRank`.

The statistics values are evaluated from the given sample.

Parameters
----------
sample : 2-d sequence of float
    Quantile level
lower_bounded : bool, optional
    True indicates the interval is bounded by a lower value.

    False indicates the interval is bounded by an upper value.

    Default value is False.

Returns
-------
ci : :class:`~openturns.Interval`
    Quantile confidence interval.

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> sample = ot.Gumbel().getSample(100)
>>> ci = algo.computeUnilateralConfidenceInterval(sample)
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeUnilateralConfidenceIntervalWithCoverage
R"RAW(Evaluate an unilateral confidence interval of a quantile.

Refer to :func:`computeUnilateralConfidenceInterval()`

Parameters
----------
sample : 2-d sequence of float
    Quantile level
lower_bounded : bool, optional
    True indicates the interval is bounded by a lower value.
    False indicates the interval is bounded by an upper value.
    Default value is False.

Returns
-------
ci : :class:`~openturns.Interval`
    Quantile confidence interval.

coverage : float
    If `lower_bounded` is True, then the `coverage` is the value of :math:`\Prob{X_{(k_1)} \leq x_{\alpha} })`.
    Otherwise, the `coverage` is the value of :math:`\Prob{x_{\alpha} \leq X_{(k_2)}}`.
    In both cases, the coverage is guaranteed to be greater or equal to :math:`\beta`.

Notes
-----
Refer to :func:`computeUnilateralConfidenceInterval()`. In addition to the confidence interval,
the method returns the supremum of :math:`\beta` for the confidence interval.

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> sample = ot.Gumbel().getSample(100)
>>> ci, coverage = algo.computeUnilateralConfidenceIntervalWithCoverage(sample)
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeBilateralConfidenceInterval
R"RAW(Evaluate a bilateral confidence interval of a quantile.

The confidence interval for the quantile of level :math:`\alpha` is given by the order statistics:

.. math::

    [X_{(k_1)}, X_{(k_2)}]

where :math:`(k_1, k_2)`  are the ranks with :math:`0 \leq k_1 \leq k_2 \leq n - 1` defined by:

.. math::

    \begin{array}{ll}
    (k_1, k_2) = & \argmin \Prob{X_{(k_1)} \leq x_{\alpha} \leq X_{(k_2)}}\\
                 & \mbox{s.t.} \Prob{X_{(k_1)} \leq x_{\alpha} \leq X_{(k_2)}} \geq \beta
    \end{array}

The ranks :math:`(k_1, k_2)` are given by :meth:`computeBilateralRank`.

The statistics values are evaluated from the given sample.

Parameters
----------
sample : 2-d sequence of float
    Sample of the variable :math:`X`.

Returns
-------
ci : :class:`~openturns.Interval`
    Quantile confidence interval.

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> sample = ot.Gumbel().getSample(60)
>>> ci = algo.computeBilateralConfidenceInterval(sample)
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeBilateralConfidenceIntervalWithCoverage
R"RAW(Evaluate a bilateral confidence interval of a quantile.

Refer to :func:`computeBilateralConfidenceInterval()`

Parameters
----------
sample : 2-d sequence of float
    Sample of the variable :math:`X`.

Returns
-------
ci : :class:`~openturns.Interval`
    Quantile confidence interval.

coverage : float
    The `coverage` is the value of :math:`\Prob{X_{(k_1)} \leq x_{\alpha} \leq X_{(k_2)}}`.
    The coverage is guaranteed to be greater or equal to :math:`\beta`.

Notes
-----
Refer to :func:`computeBilateralConfidenceInterval()`. In addition to the confidence interval,
the method returns the supremum of :math:`\beta` for the confidence interval.

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> sample = ot.Gumbel().getSample(60)
>>> ci, coverage = algo.computeBilateralConfidenceIntervalWithCoverage(sample)
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeUnilateralMinimumSampleSize
R"RAW(Evaluate the minimum sample size for the unilateral confidence interval.

The rank :math:`k \geq 0` is specified. The method computes the smallest sample size :math:`n` so that:

.. math::

    \Prob{X_{(k)} \leq x_{\alpha}} \geq \beta

in the case of a lower bound of the quantile of order :math:`\alpha`, and:

.. math::

    \Prob{x_{\alpha} \leq X_{(n-k-1)}} \geq \beta.

in the case of an upper bound.

In other words, the interval :math:`\left] -\infty, X_{(k)} \right]` or :math:`\left[ X_{(k)}, +\infty \right[` is a unilateral confidence
interval for the quantile :math:`x_\alpha` with confidence :math:`\beta`. The statistics values are evaluated from the sample.

See  :ref:`quantile_confidence_estimation`
to get the conditions of the existence of a solution and the solution.

Parameters
----------
k : int, optional, :math:`k \geq 0`
    Rank of the quantile.

    Default value is 0.
lower_bounded : bool, optional
    True indicates the interval is bounded by a lower value.

    False indicates the interval is bounded by an upper value.

    Default value is False.

Returns
-------
size : int
    Minimum sample size :math:`n`.

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> size = algo.computeUnilateralMinimumSampleSize(0)
>>> print(size)
59
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeBilateralMinimumSampleSize
R"RAW(Evaluate the minimum size of the sample for the bilateral min - max confidence interval.

The method determines the smallest sample size :math:`n` so that:

.. math::

    \Prob{X_{(0)} \leq x_{\alpha} \leq X_{(n-1)}} \geq \beta.

In other words, the interval :math:`\left[X_{(0)}, X_{(n-1)} \right]` is a bilateral  confidence
interval for the quantile :math:`x_\alpha` with confidence :math:`\beta`, where
:math:`X_{(0)} = \min(X_1, \dots, X_{n-1})`
and :math:`X_{(n-1)} = \max(X_1, \dots, X_{n-1})`.

The statistics values are evaluated from the sample.

See  :ref:`quantile_confidence_estimation`
to get the conditions of the existence of a solution and the solution.

Returns
-------
size : int
    Minimum sample size :math:`n`.

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> size = algo.computeBilateralMinimumSampleSize()
>>> print(size)
59
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeAsymptoticBilateralRank
R"RAW(Evaluate the asymptotic bilateral rank of a quantile.

This method computes two integers :math:`(k_1, k_2)` such that:

.. math::
    \lim_{n \rightarrow \infty} \Prob{X_{(k_1)} \leq x_\alpha \leq X_{(k_2)}} = \beta, \quad 0 \leq k_1 \leq k_2 \leq n - 1

In other words, the interval :math:`\left[X_{(k_1)}, X_{(k_2)} \right]` is an asymptotic
confidence interval for :math:`x_\alpha` with asymptotic confidence :math:`\beta`.
The asymptotic bilateral ranks :math:`k_1, k_2 \in \llbracket 0, n - 1 \rrbracket` are estimated from:

.. math::

    k_1 & = \left\lfloor n \alpha - \sqrt{n} z_{1-\beta/2} \sqrt{\alpha (1 - \alpha)} \right\rfloor - 1 \\
    k_2 & = \left\lfloor n \alpha + \sqrt{n} z_{1-\beta/2} \sqrt{\alpha (1 - \alpha)} \right\rfloor - 1

with :math:`z_{1-\beta/2}` the standard Gaussian quantile of order :math:`1-\beta/2`, see [delmas2006]_ proposition 12.2.13 page 257.

Parameters
----------
size : int
    Sample size

Returns
-------
ranks : :class:`~openturns.Indices` of size 2
    Pair of lower and upper ranks :math:`(k_1, k_2)` with :math:`0 \leq k_1 \leq k_2 \leq n-1`.

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> k1, k2 = algo.computeAsymptoticBilateralRank(100)
>>> print(k1, k2)
0 8
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeAsymptoticBilateralConfidenceInterval
R"RAW(Evaluate asymptotic bilateral bounds of a quantile.

The asymptotic bounds are given by the order statistics:

.. math::

    [X_{(k_1)}, X_{(k_2)}]

so that:

.. math::

    \lim_{n \rightarrow \infty} \Prob{X_{(k_1)} \leq x_{\alpha} \leq X_{(k_2)}} \geq \beta

where the ranks :math:`0 \leq k_1 \leq k_2 \leq n - 1` are given by :meth:`computeAsymptoticBilateralRank`.

Parameters
----------
sample : 2-d sequence of float
    Sample of the variable :math:`X`

Returns
-------
ci : :class:`~openturns.Interval`
    Quantile confidence interval

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> sample = ot.Gumbel().getSample(60)
>>> ci = algo.computeAsymptoticBilateralConfidenceInterval(sample)
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::setAlpha
"Quantile level accessor.

Parameters
----------
alpha : float
    Quantile level
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::getAlpha
"Quantile level accessor.

Returns
-------
alpha : float
    Quantile level
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::setBeta
"Confidence level accessor.

Parameters
----------
beta : float
    Confidence level
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::getBeta
"Confidence level accessor.

Returns
-------
beta : float
    Confidence level
"
