/**
* @license Apache-2.0
*
* Copyright (c) 2019 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var proxyquire = require( 'proxyquire' );
var randu = require( '@stdlib/random/iter/randu' );
var array2iterator = require( '@stdlib/array/to-iterator' );
var iteratorSymbol = require( '@stdlib/symbol/iterator' );
var noop = require( '@stdlib/utils/noop' );
var iterFilterMap = require( './../lib' );


// FUNCTIONS //

function createIterator( arr ) {
	var len;
	var it;
	var i;

	len = arr.length;
	i = -1;

	it = {};
	it.next = next;

	return it;

	function next() {
		var out;
		i += 1;
		if ( i < len ) {
			out = {};
			out.value = arr[ i ];
			out.done = ( i === len-1 );
			return out;
		}
		return {
			'done': true
		};
	}
}


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof iterFilterMap, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function throws an error if provided an iterator argument which is not an iterator protocol-compliant object', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		5,
		NaN,
		true,
		false,
		null,
		void 0,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			iterFilterMap( value, noop );
		};
	}
});

tape( 'the function throws an error if provided a second argument which is not a function', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		5,
		NaN,
		true,
		false,
		null,
		void 0,
		[],
		{}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			iterFilterMap( randu(), value );
		};
	}
});

tape( 'the function returns an iterator protocol-compliant object', function test( t ) {
	var it;
	var r;
	var i;

	it = iterFilterMap( randu(), fcn );
	t.equal( it.next.length, 0, 'has zero arity' );

	for ( i = 0; i < 100; i++ ) {
		r = it.next();
		t.equal( typeof r.value, 'number', 'returns a number' );
		t.equal( typeof r.done, 'boolean', 'returns a boolean' );
	}
	t.end();

	function fcn( v ) {
		if ( v > 0.5 ) {
			return v * 10.0;
		}
	}
});

tape( 'the function returns an iterator protocol-compliant object (value+done)', function test( t ) {
	var expected;
	var values;
	var actual;
	var it;
	var i;

	values = [ 1, 3, 2, 4 ];
	expected = [
		{
			'value': 30,
			'done': false
		},
		{
			'value': 40,
			'done': true
		}
	];

	it = iterFilterMap( createIterator( values ), fcn );
	t.equal( it.next.length, 0, 'has zero arity' );

	actual = [];
	for ( i = 0; i < expected.length; i++ ) {
		actual.push( it.next() );
	}
	t.deepEqual( actual, expected, 'returns expected values' );
	t.end();

	function fcn( v ) {
		if ( v > 2 ) {
			return v * 10;
		}
	}
});

tape( 'the function returns an iterator protocol-compliant object (value+done)', function test( t ) {
	var expected;
	var values;
	var actual;
	var it;
	var i;

	values = [ 1, 3, 4, 2 ];
	expected = [
		{
			'value': 30,
			'done': false
		},
		{
			'value': 40,
			'done': false
		},
		{
			'done': true
		}
	];

	it = iterFilterMap( createIterator( values ), fcn );
	t.equal( it.next.length, 0, 'has zero arity' );

	actual = [];
	for ( i = 0; i < expected.length; i++ ) {
		actual.push( it.next() );
	}
	t.deepEqual( actual, expected, 'returns expected values' );
	t.end();

	function fcn( v ) {
		if ( v > 2 ) {
			return v * 10;
		}
	}
});

tape( 'the function returns an iterator protocol-compliant object which filters and maps iterated values according to a callback function', function test( t ) {
	var expected;
	var actual;
	var values;
	var it;
	var i;

	values = [ 1, 3, 2, 4, 1 ];
	expected = [
		{
			'value': 30,
			'done': false
		},
		{
			'value': 40,
			'done': false
		},
		{
			'done': true
		}
	];

	it = iterFilterMap( array2iterator( values ), fcn );
	t.equal( it.next.length, 0, 'has zero arity' );

	actual = [];
	for ( i = 0; i < expected.length; i++ ) {
		actual.push( it.next() );
	}
	t.deepEqual( actual, expected, 'returns expected values' );
	t.end();

	function fcn( v ) {
		if ( v > 2 ) {
			return v * 10;
		}
	}
});

tape( 'the function returns an iterator protocol-compliant object which filters  and maps iterated values according to a callback function (none succeed)', function test( t ) {
	var expected;
	var actual;
	var values;
	var it;
	var i;

	values = [ 1, 3, 2, 4, 1 ];
	expected = [
		{
			'done': true
		}
	];

	it = iterFilterMap( array2iterator( values ), fcn );
	t.equal( it.next.length, 0, 'has zero arity' );

	actual = [];
	for ( i = 0; i < expected.length; i++ ) {
		actual.push( it.next() );
	}
	t.deepEqual( actual, expected, 'returns expected values' );
	t.end();

	function fcn( v ) {
		if ( v > 1e6 ) {
			return v * 10;
		}
	}
});

tape( 'the function returns an iterator protocol-compliant object which filters and maps iterated values according to a callback function (all succeed)', function test( t ) {
	var expected;
	var actual;
	var values;
	var it;
	var i;

	values = [ 1, 3, 2, 4, 1 ];
	expected = [
		{
			'value': 10,
			'done': false
		},
		{
			'value': 30,
			'done': false
		},
		{
			'value': 20,
			'done': false
		},
		{
			'value': 40,
			'done': false
		},
		{
			'value': 10,
			'done': false
		},
		{
			'done': true
		}
	];

	it = iterFilterMap( array2iterator( values ), fcn );
	t.equal( it.next.length, 0, 'has zero arity' );

	actual = [];
	for ( i = 0; i < expected.length; i++ ) {
		actual.push( it.next() );
	}
	t.deepEqual( actual, expected, 'returns expected values' );
	t.end();

	function fcn( v ) {
		if ( v > 0 ) {
			return v * 10;
		}
	}
});

tape( 'the function supports specifying the callback function execution context', function test( t ) {
	var expected;
	var actual;
	var values;
	var ctx;
	var it;
	var i;

	ctx = {
		'count': 0
	};
	values = [ 1, 3, 2, 4, 1 ];
	expected = [
		{
			'value': 10,
			'done': false
		},
		{
			'value': 30,
			'done': false
		},
		{
			'value': 20,
			'done': false
		},
		{
			'value': 40,
			'done': false
		},
		{
			'value': 10,
			'done': false
		},
		{
			'done': true
		}
	];

	it = iterFilterMap( array2iterator( values ), fcn, ctx );
	t.equal( it.next.length, 0, 'has zero arity' );

	actual = [];
	for ( i = 0; i < expected.length; i++ ) {
		actual.push( it.next() );
	}
	t.deepEqual( actual, expected, 'returns expected values' );
	t.equal( ctx.count, values.length, 'returns expected value' );
	t.end();

	function fcn( v ) {
		this.count += 1; // eslint-disable-line no-invalid-this
		if ( v > 0 ) {
			return v * 10;
		}
	}
});

tape( 'the returned iterator has a `return` method for closing an iterator (no argument)', function test( t ) {
	var it;
	var r;

	it = iterFilterMap( randu(), fcn );

	r = it.next();
	t.equal( typeof r.value, 'number', 'returns a number' );
	t.equal( r.done, false, 'returns expected value' );

	r = it.next();
	t.equal( typeof r.value, 'number', 'returns a number' );
	t.equal( r.done, false, 'returns expected value' );

	r = it.return();
	t.equal( r.value, void 0, 'returns expected value' );
	t.equal( r.done, true, 'returns expected value' );

	r = it.next();
	t.equal( r.value, void 0, 'returns expected value' );
	t.equal( r.done, true, 'returns expected value' );

	t.end();

	function fcn( v ) {
		if ( v > 0.5 ) {
			return v * 10.0;
		}
	}
});

tape( 'the returned iterator has a `return` method for closing an iterator (argument)', function test( t ) {
	var it;
	var r;

	it = iterFilterMap( randu(), fcn );

	r = it.next();
	t.equal( typeof r.value, 'number', 'returns a number' );
	t.equal( r.done, false, 'returns expected value' );

	r = it.next();
	t.equal( typeof r.value, 'number', 'returns a number' );
	t.equal( r.done, false, 'returns expected value' );

	r = it.return( 'finished' );
	t.equal( r.value, 'finished', 'returns expected value' );
	t.equal( r.done, true, 'returns expected value' );

	r = it.next();
	t.equal( r.value, void 0, 'returns expected value' );
	t.equal( r.done, true, 'returns expected value' );

	t.end();

	function fcn( v ) {
		if ( v > 0.5 ) {
			return v * 10.0;
		}
	}
});

tape( 'if an environment supports `Symbol.iterator` and the provided iterator is iterable, the returned iterator is iterable', function test( t ) {
	var iterFilterMap;
	var opts;
	var rand;
	var it1;
	var it2;
	var i;

	iterFilterMap = proxyquire( './../lib/main.js', {
		'@stdlib/symbol/iterator': '__ITERATOR_SYMBOL__'
	});

	opts = {
		'seed': 12345
	};
	rand = randu( opts );
	rand[ '__ITERATOR_SYMBOL__' ] = factory;

	it1 = iterFilterMap( rand, fcn );
	t.equal( typeof it1[ '__ITERATOR_SYMBOL__' ], 'function', 'has method' );
	t.equal( it1[ '__ITERATOR_SYMBOL__' ].length, 0, 'has zero arity' );

	it2 = it1[ '__ITERATOR_SYMBOL__' ]();
	t.equal( typeof it2, 'object', 'returns an object' );
	t.equal( typeof it2.next, 'function', 'has method' );
	t.equal( typeof it2.return, 'function', 'has method' );

	for ( i = 0; i < 100; i++ ) {
		t.equal( it2.next().value, it1.next().value, 'returns expected value' );
	}
	t.end();

	function factory() {
		return randu( opts );
	}

	function fcn( v ) {
		if ( v > 0.5 ) {
			return v * 10.0;
		}
	}
});

tape( 'if an environment does not support `Symbol.iterator`, the returned iterator is not "iterable"', function test( t ) {
	var iterFilterMap;
	var it;

	iterFilterMap = proxyquire( './../lib/main.js', {
		'@stdlib/symbol/iterator': false
	});

	it = iterFilterMap( randu(), fcn );
	t.equal( it[ iteratorSymbol ], void 0, 'does not have property' );

	t.end();

	function fcn( v ) {
		if ( v > 0.5 ) {
			return v * 10.0;
		}
	}
});

tape( 'if a provided iterator is not iterable, the returned iterator is not iterable', function test( t ) {
	var iterFilterMap;
	var rand;
	var it;

	iterFilterMap = proxyquire( './../lib/main.js', {
		'@stdlib/symbol/iterator': '__ITERATOR_SYMBOL__'
	});

	rand = randu();
	rand[ '__ITERATOR_SYMBOL__' ] = null;

	it = iterFilterMap( rand, fcn );
	t.equal( it[ iteratorSymbol ], void 0, 'does not have property' );

	t.end();

	function fcn( v ) {
		if ( v > 0.5 ) {
			return v * 10.0;
		}
	}
});
