/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */

// Grilo
#include <grilo.h>

// Google Tests
#include <gtest/gtest.h>

// Boost C++
#include <boost/filesystem.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/variant.hpp>

// Standard Library
#include <map>
#include <string>
#include <utility>
#include <vector>
#include <functional>

using namespace std::placeholders;

// Media Scanner Library
#include "mediascanner/glibutils.h"
#include "mediascanner/locale.h"
#include "mediascanner/mediaroot.h"
#include "mediascanner/propertyschema.h"
#include "mediascanner/utilities.h"
#include "mediascanner/writablemediaindex.h"

// Media Scanner Plugin for Grilo
#include "grlmediascanner/mediasource.h"

// Test Suite
#include "testlib/environments.h"
#include "testlib/loggingsink.h"
#include "testlib/testutils.h"

////////////////////////////////////////////////////////////////////////////////

namespace mediascanner {
namespace griloplugintest {

// Boost C++
using boost::filesystem::current_path;
using boost::locale::format;
using boost::locale::wformat;
using boost::posix_time::time_from_string;

////////////////////////////////////////////////////////////////////////////////

static const FileSystemPath kDataPath = current_path() / "run/griloplugintest";
static const FileSystemPath kMediaIndexPath = kDataPath / "index";

static const char kGoodAudioUri[] = "urn:x-test:good-audio";
static const char kGoodAudioTitle[] = "Good Test Audio";
static const char kGoodAudioType[] = "audio/x-fancy";
static const char kGoodAudioLastModified[] = "2012-08-31 12:30:00";

static const char kOtherAudioUri[] = "urn:x-test:other-audio";
static const char kOtherAudioTitle[] = "Other Good Test Audio";
static const char kOtherAudioType[] = "audio/x-awesome";
static const char kOtherAudioLastModified[] = "2012-09-04 13:30:00";

static const char kGoodImageUri[] = "urn:x-test:good-image";
static const char kGoodImageTitle[] = "Good Test Image";
static const char kGoodImageType[] = "image/x-fancy";

static const char kOtherImageUri[] = "urn:x-test:other-image";
static const char kOtherImageTitle[] = "Other Good Test Image";
static const char kOtherImageType[] = "image/x-awesome";

static const char kGoodVideoUri[] = "urn:x-test:good-video";
static const char kGoodVideoTitle[] = "Good Test Video";
static const char kGoodVideoType[] = "video/x-fancy";

static const char kOtherVideoUri[] = "urn:x-test:other-video";
static const char kOtherVideoTitle[] = "Other Good Test Video";
static const char kOtherVideoType[] = "video/x-awesome";

static const char kBadMediaUri[] = "urn:x-test:bad";

////////////////////////////////////////////////////////////////////////////////

typedef std::map<std::string, MediaInfo> MediaInfoMap;

static void merge_media(const std::string &url,
                        const MediaInfo &media,
                        MediaInfoMap *media_map) {
    media_map->insert(std::make_pair(full_url(url), media));
}

static void merge_media(const MediaInfoMap &input,
                        std::string uri_filter,
                        MediaInfoMap *output) {
    if (not uri_filter.empty())
        uri_filter = full_url(uri_filter);

    for (const MediaInfoMap::value_type &p: input) {
        if (uri_filter.empty() || p.first == uri_filter)
            output->insert(p);
    }
}

static MediaInfoMap make_audio_media() {
    MediaInfoMap media_map;

    {
        MediaInfo media;
        media.add_single(schema::kTitle.
                         bind_value(ToUnicode(kOtherAudioTitle)));
        media.add_single(schema::kMimeType.
                         bind_value(ToUnicode(kOtherAudioType)));
        media.add_single(schema::kDuration.bind_value(402UL));
        media.add_single(schema::kSeekable.bind_value(false));
        media.add_single(schema::kPlayCount.bind_value(42U));
        media.add_single(schema::kLastModified.
                         bind_value(time_from_string(kOtherAudioLastModified)));
        merge_media(kOtherAudioUri, media, &media_map);
    }

    {
        MediaInfo media;
        media.add_single(schema::kTitle.
                         bind_value(ToUnicode(kGoodAudioTitle)));
        media.add_single(schema::kMimeType.
                         bind_value(ToUnicode(kGoodAudioType)));
        media.add_single(schema::kDuration.bind_value(201UL));
        media.add_single(schema::kSeekable.bind_value(true));
        media.add_single(schema::kPlayCount.bind_value(13U));
        media.add_single(schema::kLastModified.
                         bind_value(time_from_string(kGoodAudioLastModified)));
        merge_media(kGoodAudioUri, media, &media_map);
    }

    return media_map;
}

static MediaInfoMap make_image_media() {
    MediaInfoMap media_map;

    {
        MediaInfo media;
        media.add_single(schema::kTitle.
                         bind_value(ToUnicode(kGoodImageTitle)));
        media.add_single(schema::kMimeType.
                         bind_value(ToUnicode(kGoodImageType)));
        merge_media(kGoodImageUri, media, &media_map);
    }

    {
        MediaInfo media;
        media.add_single(schema::kTitle.
                         bind_value(ToUnicode(kOtherImageTitle)));
        media.add_single(schema::kMimeType.
                         bind_value(ToUnicode(kOtherImageType)));
        merge_media(kOtherImageUri, media, &media_map);
    }

    return media_map;
}

static MediaInfoMap make_video_media() {
    MediaInfoMap media_map;

    {
        MediaInfo media;
        media.add_single(schema::kTitle.
                         bind_value(ToUnicode(kGoodVideoTitle)));
        media.add_single(schema::kMimeType.
                         bind_value(ToUnicode(kGoodVideoType)));
        merge_media(kGoodVideoUri, media, &media_map);
    }

    {
        MediaInfo media;
        media.add_single(schema::kTitle.
                         bind_value(ToUnicode(kOtherVideoTitle)));
        media.add_single(schema::kMimeType.
                         bind_value(ToUnicode(kOtherVideoType)));
        merge_media(kOtherVideoUri, media, &media_map);
    }

    return media_map;
}

static MediaInfoMap make_known_media
                            (const std::string &uri_filter = std::string()) {
    MediaInfoMap media;

    merge_media(make_audio_media(), uri_filter, &media);
    merge_media(make_image_media(), uri_filter, &media);
    merge_media(make_video_media(), uri_filter, &media);

    return media;
}

////////////////////////////////////////////////////////////////////////////////

static Wrapper<GrlSource> GetMediaScannerSource() {
    const Wrapper<GrlRegistry> registry = wrap(grl_registry_get_default());
    return wrap(grl_registry_lookup_source(registry.get(),
                                           GRL_MEDIA_SCANNER_PLUGIN_ID));
}

static bool has_key(GList *const key_list, GrlKeyID key_id) {
    return g_list_find(key_list, GRLKEYID_TO_POINTER(key_id)) != nullptr;
}

////////////////////////////////////////////////////////////////////////////////

TEST(LoadPlugin, Test) {
    ASSERT_TRUE(GRL_IS_SOURCE(GetMediaScannerSource().get()));
}

////////////////////////////////////////////////////////////////////////////////

TEST(TestMediaFromUri, Test) {
    const Wrapper<GrlSource> source = GetMediaScannerSource();

    EXPECT_TRUE(grl_source_test_media_from_uri(source.get(),
                                               full_url(kGoodVideoUri).
                                               c_str()));
    EXPECT_FALSE(grl_source_test_media_from_uri(source.get(),
                                                full_url(kBadMediaUri).
                                                c_str()));
}

////////////////////////////////////////////////////////////////////////////////

TEST(MediaFromUri, GoodMedia) {
    const Wrapper<GrlOperationOptions> options =
            take(grl_operation_options_new(nullptr));

    Wrapper<GList> keys =
            take(grl_metadata_key_list_new(GRL_METADATA_KEY_TITLE,
                                           GRL_METADATA_KEY_MIME,
                                           GRL_METADATA_KEY_INVALID));

    const Wrapper<GrlSource> source = GetMediaScannerSource();
    Wrapper<GError> error;

    const Wrapper<GrlMedia> media =
            take(grl_source_get_media_from_uri_sync
                        (source.get(), full_url(kGoodVideoUri).c_str(),
                         keys, options.get(), error.out_param()));

    EXPECT_FALSE(error) << to_string(error);
    ASSERT_NE(nullptr, media.get());

    EXPECT_EQ(std::string(kGoodVideoTitle),
              safe_string(grl_data_get_string(media.get<GrlData>(),
                                              GRL_METADATA_KEY_TITLE)));

    EXPECT_EQ(std::string(kGoodVideoType),
              safe_string(grl_data_get_string(media.get<GrlData>(),
                                              GRL_METADATA_KEY_MIME)));
}

TEST(MediaFromUri, BadMedia) {
    const Wrapper<GrlOperationOptions> options =
            take(grl_operation_options_new(nullptr));

    Wrapper<GList> keys =
            take(grl_metadata_key_list_new(GRL_METADATA_KEY_TITLE,
                                           GRL_METADATA_KEY_MIME,
                                           GRL_METADATA_KEY_INVALID));

    const Wrapper<GrlSource> source = GetMediaScannerSource();
    Wrapper<GError> error;

    const Wrapper<GrlMedia> media =
            take(grl_source_get_media_from_uri_sync
                        (source.get(), full_url(kBadMediaUri).c_str(),
                         keys, options.get(), error.out_param()));

    EXPECT_FALSE(media);

    ASSERT_TRUE(error);
    EXPECT_EQ(error->domain, GRL_CORE_ERROR);
    EXPECT_EQ(error->code, GRL_CORE_ERROR_MEDIA_NOT_FOUND);
    EXPECT_FALSE(safe_string(error->message).empty());
}

////////////////////////////////////////////////////////////////////////////////

class FilterParams {
public:
    typedef std::function
        <Wrapper<GrlOperationOptions>(Wrapper<GrlCaps>)> MakeOptions;

    enum CapsMode { IgnoreCaps, UseCaps };

    FilterParams() {
    }

    FilterParams(const std::string &label,
                 const MakeOptions &make_options,
                 const MediaInfoMap &expected_media,
                 CapsMode caps_mode)
        : label_(label)
        , make_options_(make_options)
        , expected_media_(expected_media)
        , caps_mode_(caps_mode) {
    }

    const std::string &label() const {
        return label_;
    }

    Wrapper<GrlOperationOptions> make_options(Wrapper<GrlSource> source,
                                              GrlSupportedOps ops) const {
        Wrapper<GrlCaps> caps;

        if (caps_mode_ == UseCaps)
            caps = wrap(grl_source_get_caps(source.get(), ops));

        return make_options_(caps);
    }

    const MediaInfoMap &expected_media() const {
        return expected_media_;
    }

    CapsMode caps_mode() const {
        return caps_mode_;
    }

private:
    std::string label_;
    MakeOptions make_options_;
    MediaInfoMap expected_media_;
    CapsMode caps_mode_;
};

static void PrintTo(const FilterParams &p, std::ostream *os) {
    *os << p.label() << (p.caps_mode() ? ", use caps" : ", ignore caps");
}

static Wrapper<GrlOperationOptions> DefaultOptions(Wrapper<GrlCaps> caps) {
    return take(grl_operation_options_new(caps.get()));
}

static std::vector<FilterParams> DefaultFilters() {
    return std::vector<FilterParams>()
            << FilterParams("any", DefaultOptions, make_known_media(),
                            FilterParams::IgnoreCaps)
            << FilterParams("any", DefaultOptions, make_known_media(),
                            FilterParams::UseCaps);
}

static Wrapper<GrlOperationOptions> type_filter_options(Wrapper<GrlCaps> caps,
                                                        GrlTypeFilter filter) {
    Wrapper<GrlOperationOptions> options =
            take(grl_operation_options_new(caps.get()));

    grl_operation_options_set_type_filter(options.get(), filter);

    return options;
}

static void add_type_filter(const std::string &label,
                            GrlTypeFilter filter,
                            const MediaInfoMap &expected_media,
                            FilterParams::CapsMode caps_mode,
                            std::vector<FilterParams> *params) {
    const FilterParams::MakeOptions options =
            std::bind(type_filter_options, _1, filter);
    params->push_back(FilterParams(label, options, expected_media, caps_mode));
}

static std::vector<FilterParams> TypeFilters() {
    std::vector<FilterParams> params;

    for (int i = 0; i < 2; ++i) {
        const FilterParams::CapsMode caps_mode =
                static_cast<FilterParams::CapsMode>(i);

        add_type_filter("all", GRL_TYPE_FILTER_ALL,
                        make_known_media(), caps_mode, &params);
        add_type_filter("audios", GRL_TYPE_FILTER_AUDIO,
                        make_audio_media(), caps_mode, &params);
        add_type_filter("images", GRL_TYPE_FILTER_IMAGE,
                        make_image_media(), caps_mode, &params);
        add_type_filter("videos", GRL_TYPE_FILTER_VIDEO,
                        make_video_media(), caps_mode, &params);
    }

    return params;
}

static Wrapper<GrlOperationOptions> key_filter_options
        (Wrapper<GrlCaps> caps, const Property::BoundValue &filter) {
    Wrapper<GrlOperationOptions> options =
            take(grl_operation_options_new(caps.get()));

    const bool key_value_set = grl_operation_options_set_key_filter_value
            (options.get(), filter.first.metadata_key().id(),
             take(filter.first.MakeGriloValue(filter.second)).get());

    EXPECT_TRUE(key_value_set)
            << filter.first.field_name();
    EXPECT_NE(nullptr,
              grl_operation_options_get_key_filter_list(options.get()))
            << filter.first.field_name();

    return options;
}

static void add_key_filter(const std::string &label,
                           const Property::BoundValue &filter,
                           const MediaInfoMap &expected_media,
                           FilterParams::CapsMode caps_mode,
                           std::vector<FilterParams> *params) {
    const FilterParams::MakeOptions options =
            std::bind(key_filter_options, _1, filter);
    params->push_back(FilterParams(label, options, expected_media, caps_mode));
}

static std::vector<FilterParams> KeyFilters() {
    const DateTime dt = time_from_string(kGoodAudioLastModified);

    std::vector<FilterParams> params;

    for (int i = 0; i < 2; ++i) {
        const FilterParams::CapsMode caps_mode =
                static_cast<FilterParams::CapsMode>(i);

        add_key_filter("title",
                       schema::kTitle.bind_value(ToUnicode(kGoodAudioTitle)),
                       make_known_media(kGoodAudioUri),
                       caps_mode, &params);
        add_key_filter("title/short",
                       schema::kTitle.bind_value(L"Good"),
                       MediaInfoMap(), caps_mode, &params);
        add_key_filter("title/long",
                       schema::kTitle.bind_value(ToUnicode(kGoodAudioTitle) +
                                                 L" Extra"),
                       MediaInfoMap(), caps_mode, &params);

        add_key_filter("mime-type",
                       schema::kMimeType.bind_value(ToUnicode(kOtherAudioType)),
                       make_known_media(kOtherAudioUri),
                       caps_mode, &params);
        add_key_filter("duration",
                       schema::kDuration.bind_value(201),
                       make_known_media(kGoodAudioUri),
                       caps_mode, &params);
        add_key_filter("seekable",
                       schema::kSeekable.bind_value(true),
                       make_known_media(kGoodAudioUri),
                       caps_mode, &params);
        add_key_filter("seekable",
                       schema::kSeekable.bind_value(false),
                       make_known_media(kOtherAudioUri),
                       caps_mode, &params);
        add_key_filter("last-modified",
                       schema::kLastModified.bind_value(dt),
                       make_known_media(kGoodAudioUri),
                       caps_mode, &params);
    }

    // FIXME(M4): also test lists of key range filters
    return params;
}

static Wrapper<GrlOperationOptions> key_range_filter_options
        (Wrapper<GrlCaps> caps, const Property::BoundValue &min_value,
         const Property::BoundValue &max_value) {
    Wrapper<GrlOperationOptions> options =
            take(grl_operation_options_new(caps.get()));

    grl_operation_options_set_key_range_filter_value
            (options.get(), min_value.first.metadata_key().id(),
             take(min_value.first.MakeGriloValue(min_value.second)).get(),
             take(min_value.first.MakeGriloValue(max_value.second)).get());

    return options;
}

static void add_key_range_filter(const std::string &label,
                                 const Property::BoundValue &min_value,
                                 const Property::BoundValue &med_value,
                                 const Property::BoundValue &max_value,
                                 const MediaInfoMap &expected_media_lower,
                                 const MediaInfoMap &expected_media_upper,
                                 const MediaInfoMap &expected_media_both,
                                 FilterParams::CapsMode caps_mode,
                                 std::vector<FilterParams> *params) {
    const FilterParams::MakeOptions options_lower =
            std::bind(key_range_filter_options, _1, min_value, med_value);
    const FilterParams::MakeOptions options_upper =
            std::bind(key_range_filter_options, _1, med_value, max_value);
    const FilterParams::MakeOptions options_both =
            std::bind(key_range_filter_options, _1, min_value, max_value);

    params->push_back(FilterParams(label + "-lower", options_lower,
                                   expected_media_lower, caps_mode));
    params->push_back(FilterParams(label + "-upper", options_upper,
                                   expected_media_upper, caps_mode));
    params->push_back(FilterParams(label + "-both", options_both,
                                   expected_media_both, caps_mode));
}

static std::vector<FilterParams> KeyRangeFilters() {
    const DateTime dt_min = time_from_string("2012-08-01 00:00:00");
    const DateTime dt_med = time_from_string("2012-09-01 00:00:00");
    const DateTime dt_max = time_from_string("2012-10-01 00:00:00");

    std::vector<FilterParams> params;

    for (int i = 0; i < 2; ++i) {
        const FilterParams::CapsMode caps_mode =
                static_cast<FilterParams::CapsMode>(i);

        add_key_range_filter("duration",
                             schema::kDuration.bind_value(100),
                             schema::kDuration.bind_value(300),
                             schema::kDuration.bind_value(500),
                             make_known_media(kGoodAudioUri),
                             make_known_media(kOtherAudioUri),
                             make_audio_media(),
                             caps_mode, &params);

        add_key_range_filter("last-modified",
                             schema::kLastModified.bind_value(dt_min),
                             schema::kLastModified.bind_value(dt_med),
                             schema::kLastModified.bind_value(dt_max),
                             make_known_media(kGoodAudioUri),
                             make_known_media(kOtherAudioUri),
                             make_audio_media(),
                             caps_mode, &params);
    }

    // FIXME(M2): also test lists of key range filters
    return params;
}

static Wrapper<GrlOperationOptions> result_count_options(Wrapper<GrlCaps> caps,
                                                         uint skip, int count) {
    Wrapper<GrlOperationOptions> options =
            take(grl_operation_options_new(caps.get()));

    grl_operation_options_set_skip(options.get(), skip);
    grl_operation_options_set_count(options.get(), count);

    return options;
}

static std::vector<FilterParams> ResultCountFilters() {
    const FilterParams::MakeOptions options_limit =
            std::bind(result_count_options, _1, 0, 100);

    return std::vector<FilterParams>()
            << FilterParams("limit-100", options_limit, make_known_media(),
                            FilterParams::IgnoreCaps)
            << FilterParams("limit-100", options_limit, make_known_media(),
                            FilterParams::UseCaps);
}

////////////////////////////////////////////////////////////////////////////////

class Browse : public testing::TestWithParam<FilterParams> { };

INSTANTIATE_TEST_CASE_P(DefaultFilters, Browse,
                        testing::ValuesIn(DefaultFilters()));
INSTANTIATE_TEST_CASE_P(TypeFilters, Browse,
                        testing::ValuesIn(TypeFilters()));
INSTANTIATE_TEST_CASE_P(KeyFilters, Browse,
                        testing::ValuesIn(KeyFilters()));
INSTANTIATE_TEST_CASE_P(KeyRangeFilters, Browse,
                        testing::ValuesIn(KeyRangeFilters()));

TEST_P(Browse, BrowseAndResolve) {
    const Wrapper<GrlSource> source = GetMediaScannerSource();
    Wrapper<GError> error;

    const Wrapper<GList> base_keys =
            take(grl_metadata_key_list_new(GRL_METADATA_KEY_MIME,
                                           GRL_METADATA_KEY_URL,
                                           GRL_METADATA_KEY_INVALID));

    const Wrapper<GList> extra_keys =
            take(grl_metadata_key_list_new(GRL_METADATA_KEY_TITLE,
                                           GRL_METADATA_KEY_INVALID));

    const Wrapper<GrlOperationOptions> browse_options =
            GetParam().make_options(source, GRL_OP_BROWSE);

    const ListWrapper<GrlMedia> media_list =
            take(grl_source_browse_sync(source.get(),
                                        nullptr, base_keys.get(),
                                        browse_options.get(),
                                        error.out_param()));

    ASSERT_FALSE(error) << error->message;

    MediaInfoMap expected_media = GetParam().expected_media();

    const Wrapper<GrlOperationOptions> resolve_options =
            take(grl_operation_options_new(nullptr));

    for (GList *l = media_list.get(); l; l = l->next) {
        GrlMedia *const media = static_cast<GrlMedia *>(l->data);

        const std::string media_url = safe_string(grl_media_get_url(media));

        ASSERT_FALSE(media_url.empty());

        const MediaInfoMap::iterator it = expected_media.find(media_url);

        ASSERT_NE(expected_media.end(), it)
                << "Unexpected media: " << media_url;

        const std::wstring mime_type = it->second.first(schema::kMimeType);

        EXPECT_FALSE(mime_type.empty());
        ASSERT_EQ(mime_type, safe_wstring(grl_media_get_mime(media)));

        grl_source_resolve_sync(source.get(), media, extra_keys,
                                resolve_options.get(), error.out_param());

        ASSERT_FALSE(error) << error->message;

        const std::wstring title = it->second.first(schema::kTitle);

        EXPECT_FALSE(title.empty());
        ASSERT_EQ(title, safe_wstring(grl_media_get_title(media)));

        expected_media.erase(it);
    }

    ASSERT_EQ(MediaInfoMap(), expected_media);
}

TEST(Resolve, BadResolve) {
    const Wrapper<GrlMedia> media = take(grl_media_new());
    grl_media_set_url(media.get(), kBadMediaUri);

    const Wrapper<GList> keys =
            take(grl_metadata_key_list_new(GRL_METADATA_KEY_TITLE,
                                           GRL_METADATA_KEY_INVALID));

    const Wrapper<GrlOperationOptions> options =
            take(grl_operation_options_new(nullptr));

    const Wrapper<GrlSource> source = GetMediaScannerSource();
    Wrapper<GError> error;

    grl_source_resolve_sync(source.get(), media.get(), keys.get(),
                            options.get(), error.out_param());

    ASSERT_TRUE(error);
    EXPECT_EQ(error->domain, GRL_CORE_ERROR);
    EXPECT_EQ(error->code, GRL_CORE_ERROR_MEDIA_NOT_FOUND);
    EXPECT_FALSE(safe_string(error->message).empty());
}

////////////////////////////////////////////////////////////////////////////////

class QueryOrSearch : public testing::TestWithParam
        <std::tr1::tuple <FilterParams, std::string> > {
protected:
    void RunTest(GrlSupportedOps operations) {
        const FilterParams &filter = std::tr1::get<0>(GetParam());
        const Wrapper<GrlSource> source = GetMediaScannerSource();

        const Wrapper<GrlOperationOptions> options =
                filter.make_options(source, operations);
        const Wrapper<GList> keys =
                take(grl_metadata_key_list_new(GRL_METADATA_KEY_URL,
                                               GRL_METADATA_KEY_INVALID));

        Wrapper<GError> error;

        const ListWrapper<GrlMedia> media_list =
                take(MakeResultList(source, options, keys, &error));

        EXPECT_FALSE(error) << error->message;

        MediaInfoMap expected_media = filter.expected_media();

        for (GList *l = media_list.get(); l; l = l->next) {
            GrlMedia *const media = static_cast<GrlMedia *>(l->data);
            const std::string media_url = safe_string(grl_media_get_url(media));

            ASSERT_FALSE(media_url.empty());
            const MediaInfoMap::iterator it = expected_media.find(media_url);

            ASSERT_NE(expected_media.end(), it)
                    << "Unexpected media: " << media_url;

            expected_media.erase(it);
        }

        ASSERT_EQ(MediaInfoMap(), expected_media);
    }

    virtual GList *MakeResultList(const Wrapper<GrlSource> source,
                                  const Wrapper<GrlOperationOptions> options,
                                  const Wrapper<GList> keys,
                                  Wrapper<GError> *error) = 0;
};

////////////////////////////////////////////////////////////////////////////////

class Query : public QueryOrSearch {
protected:
    GList *MakeResultList(const Wrapper<GrlSource> source,
                          const Wrapper<GrlOperationOptions> options,
                          const Wrapper<GList> keys,
                          Wrapper<GError> *error) {
        return grl_source_query_sync(source.get(),
                                     std::tr1::get<1>(GetParam()).c_str(),
                                     keys, options.get(), error->out_param());
    }
};

static std::vector<std::string> QueryStrings() {
    return std::vector<std::string>()
            << std::string("GOOD")
            << std::string("Good")
            << std::string("good")
            << std::string("title:Good")
            << std::string("title:go*")
            << std::string("(title:good artist:bad)")
            << std::string("\"good test\"");
}

INSTANTIATE_TEST_CASE_P(DefaultFilters, Query,
                        testing::Combine(testing::ValuesIn(DefaultFilters()),
                                         testing::ValuesIn(QueryStrings())));
INSTANTIATE_TEST_CASE_P(TypeFilters, Query,
                        testing::Combine(testing::ValuesIn(TypeFilters()),
                                         testing::ValuesIn(QueryStrings())));
INSTANTIATE_TEST_CASE_P(KeyFilters, Query,
                        testing::Combine(testing::ValuesIn(KeyFilters()),
                                         testing::ValuesIn(QueryStrings())));
INSTANTIATE_TEST_CASE_P(KeyRangeFilters, Query,
                        testing::Combine(testing::ValuesIn(KeyRangeFilters()),
                                         testing::ValuesIn(QueryStrings())));
INSTANTIATE_TEST_CASE_P(ResultCountFilters, Query,
                        testing::Combine(testing::ValuesIn(ResultCountFilters()),
                                         testing::ValuesIn(QueryStrings())));

TEST_P(Query, GoodQueries) {
    RunTest(GRL_OP_BROWSE);
}

TEST(Query, BadQueries) {
    const Wrapper<GrlSource> source = GetMediaScannerSource();

    GList *const keys = grl_metadata_key_list_new(GRL_METADATA_KEY_URL,
                                                  GRL_METADATA_KEY_INVALID);

    const Wrapper<GrlOperationOptions> options =
            take(grl_operation_options_new(nullptr));

    Wrapper<GError> error;
    GList *media_list;

    media_list = grl_source_query_sync(source.get(), "title:Bad",
                                       keys, options.get(),
                                       error.out_param());

    EXPECT_EQ(nullptr, media_list);
    EXPECT_FALSE(error) << error->message;

    media_list = grl_source_query_sync(source.get(), "*bad-query",
                                       keys, options.get(),
                                       error.out_param());

    EXPECT_EQ(nullptr, media_list);
    EXPECT_FALSE(safe_string(error->message).empty());
}

////////////////////////////////////////////////////////////////////////////////

class WildcardSearch : public QueryOrSearch {
protected:
    GList *MakeResultList(const Wrapper<GrlSource> source,
                          const Wrapper<GrlOperationOptions> options,
                          const Wrapper<GList> keys,
                          Wrapper<GError> *error) {
        g_object_set(source.get<GObject>(),
                     "search-method", GRL_MEDIA_SCANNER_SEARCH_SUBSTRING, NULL);

        return grl_source_search_sync(source.get(),
                                      std::tr1::get<1>(GetParam()).c_str(),
                                      keys, options.get(), error->out_param());
    }
};

static std::vector<std::string> WildcardSearchStrings() {
    return std::vector<std::string>()
            << std::string("GOOD") << std::string("Good")
            << std::string("good") << std::string("good test")
            << std::string("go") << std::string("oo")
            << std::string("ood");
}

INSTANTIATE_TEST_CASE_P
    (DefaultFilters, WildcardSearch,
     testing::Combine(testing::ValuesIn(DefaultFilters()),
                      testing::ValuesIn(WildcardSearchStrings())));
INSTANTIATE_TEST_CASE_P
    (TypeFilters, WildcardSearch,
     testing::Combine(testing::ValuesIn(TypeFilters()),
                      testing::ValuesIn(WildcardSearchStrings())));
INSTANTIATE_TEST_CASE_P
    (KeyFilters, WildcardSearch,
     testing::Combine(testing::ValuesIn(KeyFilters()),
                      testing::ValuesIn(WildcardSearchStrings())));
INSTANTIATE_TEST_CASE_P
    (KeyRangeFilters, WildcardSearch,
     testing::Combine(testing::ValuesIn(KeyRangeFilters()),
                      testing::ValuesIn(WildcardSearchStrings())));
INSTANTIATE_TEST_CASE_P
    (ResultCountFilters, WildcardSearch,
     testing::Combine(testing::ValuesIn(ResultCountFilters()),
                      testing::ValuesIn(WildcardSearchStrings())));

TEST_P(WildcardSearch, GoodSearches) {
    RunTest(GRL_OP_SEARCH);
}

////////////////////////////////////////////////////////////////////////////////

class FullTextSearch : public QueryOrSearch {
    GList *MakeResultList(const Wrapper<GrlSource> source,
                          const Wrapper<GrlOperationOptions> options,
                          const Wrapper<GList> keys,
                          Wrapper<GError> *error) {
        g_object_set(source.get<GObject>(),
                     "search-method", GRL_MEDIA_SCANNER_SEARCH_FULL_TEXT, NULL);

        return grl_source_search_sync(source.get(),
                                      std::tr1::get<1>(GetParam()).c_str(),
                                      keys, options.get(), error->out_param());
    }
};

static std::vector<std::string> FullTextSearchStrings() {
    // FIXME(M3): Add terms that verify stemming
    return std::vector<std::string>()
            << std::string("GOOD") << std::string("Good")
            << std::string("good") << std::string("good test");
}

INSTANTIATE_TEST_CASE_P
    (DefaultFilters, FullTextSearch,
     testing::Combine(testing::ValuesIn(DefaultFilters()),
                      testing::ValuesIn(FullTextSearchStrings())));
INSTANTIATE_TEST_CASE_P
    (TypeFilters, FullTextSearch,
     testing::Combine(testing::ValuesIn(TypeFilters()),
                      testing::ValuesIn(FullTextSearchStrings())));
INSTANTIATE_TEST_CASE_P
    (KeyFilters, FullTextSearch,
     testing::Combine(testing::ValuesIn(KeyFilters()),
                      testing::ValuesIn(FullTextSearchStrings())));
INSTANTIATE_TEST_CASE_P
    (KeyRangeFilters, FullTextSearch,
     testing::Combine(testing::ValuesIn(KeyRangeFilters()),
                      testing::ValuesIn(FullTextSearchStrings())));
INSTANTIATE_TEST_CASE_P
    (ResultCountFilters, FullTextSearch,
     testing::Combine(testing::ValuesIn(ResultCountFilters()),
                      testing::ValuesIn(FullTextSearchStrings())));

TEST_P(FullTextSearch, GoodSearches) {
    RunTest(GRL_OP_SEARCH);
}

////////////////////////////////////////////////////////////////////////////////

TEST(Search, BadSearches) {
    const Wrapper<GrlSource> source = GetMediaScannerSource();
    Wrapper<GError> error;

    GList *const keys = grl_metadata_key_list_new(GRL_METADATA_KEY_URL,
                                                  GRL_METADATA_KEY_INVALID);

    const Wrapper<GrlOperationOptions> options =
            take(grl_operation_options_new(nullptr));

    GList *media_list = grl_source_search_sync(source.get(), "bad",
                                               keys, options.get(),
                                               error.out_param());

    EXPECT_EQ(nullptr, media_list);
    EXPECT_FALSE(error) << error->message;
}

////////////////////////////////////////////////////////////////////////////////

class SimpleCaps : public testing::TestWithParam<GrlSupportedOps> { };
class FilterableCaps : public testing::TestWithParam<GrlSupportedOps> { };

INSTANTIATE_TEST_CASE_P
    (, SimpleCaps,
     testing::Values(GRL_OP_RESOLVE,
                     GRL_OP_STORE,
                     GRL_OP_STORE_PARENT,
                     GRL_OP_STORE_METADATA,
                     GRL_OP_REMOVE,
                     GRL_OP_MEDIA_FROM_URI,
                     GRL_OP_NOTIFY_CHANGE));

INSTANTIATE_TEST_CASE_P
    (, FilterableCaps,
     testing::Values(GRL_OP_BROWSE,
                     GRL_OP_SEARCH,
                     GRL_OP_QUERY,
                     GRL_OP_BROWSE | GRL_OP_SEARCH,
                     GRL_OP_BROWSE | GRL_OP_QUERY,
                     GRL_OP_SEARCH | GRL_OP_QUERY,
                     GRL_OP_BROWSE | GRL_OP_SEARCH | GRL_OP_QUERY));

TEST_P(SimpleCaps, Test) {
    const GrlSupportedOps ops = GetParam();
    const Wrapper<GrlSource> source = GetMediaScannerSource();
    const Wrapper<GrlCaps> caps = wrap(grl_source_get_caps(source.get(), ops));

    ASSERT_NE(nullptr, caps);

    const GrlTypeFilter filter = grl_caps_get_type_filter(caps.get());
    EXPECT_TRUE(filter == GRL_TYPE_FILTER_NONE ||
                filter == GRL_TYPE_FILTER_ALL);
    EXPECT_EQ(nullptr, grl_caps_get_key_range_filter(caps.get()));
    EXPECT_EQ(nullptr, grl_caps_get_key_filter(caps.get()));
}

TEST_P(FilterableCaps, Run) {
    const GrlSupportedOps ops = GetParam();
    const Wrapper<GrlSource> source = GetMediaScannerSource();
    const Wrapper<GrlCaps> caps = wrap(grl_source_get_caps(source.get(), ops));

    ASSERT_NE(nullptr, caps);

    EXPECT_TRUE(grl_caps_get_type_filter(caps.get()) & GRL_TYPE_FILTER_AUDIO);
    EXPECT_TRUE(grl_caps_get_type_filter(caps.get()) & GRL_TYPE_FILTER_IMAGE);
    EXPECT_TRUE(grl_caps_get_type_filter(caps.get()) & GRL_TYPE_FILTER_VIDEO);

    {
        GList *const value_filter = grl_caps_get_key_filter(caps.get());
        EXPECT_NE(nullptr, value_filter);

        EXPECT_TRUE(has_key(value_filter, GRL_METADATA_KEY_TITLE));
        EXPECT_TRUE(has_key(value_filter, GRL_METADATA_KEY_DURATION));
        EXPECT_TRUE(has_key(value_filter, GRL_METADATA_KEY_MODIFICATION_DATE));
        EXPECT_TRUE(has_key(value_filter, GRL_METADATA_KEY_TRACK_NUMBER));
    }

    {
        GList *const range_filter = grl_caps_get_key_range_filter(caps.get());
        EXPECT_NE(nullptr, range_filter);

        EXPECT_TRUE(has_key(range_filter, GRL_METADATA_KEY_TITLE));
        EXPECT_TRUE(has_key(range_filter, GRL_METADATA_KEY_DURATION));
        EXPECT_TRUE(has_key(range_filter, GRL_METADATA_KEY_MODIFICATION_DATE));
        EXPECT_TRUE(has_key(range_filter, GRL_METADATA_KEY_TRACK_NUMBER));
    }
}

////////////////////////////////////////////////////////////////////////////////

TEST(Store, Test) {
    const MediaRootManagerPtr root_manager(new MediaRootManager);
    WritableMediaIndex writer(root_manager);

    const bool writer_opened = writer.Open(kMediaIndexPath);
    ASSERT_TRUE(writer_opened);

    static const char kStoreTestUri[] = "urn:x-test:store";
    static const char kStoreTestTitle[] = "Store Test Media";
    static const char kStoreTestMimeType[] = "application/binary";

    const std::string test_url = full_url(kStoreTestUri);

    const Wrapper<GrlSource> source = GetMediaScannerSource();
    EXPECT_FALSE(grl_source_test_media_from_uri(source.get(),
                                                test_url.c_str()));

    Wrapper<GError> error;
    Wrapper<GrlMedia> media = take(grl_media_new());
    grl_media_set_id(media.get(), test_url.c_str());
    grl_media_set_title(media.get(), kStoreTestTitle);
    grl_source_store_sync(source.get(), nullptr, media.get(),
                          GRL_WRITE_NORMAL, error.out_param());

    EXPECT_FALSE(error) << to_string(error);
    ASSERT_TRUE(grl_source_test_media_from_uri(source.get(),
                                               test_url.c_str()));

    const Wrapper<GrlOperationOptions> options =
            take(grl_operation_options_new(nullptr));

    Wrapper<GList> keys =
            take(grl_metadata_key_list_new(GRL_METADATA_KEY_TITLE,
                                           GRL_METADATA_KEY_MIME,
                                           GRL_METADATA_KEY_URL,
                                           GRL_METADATA_KEY_INVALID));

    media = take(grl_source_get_media_from_uri_sync(source.get(),
                                                    test_url.c_str(),
                                                    keys.get(), options.get(),
                                                    error.out_param()));

    EXPECT_FALSE(error) << to_string(error);
    ASSERT_TRUE(media);

    EXPECT_EQ(test_url, safe_string(grl_media_get_id(media.get())));
    EXPECT_EQ(test_url, safe_string(grl_media_get_url(media.get())));
    EXPECT_EQ(kStoreTestTitle, safe_string(grl_media_get_title(media.get())));
    EXPECT_EQ("", safe_string(grl_media_get_mime(media.get())));

    grl_media_set_title(media.get(), "Do not store this");
    grl_media_set_mime(media.get(), kStoreTestMimeType);

    Wrapper<GList> mime_key_list =
            take(grl_metadata_key_list_new(GRL_METADATA_KEY_MIME,
                                           GRL_METADATA_KEY_INVALID));

    Wrapper<GList> failed_keys =
            take(grl_source_store_metadata_sync(source.get(), media.get(),
                                                mime_key_list.get(),
                                                GRL_WRITE_NORMAL,
                                                error.out_param()));

    EXPECT_FALSE(error) << to_string(error);
    EXPECT_FALSE(failed_keys) << g_list_length(failed_keys.get());

    media = take(grl_source_get_media_from_uri_sync(source.get(),
                                                    test_url.c_str(),
                                                    keys.get(), options.get(),
                                                    error.out_param()));

    EXPECT_FALSE(error) << to_string(error);
    ASSERT_TRUE(media);

    EXPECT_EQ(test_url, safe_string(grl_media_get_id(media.get())));
    EXPECT_EQ(test_url, safe_string(grl_media_get_url(media.get())));
    EXPECT_EQ(kStoreTestTitle, safe_string(grl_media_get_title(media.get())));
    EXPECT_EQ(kStoreTestMimeType, safe_string(grl_media_get_mime(media.get())));
}

////////////////////////////////////////////////////////////////////////////////

TEST(Remove, Test) {
    static const char kRemoveTestUri[] = "urn:x-test:remove";
    const std::string test_url = full_url(kRemoveTestUri);

    const Wrapper<GrlSource> source = GetMediaScannerSource();
    EXPECT_FALSE(grl_source_test_media_from_uri(source.get(),
                                                test_url.c_str()));

    Wrapper<GrlMedia> media = take(grl_media_new());
    grl_media_set_id(media.get(), test_url.c_str());
    grl_media_set_url(media.get(), test_url.c_str());
    grl_media_set_title(media.get(), "Remove this");

    Wrapper<GError> error;
    grl_source_store_sync(source.get(), nullptr, media.get(),
                          GRL_WRITE_NORMAL, error.out_param());
    EXPECT_FALSE(error) << to_string(error);

    EXPECT_TRUE(grl_source_test_media_from_uri(source.get(),
                                               test_url.c_str()));

    media = take(grl_media_new());
    grl_media_set_id(media.get(), test_url.c_str());
    grl_source_remove_sync(source.get(), media.get(), error.out_param());
    EXPECT_FALSE(error) << to_string(error);

    EXPECT_FALSE(grl_source_test_media_from_uri(source.get(),
                                                test_url.c_str()));
}

////////////////////////////////////////////////////////////////////////////////

class Notify : public ::testing::Test {
protected:
    typedef std::map<std::string, std::vector<std::string> > MediaMap;

    static void pull_dbus_signals() {
        const int64_t end = g_get_monotonic_time() +
                boost::posix_time::milliseconds(50).
                total_microseconds();

        // The plugin uses sync API for D-Bus communication. Therefore only
        // method reply messages get processed, while the signal messages
        // still sit in the event loop.
        // TODO(M5): Verify that theory and maybe switch to async API.
        // Well, actually the problem also could just be that we process
        // the D-Bus signal within the task manager's background thread and
        // then push the result to the main loop by idle source.
        while (g_main_context_iteration(nullptr, false)
               || g_get_monotonic_time() < end);
    }

    static void Insert(const std::string &title_prefix,
                       const std::string &url_prefix,
                       Wrapper<GrlSource> source,
                       MediaMap *collection,
                       MediaMap::iterator *it) {
        const size_t n = collection->size() + 1;
        const std::string title = (format("{1} {2}") % title_prefix % n).str();
        const std::string url = (format("{1}-{2}") % url_prefix % n).str();

        Wrapper<GrlMedia> media = take(grl_media_new());
        grl_media_set_mime(media.get(), "application/binary");
        grl_media_set_title(media.get(), title.c_str());
        grl_media_set_url(media.get(), url.c_str());
        grl_media_set_id(media.get(), url.c_str());

        std::pair<MediaMap::iterator, bool> insert_result = collection->insert
                (make_pair(url, std::vector<std::string>() << title));

        ASSERT_TRUE(insert_result.second) << "media url: <" << url << ">";

        *it = insert_result.first;

        Wrapper<GError> error;
        grl_source_store_sync(source.get(), nullptr, media.get(),
                              GRL_WRITE_NORMAL, error.out_param());

        ASSERT_FALSE(error)
                << "media url: <" << url << ">" << std::endl
                << "error: " << to_string(error);

        ASSERT_TRUE(grl_source_test_media_from_uri(source.get(), url.c_str()))
                << "media url: <" << url << ">";

        pull_dbus_signals();
    }

    static MediaMap::iterator Insert(const std::string &title_prefix,
                                     const std::string &url_prefix,
                                     Wrapper<GrlSource> source,
                                     MediaMap *collection) {
        MediaMap::iterator it;
        Insert(title_prefix, url_prefix, source, collection, &it);
        return it;
    }

    static MediaMap::iterator InsertInvisible(Wrapper<GrlSource> source,
                                              MediaMap *collection) {
        return Insert("Invisible Media",
                      full_url("urn:x-test:notify-invisible").c_str(),
                      source, collection);
    }

    static MediaMap::iterator InsertVisible(Wrapper<GrlSource> source,
                                            MediaMap *collection) {
        return Insert("Visible Media",
                      full_url("urn:x-test:notify-visible").c_str(),
                      source, collection);
    }

    static void Modify(Wrapper<GrlSource> source, MediaMap::iterator it) {
        const std::string &title = it->second.back();
        const std::string &url = it->first;

        Wrapper<GrlMedia> media = take(grl_media_new());
        const std::string new_title = title + " Modified";
        grl_media_set_title(media.get(), new_title.c_str());
        grl_media_set_url(media.get(), url.c_str());
        grl_media_set_id(media.get(), url.c_str());

        Wrapper<GError> error;

        Wrapper<GList> keys =
                take(grl_metadata_key_list_new(GRL_METADATA_KEY_TITLE,
                                               GRL_METADATA_KEY_INVALID));

        const Wrapper<GList> failed_keys =
                take(grl_source_store_metadata_sync
                     (source.get(), media.get(), keys.get(),
                      GRL_WRITE_NORMAL, error.out_param()));

        ASSERT_FALSE(error)
                << "media url: <" << url << ">" << std::endl
                << "error: " << to_string(error);
        ASSERT_FALSE(failed_keys);

        it->second.push_back(new_title);
        pull_dbus_signals();
    }

    static void Remove(Wrapper<GrlSource> source, MediaMap::iterator it) {
        const std::string &title = it->second.back();
        const std::string &url = it->first;

        Wrapper<GrlMedia> media = take(grl_media_new());
        grl_media_set_id(media.get(), url.c_str());

        Wrapper<GError> error;
        grl_source_remove_sync(source.get(), media.get(), error.out_param());

        ASSERT_FALSE(error)
                << "media url: <" << url << ">" << std::endl
                << "error: " << to_string(error);

        it->second.push_back(title);
        pull_dbus_signals();
    }

    struct Notification {
        Wrapper<GrlMedia> media;
        GrlSourceChangeType type;
    };

    typedef std::map<std::string, std::vector<Notification> > NotificationMap;

    static void on_media_changed(GrlSource *, GPtrArray *changed_media,
                                 GrlSourceChangeType change_type,
                                 gboolean location_unknown,
                                 NotificationMap *notifications) {
        EXPECT_FALSE(location_unknown);
        EXPECT_LT(0, changed_media->len);

        for (unsigned i = 0; i < changed_media->len; ++i) {
            const Notification notification = {
                wrap(GRL_MEDIA(g_ptr_array_index(changed_media, i))),
                change_type
            };

            const std::string url = grl_media_get_url(notification.media.get());
            (*notifications)[url].push_back(notification);
        }
    }

    static void verify_notifications(const MediaMap &visible_media,
                                     const MediaMap &invisible_media,
                                     const NotificationMap &notifications) {
        // Check that all "visible" media got notified.

        EXPECT_EQ(visible_media.size(), notifications.size());

        for (const auto vm: visible_media) {
            const std::string &url = vm.first;

            const NotificationMap::const_iterator nit = notifications.find(url);

            EXPECT_TRUE(nit != notifications.end())
                    << "media url: <" << url << ">";
            ASSERT_EQ(vm.second.size(), nit->second.size())
                    << "media url: <" << url << ">";

            for (size_t i = 0; i < vm.second.size(); ++i) {
                const GrlSourceChangeType expected_change_type =
                        i == 0 ? GRL_CONTENT_ADDED
                               : i == 2 ? GRL_CONTENT_REMOVED
                                        : GRL_CONTENT_CHANGED;

                const std::string notified_title = safe_string
                        (grl_media_get_title(nit->second.at(i).media.get()));

                EXPECT_EQ(expected_change_type, nit->second.at(i).type)
                        << "media url: <" << url << ">#" << i;

                if (expected_change_type != GRL_CONTENT_REMOVED) {
                    EXPECT_EQ(vm.second.at(i), notified_title)
                            << "media url: <" << url << ">#" << i;
                }
            }
        }

        // Ensure there were no notifications for "invisible" media.

        for (const auto vm: invisible_media) {
            const std::string &url = vm.first;

            EXPECT_TRUE(notifications.find(url) == notifications.end())
                    << "media url: <" << url << ">";
        }
    }
};

TEST_F(Notify, Test) {
    const Wrapper<GrlSource> source = GetMediaScannerSource();
    NotificationMap notifications;
    MediaMap invisible_media;
    MediaMap visible_media;
    Wrapper<GError> error;

    g_signal_connect(source.get(), "content-changed",
                     G_CALLBACK(&Notify::on_media_changed),
                     &notifications);

    for (int i = 0; i < 10 && not HasFailure(); ++i) {
        // Insert and update "invisible" media...

        InsertInvisible(source, &invisible_media);
        verify_notifications(visible_media, invisible_media, notifications);

        MediaMap::iterator it = InsertInvisible(source, &invisible_media);
        verify_notifications(visible_media, invisible_media, notifications);

        Modify(source, it);
        verify_notifications(visible_media, invisible_media, notifications);

        Remove(source, it);
        verify_notifications(visible_media, invisible_media, notifications);

        // Start notification service...

        const bool change_notifications_started =
                grl_source_notify_change_start(source.get(), error.out_param());

        EXPECT_FALSE(error) << to_string(error);
        ASSERT_TRUE(change_notifications_started);

        // Insert and update "visible" media...

        InsertVisible(source, &visible_media);
        verify_notifications(visible_media, invisible_media, notifications);

        it = InsertVisible(source, &visible_media);
        verify_notifications(visible_media, invisible_media, notifications);

        Modify(source, it);
        verify_notifications(visible_media, invisible_media, notifications);

        Remove(source, it);
        verify_notifications(visible_media, invisible_media, notifications);

        // Stop notification service...

        const bool change_notifications_stopped =
                grl_source_notify_change_stop(source.get(), error.out_param());

        EXPECT_FALSE(error) << to_string(error);
        ASSERT_TRUE(change_notifications_stopped);
    }

    // Some final paranoia...

    pull_dbus_signals();
    verify_notifications(visible_media, invisible_media, notifications);
}

TEST_F(Notify, Recursive) {
    const Wrapper<GrlSource> source = GetMediaScannerSource();
    Wrapper<GError> error;

    ASSERT_TRUE(grl_source_notify_change_start(source.get(),
                                               error.out_param()));
    EXPECT_FALSE(error) << to_string(error);
    ASSERT_TRUE(grl_source_notify_change_start(source.get(),
                                               error.out_param()));
    EXPECT_FALSE(error) << to_string(error);

    ASSERT_TRUE(grl_source_notify_change_stop(source.get(),
                                              error.out_param()));
    EXPECT_FALSE(error) << to_string(error);
    ASSERT_TRUE(grl_source_notify_change_stop(source.get(),
                                              error.out_param()));
    EXPECT_FALSE(error) << to_string(error);

    // Error raised on unbalanced call to notify_change_stop()
    ASSERT_FALSE(grl_source_notify_change_stop(source.get(),
                                               error.out_param()));
    ASSERT_TRUE(error);

    pull_dbus_signals();
}

////////////////////////////////////////////////////////////////////////////////

class LuceneEnvironment : public mediascanner::LuceneEnvironment {
public:
    LuceneEnvironment()
        : mediascanner::LuceneEnvironment(kMediaIndexPath) {
    }

    void FillMediaIndex(WritableMediaIndex *writer) {
        for (const auto &p: make_known_media()) {
            writer->Insert(ToUnicode(p.first), p.second);

            const std::string error_message = writer->error_message();

            ASSERT_TRUE(error_message.empty())
                    << " Message: " << error_message << std::endl
                    << "   Media: " << p.first;
        }
    }
};

void SetupTestEnvironments() {
    using testing::AddGlobalTestEnvironment;

    AddGlobalTestEnvironment(new LuceneEnvironment);
    AddGlobalTestEnvironment(new DBusEnvironment(kDataPath, kMediaIndexPath));
    AddGlobalTestEnvironment(new MediaScannerEnvironment(kMediaIndexPath));
}

} // namespace griloplugintest
} // namespace mediascanner

////////////////////////////////////////////////////////////////////////////////

int main(int argc, char *argv[]) {
    mediascanner::InitTests(&argc, argv);

    grl_init(&argc, &argv);

    mediascanner::griloplugintest::SetupTestEnvironments();

    return RUN_ALL_TESTS();
}
