package latexDraw.parsers.latexdraw;

import java.awt.geom.Point2D;
import java.util.Stack;
import java.util.Vector;

import latexDraw.figures.AkinPoints;
import latexDraw.figures.Figure;
import latexDraw.figures.FramedBox;
import latexDraw.figures.Text;
import latexDraw.parsers.*;
import latexDraw.parsers.pst.PSTParameters;
import latexDraw.parsers.pst.PSTParser;
import latexDraw.parsers.pst.PSTParameters.PositionParameters;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * Defines an PSTricks parser specific to latexdraw.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/21/08<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.2<br>
 */
public class LPSTParser extends PSTParser
{
	/** The parsed shapes. */
	protected Vector<Figure> shapes;
	
	/** The figures contained in the pscustom command. Cleared at the end of each <code>pscustom</code> command.*/
	public Vector<Figure> psCustFigures;
	
	
	/**
	 * Creates and initialises the pstricks parser.
	 * @param code The code to parse.
	 * @throws IllegalArgumentException If the given code is null.
	 */
	public LPSTParser(String code)
	{
		super(code);
		
		shapes 		  = new Vector<Figure>();
		psCustFigures = new Vector<Figure>();
	}

	
	
	/**
	 * Parses only the framed boxes of the code. Used to save and read text shape
	 * saved into SVG documents.
	 * @throws TooManyBracketsException If there is a problem of brackets
	 * @throws InvalidFormatCommandException If the format of a command is not respected.
	 * @throws UnclosedBracketsException If a bracket is not closed.
	 * @since 2.0.2
	 */
	public void parseFramedBox() throws TooManyBracketsException, InvalidFormatCommandException, UnclosedBracketsException
	{
		boolean isFramedBox = false;
		Stack<Boolean> ignorableBracket = new Stack<Boolean>();
		
		initialise();
		started = true;
		params.add(new PSTParameters());
		
		while(!isEOC())
			try
			{
				switch(getChar())
				{
					case '\\':
						int pos 			= getPosition();
						boolean ok 			= true;
						PSTParameters tp 	= params.lastElement();
						
						switch(nextChar())
						{
							case 'p':
								switch(nextChar())
								{
									case 's':// \ps
										switch(nextChar())
										{
											case 'f': // \psf
												if(nextChar()=='r' && nextChar()=='a' && nextChar()=='m' && nextChar()=='e' && 
												   nextChar()=='b' && nextChar()=='o' && nextChar()=='x' && !Character.isLetter(nextChar()))
												{// \psframebox
													if(tp.psBox.isEmpty())
														tp.psBoxText.setText("");//$NON-NLS-1$
													
													actionsOnFramedBox(0);
													isFramedBox = true;
												}
												else
													ok = false;
												
												break;
												
											case 'd': // \psd
												switch(nextChar())
												{
													case 'i': // \psdi
														if(nextChar()=='a' && nextChar()=='b' && nextChar()=='o' && 
														   nextChar()=='x' && !Character.isLetter(nextChar()))
														{// \psdiabox
															actionsOnFramedBox(3);
															isFramedBox = true;
														}
														else
															ok = false;
														
														break;
														
													case 'b': // \psdb
														if(nextChar()=='l' && nextChar()=='f' && nextChar()=='r' && 
														   nextChar()=='a' && nextChar()=='m' && nextChar()=='e' && 
														   nextChar()=='b' && nextChar()=='o' && nextChar()=='x' && 
														   !Character.isLetter(nextChar())) 
														{// \psdblframebox
															actionsOnFramedBox(5);
															isFramedBox = true;
														}
														else
															ok = false;
														break;
														
													default:
														ok = false;
														break;
												}
												break;
												
											case 'c': // \psc
												if(nextChar()=='i' && nextChar()=='r' && nextChar()=='c' && 
												   nextChar()=='l' && nextChar()=='e' && nextChar()=='b' && 
												   nextChar()=='o' && nextChar()=='x' && !Character.isLetter(nextChar())) 
												{// \pscirclebox
													actionsOnFramedBox(1);
													isFramedBox = true;
												}
												else
													ok = false;
												
												break;
												
											case 't': // \pst
												if(nextChar()=='r' && nextChar()=='i' && nextChar()=='b' && 
												   nextChar()=='o' && nextChar()=='x' && !Character.isLetter(nextChar())) 
												{// \pstribox
													actionsOnFramedBox(2);
													isFramedBox = true;
												}
												else
													ok = false;
												
												break;
												
											case 'o': // \pso
												if(nextChar()=='v' && nextChar()=='a' && nextChar()=='l' &&
												   nextChar()=='b' && nextChar()=='o' && nextChar()=='x' && !Character.isLetter(nextChar())) 
												{// \psovalbox
													actionsOnFramedBox(4);
													isFramedBox = true;
												}
												else
													ok = false;
			
												break;
												
											case 's': // \pss
												if(nextChar()=='h' && nextChar()=='a' && nextChar()=='d' &&
												   nextChar()=='o' && nextChar()=='w' && nextChar()=='b' && 
												   nextChar()=='o' && nextChar()=='x' && !Character.isLetter(nextChar())) 
												{// \psshadowbox
													actionsOnFramedBox(6);
													isFramedBox = true;
												}
												else
													ok = false;
			
												break;
										}
										break;
										
									case 'u': // \pu
										if(nextChar()=='t' && !Character.isLetter(nextChar())) // \put
										{
											skipWSPComments();
											parse_put();
											skipWSPComments();
											
											if(getChar()!='{') throw new InvalidFormatCommandException(getLinePosition());
											ignorableBracket.add(false);
											nextChar();
										}
										else ok = false;
										break;
										
									default:
									ok = false;
								}
								break;
								
							case 'd':// \d
								if(nextChar()=='e' && nextChar()=='f' && nextChar()=='i' && nextChar()=='n' && 
									nextChar()=='e' && nextChar()=='c' && nextChar()=='o' && nextChar()=='l' && 
									nextChar()=='o' && nextChar()=='r' && !Character.isLetter(nextChar()))
								{// \definecolor
									parse_definecolor();
								}
								else ok = false;
								break;
								
								
							case 'c': // \c
								if(nextChar()=='o' && nextChar()=='l' && nextChar()=='o' && nextChar()=='r' && 
								   !Character.isLetter(nextChar()))
								{// \color
									parse_color();
								}
								else ok = false;
								break;
								
							case '{' : // \{
							case '}' : // \}
							case '$' : // \$
							case '&' : // \&
							case '#' : // \#
							case '%' : // \%
							case '\\': // \\
								setPosition(getPosition()-1);
								addParsedText();
								nextChar();
								addParsedText();
								nextChar();
								break;
								
							default:
								ok = false;
							break;
						}
						
						if(!ok) {
							setPosition(pos);
							addParsedText();
							nextChar();
						}
						
						break;
					
					case '{' :
						if(isFramedBox) {
							params.add(new PSTParameters(params.lastElement()));
							params.lastElement().fromFramedBox = true;
							isFramedBox = false;
							ignorableBracket.add(false);
						}
						else {
							addParsedText();
							ignorableBracket.add(true);
						}
						
						nextChar();
						break;
						
					case '}':
						if(params.size()<1) throw new TooManyBracketsException(getLinePosition());
							
						if(!ignorableBracket.isEmpty() && ignorableBracket.lastElement())
							addParsedText();
						else
						{
							PSTParameters tp1 = params.lastElement();
							
							if(tp1.fromFramedBox)
							{
								PSTParameters pp = null;
								int k = params.size()-1;
								
								while(pp==null && k>=0)
									if(params.elementAt(k).getPosition()==null)
										k--;
									else
										pp = params.elementAt(k);
								
								if(pp==null)
									pp = params.firstElement();
								
								pp.psBox.lastElement().setEnd(pp.textForFrame.length());
								pp.psBox.add(0, pp.psBox.remove(pp.psBox.size()-1));
							}
							else
								if(tp1.getPosition()!=null && !tp1.psBox.isEmpty())
								{
									Vector<FramedBox> fbs = new Vector<FramedBox>();
									while(!tp1.psBox.isEmpty())
										fbs.add(tp1.psBox.remove(tp1.psBox.size()-1));
									tp1.psBox = fbs;
										
									tp1.psBoxText.setText(tp1.textForFrame);
									actionsOnTerminatedFramedBoxes();
								}
								else
									try {actionsOnText();}
									catch(InvalidFormatCommandException e) { /* Nothing to do.*/ }
									
							params.removeElementAt(params.size()-1);
						}
						
						if(!ignorableBracket.isEmpty()) ignorableBracket.remove(ignorableBracket.size()-1);
						nextChar();
						break;
						
					case '$':
						parseMathText();
						break;
						
					default:
						if(isEOL())
							incLinePosition();
					
						addParsedText();
						
						if(isWSP())
						{
							nextChar();
							skipWSP();// Only a piece of blank must be took in account.
						}
						else
							nextChar();
						
						break;
				}//switch
			}//while
		catch(UnexceptedEndOfFileException ex) { ex.printStackTrace(); }
		
		if(params.size()>1) throw new UnclosedBracketsException();
	}
	
	
	

	@Override
	public void actionsOnterminatedPsCustom() throws NotFullyManagedException
	{
		Figure f;
		int i=0;
		
		while(!psCustFigures.isEmpty())
		{
			f = psCustFigures.remove(0);
			
			if(!(f instanceof AkinPoints) || ((AkinPoints)f).getNbPoints()>1)
			{
				if(f instanceof AkinPoints && ((AkinPoints)f).getType()==AkinPoints.TYPE_CURVES)
				{// We must transform the read points.
					AkinPoints ak = (AkinPoints)f;
					int j, size = ak.getNbPoints();
					LaTeXDrawPoint2D prev, curr = ak.getPoint(0);
					
					for(j=1; j<size; j++)
					{
						prev = curr;
						curr = ak.getPoint(j);
						curr.setLocation(2*curr.x-prev.x, 2*curr.y-prev.y);
					}
					
					((AkinPoints)f).updateBorders();
				}
				
				shapes.add(f);
				i++;
			}
		}
	}

	
	
	@Override
	public void initialise()
	{
		super.initialise();
		
		setPosition(0);
		started  = false;
		shapes.clear();
		PSTParameters.reinitialize();
	}
	
	
	
	/**
	 * @return the parsed shapes.
	 * @since 2.0.2
	 */
	public Vector<Figure> getShapes()
	{
		return shapes;
	}




	@Override
	public void actionsOnFramedBox(int type) throws UnclosedBracketsException
	{
		boolean hasStar;
		int i = params.size()-1;
		PSTParameters pp = null;
		
		while(pp==null && i>=0)
			if(params.elementAt(i).getPosition()!=null)
				pp = params.elementAt(i);
			else 
				i--;
			
		if(pp==null)
			pp = params.firstElement();
		
		FramedBox fb = new FramedBox(pp.psBoxText);
		
		fb.setStart(pp.textForFrame.length());
		pp.psBox.add(fb);
		PSTParameters newP = new PSTParameters(params.lastElement());
		
		if(getChar()=='*') {
			hasStar = true;
			nextChar();
		}
		else hasStar = false;
		
		paramParser.setParam(newP);
		paramParser.parse();
		paramParser.setParam(null);
		
		skipWSPComments();
		
		switch(type)
		{
			case 0:
				fb.setBoxType(FramedBox.BOX_RECTANGLE); 
				break;
			case 1: 
				fb.setBoxType(FramedBox.BOX_CIRCLE); 
				break;
			case 2: 
				fb.setBoxType(FramedBox.BOX_TRIANGLE); 
				break;
			case 3: 
				fb.setBoxType(FramedBox.BOX_DIAMOND); 
				break;
			case 4: 
				fb.setBoxType(FramedBox.BOX_ELLIPSE);
				break;
			case 5: 
				fb.setBoxType(FramedBox.BOX_RECTANGLE); 
				break;
			case 6: 
				fb.setBoxType(FramedBox.BOX_RECTANGLE); 
				break;
		}
		
		if(getChar()!='{')//psframebox3, psframebox[framesep=0.3]3 for instance.
		{
			PSTParameters p;
			boolean again = true;
			int k = params.size()-1;
			
			while(again && k>=0)
			{
				p = params.elementAt(k);
				p.textForFrame+=(char)getChar();
				
				if(p.getPosition()==null)
					k--;
				else
					again = false;
			}
			
			k = params.size()-1;
			p=null;
			
			while(p==null && k>=0)
				if(params.elementAt(k).getPosition()==null)
					k--;
				else
					p = params.elementAt(k);
			
			if(p==null)
				p = params.firstElement();
			
			p.psBox.lastElement().setEnd(p.textForFrame.length());
			p.psBox.add(0, p.psBox.remove(p.psBox.size()-1));
		}
		
		fb.setFrameSep(newP.frameSep*Figure.PPC);
		fb.setBoxSep(newP.boxSep);
		
		setFigureParameters(newP, fb.getBox(), hasStar);
		
		if(hasStar)
		{
			fb.getBox().setIsFilled(true);
			fb.getBox().setInteriorColor(newP.fillColor);
			fb.getBox().setLinesColor(newP.fillColor);
		}
		
		switch(type)
		{
			case 5: 
				fb.getBox().setHasDoubleBoundary(true);
				break;
			case 6: 
				fb.getBox().setHasShadow(true);
				break;
		}
	}
	
	
	
	@Override
	public void actionsOnTerminatedFramedBoxes() throws InvalidFormatCommandException
	{
		PSTParameters pp = params.lastElement();
		String str 		 = pp.psBoxText.getText();
		
		if(str.length()==0)
			str = " ";//$NON-NLS-1$
		
		pp.textParsed = pp.textForFrame;
		actionsOnText();
		
		if(pp.psBox.isEmpty())
			return;
		
		if(shapes.isEmpty())
			return;
		
		Text txt = (Text)shapes.remove(shapes.size()-1);
		boolean simpleBox = pp.psBox.size()==1 && pp.psBox.firstElement().getBoxedText().equals(txt.getText());
		
		if(simpleBox)
		{
			FramedBox fb = pp.psBox.firstElement();
			fb.setText(txt);
			fb.setStart(-1);
			fb.setEnd(-1);
			fb.getBox().setBordersPosition(PSTricksConstants.BORDERS_OUTSIDE);
			txt.setSimpleBox(fb);
		}
		else
			for(FramedBox fb : pp.psBox)
			{
				fb.setText(txt);
				txt.addBox(fb);
				fb.getBox().setBordersPosition(PSTricksConstants.BORDERS_OUTSIDE);
			}
		
		txt.setHasFramedBox(!pp.psBox.isEmpty());
		txt.setHasSimpleFramedBox(simpleBox);
		txt.updateFramedBoxes();
		txt.updateFontsBorders();
		
		if(!pp.psCustomP.fromPsCustom)
			shapes.add(txt);
		
		if(simpleBox)
		{
			if(txt.getSimpleBox().getBoxType()==FramedBox.BOX_TRIANGLE)
			{
				double height = (txt.getSimpleBox().getBox().getTheSEBoundPoint().y-
						txt.getSimpleBox().getBox().getTheNWBoundPoint().y)/4.;
				LaTeXDrawPoint2D pos = txt.getPosition();
				txt.setPosition(new LaTeXDrawPoint2D(pos.x, pos.y+height));
				txt.getSimpleBox().getBox().updateShape();
			}
		}
		else
		{
			FramedBox fb, max=null;
			double xMin=Double.MAX_VALUE;
			int i=0, size = txt.getMultipleBox().size();
			
			while(i<size)
			{
				fb = txt.getMultipleBox().elementAt(i);

				if(fb.isBoxSep() && fb.getBoxType()==FramedBox.BOX_TRIANGLE &&
					fb.getBox().getTheNWBoundPoint().x<xMin)
				{
					xMin = fb.getBox().getTheNWBoundPoint().x;
					max = fb;
				}
				else i++;
			}

			if(max!=null)
			{
				double height = (max.getBox().getTheSEBoundPoint().y-max.getBox().getTheNWBoundPoint().y)/4.;
									
				LaTeXDrawPoint2D pos = txt.getPosition();
				txt.setPosition(new LaTeXDrawPoint2D(pos.x, pos.y+height));
				max.getBox().updateShape();
			}
			
		}
		
		txt.updateFramedBoxes();
		txt.updateFontsBorders();
	}
	
	
	
	/**
	 * Sets the parameters of a figure <code>f</code>.
	 * @param p The set of pstricks parameters to use to set the shape.
	 * @param f The shape to set.
	 * @param hasStar True if the command of the pstricks shape is followed by a star
	 * (for instance: psframe*[...](..)(..)).
	 */
	public void setFigureParameters(PSTParameters p, Figure f, boolean hasStar)
	{
		if(f==null || p==null)
			return ;
		
		if(f.canHaveShadow())
		{
			f.setHasShadow(p.isShadow);
			f.setShadowAngle(Math.toRadians(p.shadowAngle));
			f.setShadowSize(Math.abs(p.shadowSize)*Figure.PPC);
			f.setShadowColor(p.shadowCol);
		}
		
		if(f.canBeHatched())
		{
			f.setHatchingSep(p.hatchSep*Figure.PPC);
			f.setGradientAngle(Math.toRadians(p.gradAngle));
			f.setGradientEndColor(p.gradEnd);
			f.setGradientMidPoint(p.gradMidPoint);
			f.setGradientStartColor(p.gradBegin);
		}
	
		f.setLinesColor(p.lineColor);
		f.setBordersPosition(p.borderPos);
		f.setDoubleColor(p.dbleColor);
		f.setDoubleSep(Math.abs(p.dbleSep*Figure.PPC));
		f.setHasDoubleBoundary(p.dbleLine);
		f.setHatchingColor(p.hatchCol);
		if(p.fillStyle.equals(PSTricksConstants.TOKEN_FILL_SOLID))
			 f.setHatchingStyle(PSTricksConstants.TOKEN_FILL_NONE);
		else f.setHatchingStyle(
				p.fillStyle.endsWith("*") ? p.fillStyle.substring(0, p.fillStyle.length()-1) : p.fillStyle);//$NON-NLS-1$
		f.setHatchingWidth((float)Math.abs(p.hatchWidth*Figure.PPC));
		f.setDotSep((float)  p.dotStep*Figure.PPC);
		f.setHatchingAngle(Math.toRadians(p.hatchAngle));
		f.setBlackDashLength((float)p.dashBlack*Figure.PPC);
		f.setWhiteDashLength((float)p.dashWhite*Figure.PPC);
		f.setInteriorColor(p.fillColor);
		f.setIsFilled(p.fillStyle.equals(PSTricksConstants.TOKEN_FILL_SOLID) ||
				p.fillStyle.equals(PSTricksConstants.TOKEN_FILL_CROSSHATCH_F) ||
				p.fillStyle.equals(PSTricksConstants.TOKEN_FILL_HLINES_F) ||
				p.fillStyle.equals(PSTricksConstants.TOKEN_FILL_VLINES_F));
		f.setLineStyle(p.lineStyle);
		f.setThickness((float)(Math.abs(p.lineWidth*Figure.PPC)));
		
		if(hasStar)
		{
			f.setIsFilled(true);
			f.setInteriorColor(f.getLinesColor());
			f.setHasDoubleBoundary(false);	
			f.setLineStyle(PSTricksConstants.LINE_NONE_STYLE);
			f.setHatchingStyle(PSTricksConstants.TOKEN_FILL_NONE);
			f.setBordersPosition(PSTricksConstants.BORDERS_INSIDE);
		}
	}




	@Override
	public void actionsOnText() throws InvalidFormatCommandException
	{
		if(params.isEmpty()) return;
		
		if(params.lastElement().textParsed.length()!=0)
		{
			PSTParameters last = params.lastElement();
			PositionParameters pp = null;
			int i = params.size()-1;
			
			while(pp==null && i>=0)
			{
				pp = params.elementAt(i).getPosition();
				i--;
			}
			
			if(pp==null)
				pp = new PositionParameters();
			
			Text t = new Text(true);
			Point2D.Double posit = moveCoordinate(new Point2D.Double(0,0));
			double angle = Math.toRadians(PSTParameters.getWholeRotationAngle(params));
			
			if(posit==null) throw new InvalidFormatCommandException("text", -1); //$NON-NLS-1$
			
			if(last.textSize!=null && last.textSize.length()!=0)
				t.setSizeByCommand(last.textSize);
			
			if((last.textSeries!=null && last.textSeries.equals("b")) || last.textBold) //$NON-NLS-1$
				 t.setIsBold(true);
			else t.setIsBold(false);
			
			if((last.textShape!=null && (last.textShape.equals("sl") || //$NON-NLS-1$
			   last.textShape.equals("it")) || last.textItalic)) //$NON-NLS-1$
				 t.setIsItalic(true);
			else t.setIsItalic(false);
			
			t.setRotationAngle(-angle);
			t.setText(last.textParsed);
			t.setLinesColor(last.textColor);
			
			if(last.textFamily!=null && last.textFamily.length()!=0)
				t.setTextFontByFamily(last.textFamily);
			
			t.updateFontsBorders();
			
			LaTeXDrawPoint2D pos = new LaTeXDrawPoint2D(posit.x*Figure.PPC-t.getWidth()/2., 
					(posit.y*Figure.PPC-t.getFontMetrics().getHeight()/4.)*-1);
			
			if(pp.refPoint!=PositionParameters.REF_POINT_DEFAULT_CENTER)
			{
				double addX=0, addY=0;
				Point2D.Double gc = new Point2D.Double((pos.x+t.getWidth())/2.,(pos.y+t.getHeight())/2.);
				
				if(pp.refPoint==PositionParameters.REF_POINT_BASE)
					addY=-t.getFontMetrics().getDescent();
				else if(pp.refPoint==PositionParameters.REF_POINT_TOP)
					addY=t.getFontMetrics().getAscent()/2.;
				else if(pp.refPoint==PositionParameters.REF_POINT_BOTTOM)
					addY=-t.getFontMetrics().getDescent()*2;
				else if(pp.refPoint==PositionParameters.REF_POINT_LEFT)
					addX=t.getWidth()/2.;
				else if(pp.refPoint==PositionParameters.REF_POINT_RIGHT)
					addX=-t.getWidth()/2.;
				else if(pp.refPoint==PositionParameters.REF_POINT_RIGHT+PositionParameters.REF_POINT_BASE)
				{
					addY=-t.getFontMetrics().getDescent();
					addX=-t.getWidth()/2.;
				}else if(pp.refPoint==PositionParameters.REF_POINT_RIGHT+PositionParameters.REF_POINT_BOTTOM)
				{
					addY=-t.getFontMetrics().getDescent()*2;
					addX=-t.getWidth()/2.;
				}else if(pp.refPoint==PositionParameters.REF_POINT_RIGHT+PositionParameters.REF_POINT_TOP)
				{
					addY=t.getFontMetrics().getAscent()/2.;
					addX=-t.getWidth()/2.;
				}else if(pp.refPoint==PositionParameters.REF_POINT_LEFT+PositionParameters.REF_POINT_BASE)
				{
					addY=-t.getFontMetrics().getDescent();
					addX=t.getWidth()/2.;
				}else if(pp.refPoint==PositionParameters.REF_POINT_LEFT+PositionParameters.REF_POINT_BOTTOM)
				{
					addY=-t.getFontMetrics().getDescent()*2;
					addX=t.getWidth()/2.;
				}else if(pp.refPoint==PositionParameters.REF_POINT_LEFT+PositionParameters.REF_POINT_TOP)
				{
					addY=t.getFontMetrics().getAscent()/2.;
					addX=t.getWidth()/2.;
				}
				
				addX+=pp.labelSep*Figure.PPC;
				
				Point2D.Double pt = rotatePoint(new Point2D.Double(gc.x+addX, gc.y-addY), gc, angle);
				pos.setLocation(pos.x+(pt.x-gc.x), pos.y-(pt.y-gc.y));
			}
			
			t.setPosition(pos);
			t.updateFontsBorders();
			
			if(last.psCustomP.fromPsCustom)
				psCustFigures.add(t);
			else
				shapes.add(t);
		}
	}
}
