/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#define _GNU_SOURCE /* strndup() */

/**
   \file cdw_ncurses.c
   \brief Set of functions that support use of ncurses library

   termcap/terminfo: it seems that in some circumstances keys F1-F4 emit
   very different values than keys F5-F12. Output from demo_altkeys tool
   (from ncurses examples) gives following results for F1-F10 keys:
   Keycode 27, name ^[
   Keycode 91, name [
   Keycode 49, name 1
   Keycode 49, name 1
   Keycode 126, name ~
   Keycode 27, name ^[
   Keycode 91, name [
   Keycode 49, name 1
   Keycode 50, name 2
   Keycode 126, name ~
   Keycode 27, name ^[
   Keycode 91, name [
   Keycode 49, name 1
   Keycode 51, name 3
   Keycode 126, name ~
   Keycode 27, name ^[
   Keycode 91, name [
   Keycode 49, name 1
   Keycode 52, name 4
   Keycode 126, name ~
   Keycode 269, name KEY_F(5)
   Keycode 270, name KEY_F(6)
   Keycode 271, name KEY_F(7)
   Keycode 272, name KEY_F(8)
   Keycode 273, name KEY_F(9)
   Keycode 274, name KEY_F(10)

   Key F1 emits n characters instead of one: "^[OP" or "^[[11~", where
   "^[" is an escape sequence ("\033"). Normally such sequences can't be
   recognized as function keys, and any code awaiting for function keys
   may fail. Using define_key() adds new definitions of function keys,
   and that fixes the problem.

   Initial bug report on this was sent by robwoj44, more info on function
   keys was found in mc-4.7.0.2/lib/tty/key.c and here:

   [1] http://fixunix.com/bsd/88211-ncurses-function-keys.html
   [2] http://invisible-island.net/ncurses/ncurses.faq.html#xterm_color
   [3] http://totalrecall.wordpress.com/2008/12/22/xterm-color-key-issues-fixed/
   [4] https://bugs.launchpad.net/gnome-terminal/+bug/96676

   [2] mentions terminal type (xterm-color) as one of conditions under which
   the problem occurs.

   Example of usage of define_key() was found in
   ncurses-5.7+20090803/test/demo_altkeys.c

*/

#include <stddef.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>

#include "cdw_string.h"
#include "cdw_debug.h"
#include "main.h"
#include "cdw_ncurses.h"
#include "cdw_utils.h"
#include "cdw_widgets.h"
#include "gettext.h"


extern cdw_arguments_t cdw_commandline_arguments;

/* used by cdw_ncurses_get_field_string() */
static char empty_string[] = "";

static void cdw_ncurses_get_field_buffer_check(FIELD *field, size_t limit);
static void cdw_ncurses_make_custom_key_definitions(void);


/**
   \brief Run common ncurses setup code

   Simple wrapper for few ncuses functions that should be called
   when every ncurses program starts.

   \return CDW_OK on success
   \return CDW_GEN_ERROR on failure
*/
cdw_rv_t cdw_ncurses_init(void)
{
	initscr();
	start_color();
	cbreak();
	noecho();
	keypad(stdscr, TRUE);

	if (cdw_commandline_arguments.escdelay_ms != -1) {
		set_escdelay(cdw_commandline_arguments.escdelay_ms);
	} else {
		/* we are about to modify global variable ESCDELAY;
		   this is a library variable, not a shell variable;
		   I'm not sure what "global" means in this context,
		   man page for set_escdelay suggests that it is global
		   in "screen" scope; just to be on the safe side I will
		   check if the variable isn't already modified by other
		   apps to "low enough" level;

		   of course this does not prevent cdw from modifying
		   ESCDELAY if some other app modified it to some "large
		   enough" value... */
		if (ESCDELAY <= 150) {
			/* low enough, don't modify */
		} else {
			set_escdelay(50); /* milliseconds */
		}
	}

	/* turn cursor off - now cursor will be	turned on only if needed */
	curs_set(0);

	/* can't do this here since cdw colors module depends on
	   paths being set in fs module; fs module is called after
	   cdw_curses_init(); you have to call curses_colors_init()
	   after cdw_fs_init(); */
	/* curses_colors_init(); */

	int lines = getmaxy(stdscr);
	int cols = getmaxx(stdscr);
	if ((lines < 24) || (cols < 80)) {
		cdw_ncurses_clean();
		/* 2TRANS: this message displayed in console when program
		   detects that requirement about console size is not met
		   and exits; please keep '\n' chars */
		fprintf(stderr, _("\nCannot start cdw: cdw needs at least 80x24 terminal!\n"));
		cdw_vdm ("ERROR: failed to initialize curses, either lines (%d) or cols (%d) value is too small\n", lines, cols);
		return CDW_ERROR;
	} else {
		cdw_sdm ("INFO: lines = %d, cols = %d\n", lines, cols);
	}

	cdw_ncurses_make_custom_key_definitions();

	return CDW_OK;
}





void cdw_ncurses_clean(void)
{
	endwin();

	return;
}





/**
   \brief Copy content of buffer associated with given field to given buffer

   Function copies content of buffer nr 0 of given field to given buffer.
   If \p limit > 0, then no more than \p limit chars are copied. Data in
   target buffer is ended with '\0'.

   The function was closely related to cdw_one_line_input() and copies its
   specification of \p limit and \p buffer and relation between them.

   \param field - field from which data will be copied
   \param buffer - buffer to which the data will be copied
   \param limit - maximal number of chars to copy, does not include ending nul

   \return CDW_OK on success
   \return CDW_ERROR on malloc error
 */
cdw_rv_t cdw_ncurses_get_field_buffer(FIELD *field, char **buffer, size_t limit)
{
#ifndef NDEBUG
	cdw_ncurses_get_field_buffer_check(field, limit);
#endif

	/* note that this also works for empty field buffer
	   (field buffer = ""); caller of this function must
	   decide if empty string is a valid value */
	size_t len = strlen((char *) cdw_string_rtrim(field_buffer(field, 0)));
	cdw_vdm ("INFO: len of R-trimmed field's buffer = %zd, R-trimmed buffer = \"%s\"\n",
		 len, cdw_string_rtrim(field_buffer(field, 0)));

	if (limit != 0) {
		if (len > limit) {
			/* this shouldn't happen since we limited size of
			   field with set_max_field(field[0], limit) */
			cdw_vdm ("ERROR: field's buffer will be truncated from %zd to %zd because of len limit\n", len, limit);
			len = limit;
		}
	}

	cdw_vdm ("INFO: reallocing buffer for len = %zd\n", len);
	char *tmp = (char *) realloc(*buffer, len + 1);
	if (tmp == (char *) NULL) {
		cdw_vdm ("ERROR: failed to reallocate buffer with new len %zd\n", len);
		return CDW_ERROR;
	} else {
		*buffer = tmp;
		strncpy(*buffer, (char *) cdw_string_rtrim(field_buffer(field, 0)), len);
		*(*buffer + len) = '\0';
		cdw_vdm ("INFO: field's buffer saved as \"%s\"\n", *buffer);
		cdw_vdm ("INFO: %zd chars in saved string / %zd chars limit\n", len, limit);
		return CDW_OK;
	}
}





/**
   \brief Check some assertions about cdw_ncurses_get_field_buffer()'s arguments
*/
void cdw_ncurses_get_field_buffer_check(FIELD *field, size_t limit)
{
	Field_Options opts = field_opts(field);

	if ((unsigned int) opts & O_STATIC) {
		int rows;
		int cols;
		int frow;
		int fcol;
		int nrow;
		int nbuf;
		int res = field_info(field, &rows, &cols, &frow, &fcol, &nrow, &nbuf);
		cdw_assert (res == CDW_OK, "ERROR: failed to fetch dynamic field info\n");
		if (rows * cols >= (int) limit) {
			cdw_vdm ("ERROR: field->rows * field->cols < (int) limit (%d < %zd), you passed wrong limit to the function\n",
				 rows * cols , limit);
		}
	} else {
		int rows;
		int cols;
		int max;
		int res = dynamic_field_info(field, &rows, &cols, &max);
		cdw_assert (res == CDW_OK, "ERROR: failed to fetch dynamic field info\n");
		if (max != (int) limit) {
			cdw_vdm ("ERROR: field->max_size != field limit (%d != %zd), you passed wrong limit to the function\n",
				 max, limit);
		}
	}

	return;
}





/* names are from /usr/include/ncursesw/eti.h,
   descriptions come from ncurses man pages */
static cdw_id_clabel_t cdw_ncurses_error_strings[] = {
	{ E_OK,                "E_OK"              },  /* The routine succeeded */
	{ E_SYSTEM_ERROR,      "E_SYSTEM_ERROR"    },  /* System error occurred, e.g., malloc failure. */
	{ E_BAD_ARGUMENT,      "E_BAD_ARGUMENT"    },  /* Routine detected an incorrect or out-of-range argument. */
	{ E_POSTED,            "E_POSTED"          },  /* The form has already been posted. */
	{ E_CONNECTED,         "E_CONNECTED"       },  /* Field is connected. */
	{ E_BAD_STATE,         "E_BAD_STATE"       },  /* Routine was called from an initialization or termination function. */
	{ E_NO_ROOM,           "E_NO_ROOM"         },  /* Form is too large for its window. */
	{ E_NOT_POSTED,        "E_NOT_POSTED"      },  /* The form has not been posted. */
	{ E_UNKNOWN_COMMAND,   "E_UNKNOWN_COMMAND" },  /* The form driver code saw an unknown request code. */
	{ E_NO_MATCH,          "E_NO_MATCH"        },  /*  */
	{ E_NOT_SELECTABLE,    "E_NOT_SELECTABLE"  },  /*  */
	{ E_NOT_CONNECTED,     "E_NOT_CONNECTED"   },  /* No items are connected to the form. */
	{ E_REQUEST_DENIED,    "E_REQUEST_DENIED"  },  /* The form driver could not process the request. */
	{ E_INVALID_FIELD,     "E_INVALID_FIELD"   },  /* Contents of field is invalid. */
	{ E_CURRENT,           "E_CURRENT"         },  /* The field is the current field */
	{ 0,                   (char *) NULL       }};


const char *cdw_ncurses_error_string(int error)
{
	return cdw_utils_id_label_table_get_label(cdw_ncurses_error_strings, error);
}





/**
   \brief Remove given ncurses MENU and all its ITEMs

   Unpost and deallocate given ncurses MENU data and all ITEMs that belong
   to this menu.

   \param menu - menu that you want to remove
*/
void cdw_ncurses_delete_menu_and_items(MENU **menu)
{
	if (*menu != (MENU *) NULL) {
		/* order of cleaning up a menu
		   (from http://invisible-island.net/ncurses/ncurses-intro.html#moverview):

		   # Unpost the menu using unpost_menu().
		   # Free the menu, using free_menu().
		   # Free the items using free_item(). */

		unpost_menu(*menu);
		ITEM **items = menu_items(*menu);

		int rv = free_menu(*menu);
		*menu = (MENU *) NULL;
		if (rv == E_OK) {
			;
		} else if (rv == E_SYSTEM_ERROR) {
			int e = errno;
			cdw_vdm ("ERROR: failed to free menu, error = \"%s\"\n", strerror(e));
		} else if (rv == E_BAD_ARGUMENT) {
			cdw_vdm ("ERROR: failed to free menu, bad argument\n");
		} else if (rv == E_POSTED) {
			cdw_vdm ("ERROR: failed to free menu, menu is posted\n");
		} else {
			cdw_vdm ("ERROR: unrecognized error when calling free_menu: %d\n", rv);
		}

		if (items != (ITEM **) NULL) {
			for (int i = 0; items[i] != (ITEM *) NULL; i++){
				free_item(items[i]);
				items[i] = (ITEM *) NULL;
			}

			free(items);
			items = (ITEM **) NULL;
		} else {
			cdw_sdm ("WARNING: menu items were NULL for non-NULL menu\n");
		}
	}

	return;
}





static cdw_id_clabel_t cdw_ncurses_key_labels[] =
	{
		{ CDW_KEY_ENTER,  "ENTER" },
		{ CDW_KEY_ESCAPE, "ESCAPE" },
		{ KEY_DOWN,     "KEY_DOWN" },
		{ KEY_UP,       "KEY_UP" },
		{ KEY_LEFT,     "KEY_LEFT" },
		{ KEY_RIGHT,    "KEY_RIGHT" },
		{ KEY_PPAGE,    "Page Up" },
		{ KEY_NPAGE,    "Page Down" },
		{ KEY_HOME,     "KEY_HOME" },
		{ KEY_END,      "KEY_END" },
		{ ' ',          "SPACE" },
		{ '\t',         "TAB" },
		{ CDW_KEY_TAB,  "TAB" },
		{ KEY_F(1),     "F1" },
		{ KEY_F(2),     "F2" },
		{ KEY_F(3),     "F3" },
		{ KEY_F(4),     "F4" },
		{ KEY_F(5),     "F5" },
		{ KEY_F(6),     "F6" },
		{ KEY_F(7),     "F7" },
		{ KEY_F(8),     "F8" },
		{ KEY_F(9),     "F9" },
		{ KEY_F(10),    "F10" },
		{ KEY_F(11),    "F11" },
		{ KEY_F(12),    "F12" },
		{ KEY_IC,       "Insert" },
		{ 0,            (char *) NULL }};


static char cdw_ncurses_key_printable[2];

const char *cdw_ncurses_key_label(int key)
{
	if (isprint(key)) {
		snprintf(cdw_ncurses_key_printable, 1 + 1, "%c", toascii(key));
		return cdw_ncurses_key_printable;
	} else {
		return cdw_utils_id_label_table_get_label(cdw_ncurses_key_labels, key);
	}
}





/**
   \brief Create input field

   Function creates one ncurses form fiels of given size and type.
   Function takes a lot of arguments, but it returns fully configured
   and tested field.

   \param n_lines - number of lines in a field
   \param n_cols - number of columns in a field
   \param begin_y
   \param begin_x
   \param initial_string - initial content of a field, may be NULL
   \param input_type - specifies behavior or input field
   \param len_max - specifies maximal number of characters that can be entered into the field, may be zero for no limit
   \param colors - colors scheme for the field

   \return pointer to new field on success
   \return NULL on errors
*/
FIELD *cdw_ncurses_new_input_field(int n_lines, int n_cols, int begin_y, int begin_x, const char *initial_value, size_t len_max, int input_type, cdw_colors_t colors)
{
	int e = 0;
	/* input field
	   new_field params: height, width, starty, startx, number of
	   offscreen rows and number of additional working buffers */
	FIELD *field = new_field((int) n_lines, (int) n_cols, (int) begin_y, (int) begin_x, 0, 0);
	if (field == (FIELD *) NULL) {
		cdw_vdm ("ERROR: failed to create new field, new_field() returns NULL, error = \"%s\"\n",
			 cdw_ncurses_error_string(e));
		return (FIELD *) NULL;
	}

	/* NOTE: call "field_opts_off(field[0], O_STATIC)" _before_ calling
	   set_field_buffer(), otherwise string in field will be truncated
	   to field width; */

	/* data buffer can be longer than visible length of field */
	e = field_opts_off(field, O_STATIC);
	if (e != E_OK) {
		cdw_vdm ("ERROR: failed to turn off static with field_opts_off(), error = \"%s\"\n", cdw_ncurses_error_string(e));
	}
	/* either 0 (no limit to data size) or real limit */
	e = set_max_field(field, (int) len_max);
	if (e != E_OK) {
		cdw_vdm ("ERROR: failed to set proper limit=%zd for field with set_max_field(), error = \"%s\"\n", len_max, cdw_ncurses_error_string(e));
	}

	if (input_type & CDW_NCURSES_INPUT_NUMERIC) {
		cdw_vdm ("INFO: setting field type to numeric\n");
		/* require validation of field,
		   min value is 0, max is 0 (which gives no range checking) */
		e = set_field_type(field, TYPE_NUMERIC, 0, 0.0, 0.0);
		if (e != E_OK) {
			cdw_vdm ("ERROR: failed to set field type to numeric with set_field_type(), error = \"%s\"\n", cdw_ncurses_error_string(e));
		}
	} else if (input_type & CDW_NCURSES_INPUT_INTEGER) {
		cdw_vdm ("INFO: setting field type to integer\n");
		/* require validation of field,
		   min value is 0, max is 0 (which gives no range checking) */
		e = set_field_type(field, TYPE_INTEGER, 0, 0.0, 0.0);
		if (e != E_OK) {
			cdw_vdm ("ERROR: failed to set field type to integer with set_field_type(), error = \"%s\"\n", cdw_ncurses_error_string(e));
		}

	} else {
		/* no specific type */
	}

	if (input_type & CDW_NCURSES_INPUT_HIDDEN) {
		/* don't display entered chars */
		e = field_opts_off(field, O_PUBLIC);
		if (e != E_OK) {
			cdw_vdm ("ERROR: failed to turn off O_PUBLIC with field_opts_off(), error = \"%s\"\n", cdw_ncurses_error_string(e));
		}
	} else {
		if (initial_value != (char *) NULL) {
			/* this may also set content of buffer to "" */
			e = set_field_buffer(field, 0, initial_value);
			if (e != E_OK) {
				const char *tmp = (e == E_SYSTEM_ERROR) ? strerror(errno) : cdw_ncurses_error_string(e);
				cdw_assert (0, "ERROR: failed to set buffer content with string \"%s\", error = \"%s\"\n", initial_value, tmp);
			}
		}
	}

	e = field_opts_off(field, O_BLANK);
	if (e != E_OK) {
		cdw_vdm ("ERROR: failed to turn off blanking of field, error = \"%s\"\n", cdw_ncurses_error_string(e));
	}
	e = set_field_just(field, JUSTIFY_LEFT);
	if (e != E_OK) {
		cdw_vdm ("ERROR: failed to justify field, error = \"%s\"\n", cdw_ncurses_error_string(e));
	}
	/* disable moving cursor out of field when it reaches end of field */
	e = field_opts_off(field, O_AUTOSKIP);
	if (e != E_OK) {
		cdw_vdm ("ERROR: failed to turn off autoskip with field_opts_off(), error = \"%s\"\n", cdw_ncurses_error_string(e));
	}
	e = set_field_back(field, COLOR_PAIR(colors));
	if (e != E_OK) {
		const char *tmp = (e == E_SYSTEM_ERROR) ? strerror(errno) : cdw_ncurses_error_string(e);
		cdw_assert (0, "ERROR: failed to set buffer content with string \"%s\", error = \"%s\"\n", initial_value, tmp);
	}
	e = set_field_fore(field, COLOR_PAIR(colors));
	if (e != E_OK) {
		const char *tmp = (e == E_SYSTEM_ERROR) ? strerror(errno) : cdw_ncurses_error_string(e);
		cdw_assert (0, "ERROR: failed to set buffer content with string \"%s\", error = \"%s\"\n", initial_value, tmp);
	}

	return field;
}





/**
 * \brief Create new non-editable ncurses field that will be used as text label
 *
 * Create XxY input field that will be used in a page as text label.
 *
 * \return field - newly created field
 */
FIELD *cdw_ncurses_new_label_field(int n_lines, int n_cols, int begin_y, int begin_x, const char *initial_value)
{
	FIELD *field = new_field(n_lines, n_cols, begin_y, begin_x, 0, 0);
	if (!field) {
		int e = errno;
		cdw_vdm ("ERROR: failed to create a field with new_field(), \"%s\"\n", cdw_ncurses_error_string(e));
		return (FIELD *) NULL;
	}

	int rv = field_opts_on(field, O_STATIC); /* otherwise justification might fail */
	if (rv != E_OK) {
		cdw_vdm ("ERROR: field_opts_on() returns !E_OK\n");
	}
	rv = set_field_buffer(field, 0, initial_value);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: failed to set field buffer with string = \"%s\"\n", initial_value);
	}
	/* supposedly you can't justify if field is inactive, so first justify and then deactivate */
	rv = set_field_just(field, JUSTIFY_LEFT);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: set_field_just() returns !E_OK\n");
	}
	rv = field_opts_off(field, O_ACTIVE);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: field_opts_off() returns !E_OK\n");
	}
	rv = set_field_back(field, COLOR_PAIR(CDW_COLORS_DIALOG));
	if (rv != E_OK) {
		cdw_vdm ("ERROR: set_field_back() returns !E_OK\n");
	}
	rv = set_field_fore(field, COLOR_PAIR(CDW_COLORS_DIALOG));
	if (rv != E_OK) {
		cdw_vdm ("ERROR: set_field_fore() returns !E_OK\n");
	}

	return field;
}





/**
 * \brief Create new ncurses input field that will be used as checkbox
 *
 * Create 1x1 input field that will be used in a page as checkbox. Function
 * does not write '[' ']' characters around the field.
 *
 * \param begin_y - row in which the field will be put
 * \param begin_x - column in which the field will be put
 * \param buffer_content - initial state of the checkbox
 *
 * \return field - newly created field
 */
FIELD *cdw_ncurses_new_checkbox_field(int begin_y, int begin_x, const char *initial_value)
{
	/* FIELD *new_field(int height, int width, int toprow, int leftcol,
	                    int offscreen, int nbuffers); */

	FIELD *field = new_field(1, 1, begin_y, begin_x, 0, 0);
	if (field == (FIELD *) NULL) {
		cdw_vdm ("ERROR: failed to create a field with new_field\n");
		return (FIELD *) NULL;
	}
	int rv = 0;
	if (initial_value != (char *) NULL) {
		rv = set_field_buffer(field, 0, initial_value);
		if (rv != E_OK) {
			cdw_vdm ("ERROR: set_field_buffer() returns !E_OK\n");
		}
	}
	field_opts_off(field, O_AUTOSKIP);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: field_opts_off() returns !E_OK\n");
	}
	field_opts_on(field, O_STATIC);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: field_opts_on() returns !E_OK\n");
	}
	set_field_just(field, JUSTIFY_CENTER);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: set_field_just() returns !E_OK\n");
	}

	return field;
}





void cdw_ncurses_make_custom_key_definitions(void)
{
	/* new definitions of keys: a workaround for problems with
	   underlying termcap and/or terminfo; see "termcap/terminfo" note
	   in header of this file */
	struct { const char *string; int val; } keycodes[] = {
		{ "\033[11~", KEY_F(1) },
		{ "\033[12~", KEY_F(2) },
		{ "\033[13~", KEY_F(3) },
		{ "\033[14~", KEY_F(4) },
		{ "\033OP",   KEY_F(1) },
		{ "\033OQ",   KEY_F(2) },
		{ "\033OR",   KEY_F(3) },
		{ "\033OS",   KEY_F(4) },

		{ "\x7f",     KEY_BACKSPACE }, /* with xterm-color, found in hnb-1.9.14/src/ui.c */
#if 0
		/* this doesn't work */
		{ "\x0a",     KEY_ENTER },     /* 10, maybe this will replace #define in main.h */
#endif
		{ "\033OH",   KEY_HOME },      /* with xterm-color */
		{ "\033OF",   KEY_END},        /* with xterm-color */

		{ (const char *) NULL, 0 }};   /* guard */

	int i = 0;
	while (keycodes[i].val != 0) {
		int rv = define_key(keycodes[i].string, keycodes[i].val);
		if (rv != OK) {
			cdw_vdm ("WARNING: can't define key #%d / \"%s\"\n", keycodes[i].val, keycodes[i].string);
			/* not returning, missing key definitions aren't critical error */
		}
		i++;
	}

	return;
}





/**
   \brief Create new form

   Create new form, do this with proper assignment of its window, subwindow
   and form. Function does full clean up and returns NULL in case of failure.
   Otherwise pointer to properly created form is returned.

   Since "Every form has an associated pair of curses windows" (quote from man
   pages for set_form_win() and set_form_sub()), you have to create two
   windows (main window and subwindow) before calling this function.

   \param window - window in which form will be placed
   \param subwindow - subwindow of the form
   \param field - set of fields to be put in the form

   \return form on success
   \return NULL on failure
*/
FORM *cdw_ncurses_new_form(WINDOW *window, WINDOW *subwindow, FIELD *field[])
{
	cdw_assert (window != (WINDOW *) NULL, "ERROR: function called with window == NULL\n");
	cdw_assert (subwindow != (WINDOW *) NULL, "ERROR: function called with subwindow == NULL\n");
	cdw_assert (field != (FIELD **) NULL, "ERROR: function called with fields array == NULL\n");

	FORM *form = new_form(field);
	if (form == (FORM *) NULL) {
#ifndef NDEBUG
		int e = errno;
		cdw_vdm ("ERROR: failed to create form with new_form(), error code is \"%s\"\n",
			 cdw_ncurses_error_string(e));
#endif
		return (FORM *) NULL;
	}
	int r;
	r = set_form_win(form, window);
	if (r != E_OK) {
		cdw_vdm ("ERROR: failed to set form window with set_form_win(), error code is \"%s\"\n", cdw_ncurses_error_string(r));
		free_form(form);
		form = (FORM *) NULL;
		return (FORM *) NULL;
	}
	r = set_form_sub(form, subwindow);
	if (r != E_OK) {
		cdw_vdm ("ERROR: failed to set form subwindow with set_form_sub(), error code is \"%s\"\n", cdw_ncurses_error_string(r));
		free_form(form);
		form = (FORM *) NULL;
		return (FORM *) NULL;
	}
	r = post_form(form);
	if (r != E_OK) {
		cdw_vdm ("ERROR: failed to post form with post_form(), error code is \"%s\"\n", cdw_ncurses_error_string(r));
		free_form(form);
		form = (FORM *) NULL;
		return (FORM *) NULL;
	}

	return form;
}





/**
   \brief Get string from given field's buffer

   Function removes trailing white chars from te string before returning it.

   Function returns empty string ("") if given field's buffer is NULL.

   \param field - field that you want to inspect

   \return string from field's buffer, may be empty string ("")
*/
char *cdw_ncurses_get_field_string(FIELD *field)
{
	assert(field != (FIELD *) NULL);

	char *s = field_buffer(field, 0);
	if (s == (char *) NULL) {
		return empty_string;
	} else {
		cdw_vdm ("INFO: returning field string \"%s\"\n", s);
		return cdw_string_rtrim(s);
	}

}



/* *** unused code below */





# if 0



/**
   \brief Get 0/1 state from a checkbox widget associated with given field

   Check content of field's buffer (if it is "X" or not) and return 0 or 1.

   The field must be configured and used as a checkbox (field of size 1x1),
   because only first character of field's buffer is checked.

   \param field - field that you want to inspect

   \return true if given field's buffer contains "X" string
   \return false if given field's buffer does not contain "X" string
*/
bool cdw_ncurses_get_field_bit(FIELD *field)
{
	cdw_assert (field != (FIELD *) NULL, "ERROR: field argument passed for examination is NULL\n");
	return !strncmp(field_buffer(field, 0), "X", 1) ? true : false;
}

#endif
