//  Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef PROFILE_H
#define PROFILE_H

#include <gtkmm.h>
#include <string>

class XML_Helper;

//! A single entry in the profile list.
/**
 * This is an identity that the user assumes when playing a network game.
 */
class Profile
{
    public:

	//! The xml tag of this object in a profiles file.
	static std::string d_tag; 

	//! Loading constructor.
        /**
	 * Make a new profile object by reading it in from an opened profiles 
         * file.
	 *
         * @param helper  The opened profiles file to read the profile from.
         */
        Profile(XML_Helper* helper);

	//! Default constructor.
	/**
	 * Make a new profile object.
	 */
        Profile(std::string nickname);
        
        //! Copy constructor
        Profile(const Profile &orig);

	//! Destructor.
        virtual ~Profile();

	// Get Methods

        //! Get the guid of the profile.
	std::string getId() const {return d_id;};

        //! Get the name of the profile.
        std::string getNickname() const {return d_nickname;};

        //! Get the date of the last time this profile played a network game.
        Glib::TimeVal getLastPlayedOn() const {return d_last_played_date;};

        //! Get the date of when this profile was created.
        Glib::TimeVal getCreatedOn() const {return d_creation_date;};

        //! Get the user name of the profile.
        std::string getUserName() const {return d_user;};

	// Methods that operate on the class data but do not modify it.

	//! Save the profile to an opened file.
	bool save(XML_Helper* helper) const;

	//! Save the profile, but not the enclosing tags.
	bool saveContents(XML_Helper *helper) const;

        // Methods that operate ont he class and modify it.
        
        //! This profile is engaging in a game.  Record the date.
        void play();

	// Static Methods

	/**
	 * static load function (see XML_Helper)
	 * 
	 * @param helper       the XML_Helper instance for the profiles file.
	 */
	static Profile* handle_load(XML_Helper *helper);

    protected:

	// DATA
	
	//! The id of the game.
	std::string d_id;

        std::string d_nickname;

        std::string d_user;

        Glib::TimeVal d_creation_date;

        Glib::TimeVal d_last_played_date;

};

#endif // PROFILE_H
