/*
 * Copyright (C) 2014-2016 Canonical Ltd.
 *
 * This file is part of Lomiri Clock App
 *
 * Lomiri Clock App is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * Lomiri Clock App is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.12
import ClockApp.WorldClock 1.0
import ClockApp.Alarm 1.0
import Lomiri.Components 1.3
import "../components"

Page {
    id: _alarmSettings

    visible: false

    header: PageHeader {
        title: i18n.tr("Settings")
        flickable: settingsPlugin

        trailingActionBar {
            actions: [
                Action {
                    id: infoIcon
                    objectName: "infoIcon"
                    text: i18n.tr("About")
                    iconName: "info"
                    onTriggered: {
                        mainStack.push(Qt.resolvedUrl("../components/Information.qml"))
                    }
                }
            ]
        }
    }

    Connections {
        target: rootWindow
        onApplicationStateChanged: {
            localTimeSource.update()
        }
    }

    DateTime {
        id: localTimeSource
    }

    AlarmSettings {
        id: alarmSettings
    }

    AlarmSound {
        id: alarmSoundHelper
    }

    ListModel {
        id: durationModel
        Component.onCompleted: initialise()

        function initialise() {
            var durations = [1, 2, 5, 10]

            for (var i = 0; i < durations.length; ++i) {
                // TRANSLATORS: Alarm stops after
                durationModel.append({ "duration": durations[i], "text": i18n.tr("%1 minute", "%1 minutes", durations[i]).arg(durations[i]) })
            }
        }
    }

    ListModel {
        id: snoozeModel
        Component.onCompleted: initialise()

        function initialise() {
            // TRANSLATORS: Snooze for
            snoozeModel.append({ "duration": 2, "text": i18n.tr("%1 minute", "%1 minutes", 2).arg(2) })
            snoozeModel.append({ "duration": 5, "text": i18n.tr("%1 minute", "%1 minutes", 5).arg(5) })
            snoozeModel.append({ "duration": 10, "text": i18n.tr("%1 minute", "%1 minutes", 10).arg(10) })
            snoozeModel.append({ "duration": 15, "text": i18n.tr("%1 minute", "%1 minutes", 15).arg(15) })
        }
    }

    Flickable {
        id: settingsPlugin

        contentHeight: _settingsColumn.height
        anchors.fill: parent

        Column {
            id: _settingsColumn

            anchors {
                top: parent.top
                left: parent.left
                right: parent.right
            }

            

            

            
            ListItemHeader{
                 title.text: i18n.tr("Alarm Settings")
            }


            ListItem {
                height: alarmVolumeSlot.height + divider.height
                divider.visible: false

                SlotsLayout {
                    id: alarmVolumeSlot
                    mainSlot: Column {
                        Label {
                            text: i18n.tr("Alarm volume")
                        }

                        Slider {
                            width: Math.min(parent.width, units.gu(40))
                            minimumValue: 1
                            maximumValue: 100
                            live: true
                            value: alarmSettings.volume
                            function formatValue(v) { return v = "" }
                            onValueChanged: {
                                alarmSettings.volume = value
                            }
                        }
                    }
                }
            }

            ExpandableListItem {
                id: _alarmDuration

                listViewHeight: units.gu(28)
                titleText.text: i18n.tr("Alarm stops after")
                subText.text: i18n.tr("%1 minute", "%1 minutes", alarmSettings.duration).arg(alarmSettings.duration)
                subText.textSize: Label.Medium

                model: durationModel

                delegate: ListItem {
                    divider.visible: false
                    ListItemLayout {
                        title.text: model.text

                        Icon {
                            SlotsLayout.position: SlotsLayout.Trailing
                            width: units.gu(2)
                            height: width
                            name: "tick"
                            visible: alarmSettings.duration === duration
                            asynchronous: true
                        }
                    }

                    onClicked: {
                        alarmSettings.duration = duration
                        _alarmDuration.expansion.expanded = false
                    }
                }
            }

            ExpandableListItem {
                id: _alarmSnooze

                listViewHeight: units.gu(28)
                titleText.text: i18n.tr("Snooze for")
                subText.text: i18n.tr("%1 minute", "%1 minutes", alarmSettings.snoozeDuration).arg(alarmSettings.snoozeDuration)
                subText.textSize: Label.Medium

                model: snoozeModel

                delegate: ListItem {
                    divider.visible: false
                    ListItemLayout {
                        title.text: model.text

                        Icon {
                            SlotsLayout.position: SlotsLayout.Trailing
                            width: units.gu(2)
                            height: width
                            name: "tick"
                            visible: alarmSettings.snoozeDuration === duration
                            asynchronous: true
                        }
                    }

                    onClicked: {
                        alarmSettings.snoozeDuration = duration
                        _alarmSnooze.expansion.expanded = false
                    }
                }
            }

             ListItem {
                height: defaultSoundLayout.height + divider.height
                divider.visible: false
                ListItemLayout {
                    id: defaultSoundLayout
                    title.text: i18n.tr("Default Alarm Sound")
                    subtitle.text : clockAppSettings.defaultAlarmSound ? alarmSoundHelper.getSoundName(clockAppSettings.defaultAlarmSound.toString()) : "Alarm clock"

                     Icon {
                        SlotsLayout.position: SlotsLayout.Trailing
                        width: units.gu(2)
                        height: width
                        name: "go-next"
                        asynchronous: true
                    }
                }
                onClicked:  {
                    pageStack.push(Qt.resolvedUrl("AlarmSound.qml"), {
                        "alarmSoundName": alarmSoundHelper.getSoundName(clockAppSettings.defaultAlarmSound),
                        "alarmUrl": clockAppSettings.defaultAlarmSound,
                        "defaultAlarmSound":  "Suru arpeggio.ogg"
                    });
                    pageStack.currentPage.setAlarm.connect(setDefaultAlarm);
                }

                function setDefaultAlarm(newAlarmUrl) {
                    clockAppSettings.defaultAlarmSound = newAlarmUrl;
                    defaultSoundLayout.subtitle.text = alarmSoundHelper.getSoundName(clockAppSettings.defaultAlarmSound.toString())
                }
            }

            ListItem {
                height: vibrationLayout.height + divider.height
                divider.visible: true
                ListItemLayout {
                    id: vibrationLayout
                    title.text: i18n.tr("Vibration")

                    Switch {
                        id: vibrateSwitch
                        SlotsLayout.position: SlotsLayout.Trailing
                        checked: alarmSettings.vibration === "pulse"
                        onCheckedChanged: {
                            if(checked) {
                                alarmSettings.vibration = "pulse"
                            } else {
                                alarmSettings.vibration = "none"
                            }
                        }
                    }
                }

                onClicked: {
                    vibrateSwitch.checked = !vibrateSwitch.checked
                }
            }

            ListItem {
                height: timeAndDateLayout.height + divider.height
                divider.visible: false
                ListItemLayout {
                    id: timeAndDateLayout

                    title.text: i18n.tr("Change time and date")
                    subtitle.text: localTimeSource.localizedCurrentDateString + " " + localTimeSource.localizedCurrentTimeString
                    subtitle.textSize: Label.Medium

                    Icon {
                        SlotsLayout.position: SlotsLayout.Trailing
                        width: units.gu(2)
                        height: width
                        name: "go-next"
                        asynchronous: true
                    }
                }

                onClicked: {
                    Qt.openUrlExternally("settings:///system/time-date")
                }
            }





            ExpandableListItem {
                id: _selectedTheme

                property QtObject selectedItem: null

                listViewHeight: units.gu(6) + (model.count * units.gu(5))
                titleText.text: i18n.tr("Theme")
                subText.textSize: Label.Medium

                //TODO This list should be retrived form the system/UITK but I couldn't find a way to do that elegently
                //     so it`s currently hard coded with the themes that Ubuntu SDK docs says are available by default.
                model: ListModel {
                        ListElement {name: "System Theme"; value : "" }
                        ListElement {name: "Ambiance"; value : "Lomiri.Components.Themes.Ambiance"}
                        ListElement {name: "Suru Dark"; value : "Lomiri.Components.Themes.SuruDark"}
                }

                onSelectedItemChanged:  {
                    subText.text = selectedItem.name;
                }

                function updateSelectedItem(itemValue) {
                    for(var i=0; i < model.count;i++) {
                        if(model.get(i).value === itemValue) {
                            selectedItem = model.get(i);
                            break;
                        }
                    }
                }

                Timer {
                   id: theme_timer
                   interval: 200; running: false; repeat: false;
                   onTriggered: {
                      clockAppSettings.theme = _selectedTheme.selectedItem.value
                      clockApp.theme.name = clockAppSettings.theme
                      _selectedTheme.isActivityVisible = false
                      _selectedTheme.isActivityRunning = false
                   }
                }

                Component.onCompleted: updateSelectedItem(clockAppSettings.theme);

                delegate: ListItem {
                    divider.visible: false
                    ListItemLayout {
                        title.text: model.name

                        Icon {
                            SlotsLayout.position: SlotsLayout.Trailing
                            width: units.gu(2)
                            height: width
                            name: "tick"
                            visible: model.value == clockAppSettings.theme
                            asynchronous: true
                        }
                    }

                    onClicked: {
                        _selectedTheme.updateSelectedItem(model.value)
                        _selectedTheme.expansion.expanded = false
                        _selectedTheme.isActivityRunning = true
                        _selectedTheme.isActivityVisible = true
                        theme_timer.start()
                    }
                }
            }

            
        }
    }
}
