//===-- Double-precision log10(x) function --------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#include "src/math/log10.h"
#include "src/__support/FPUtil/FEnvImpl.h"
#include "src/__support/FPUtil/FPBits.h"
#include "src/__support/FPUtil/PolyEval.h"
#include "src/__support/FPUtil/double_double.h"
#include "src/__support/FPUtil/dyadic_float.h"
#include "src/__support/FPUtil/multiply_add.h"
#include "src/__support/common.h"
#include "src/__support/macros/optimization.h" // LIBC_UNLIKELY

#include "common_constants.h"
#include "log_range_reduction.h"

namespace __llvm_libc {

// 128-bit precision dyadic floating point numbers.
using Float128 = typename fputil::DyadicFloat<128>;
using MType = typename Float128::MantissaType;

namespace {

constexpr fputil::DoubleDouble LOG10_E = {0x1.95355baaafad3p-57,
                                          0x1.bcb7b1526e50ep-2};

// A simple upper bound for the error of e_x * log(2) - log(r).
constexpr double HI_ERR = 0x1.0p-85;

// Extra errors from P is from using x^2 to reduce evaluation latency.
constexpr double P_ERR = 0x1.0p-51;

// log10(2) with 128-bit prepcision generated by SageMath with:
//   sage: (s, m, e) = RealField(128)(2).log10().sign_exponent_mantissa();
//   sage: print("MType({", hex(m % 2^64), ",", hex((m >> 64) % 2^64), "})");
const Float128 LOG10_2(/*sign=*/false, /*exponent=*/-129, /*mantissa=*/
                       MType({0x8f8959ac0b7c9178, 0x9a209a84fbcff798}));

const LogRR LOG10_TABLE = {
    // -log10(r) with 128-bit precision generated by SageMath with:
    //
    // for i in range(128):
    //   r = 2^-8 * ceil( 2^8 * (1 - 2^(-8)) / (1 + i*2^(-7)) );
    //   s, m, e = RealField(128)(r).log10().sign_mantissa_exponent();
    //   print("{false,", e, ", MType({", hex(m % 2^64), ",", hex((m >> 64) %
    //   2^64),
    //         "})},");
    /* .step_1 = */ {
        {false, 0, MType(0)},
        {false, -136, MType({0x65af394fe05eafd3, 0xdf3b5ebbda7e186b})},
        {false, -135, MType({0xa8fb8d87b30163b5, 0xe01d40572f029c16})},
        {false, -134, MType({0x6bb0170e5bb5d630, 0xa8c1263ac3f57eb3})},
        {false, -134, MType({0xfc2ea6eb0ea1370e, 0xe1e841bbc26204e5})},
        {false, -133, MType({0xdc8a199a4bb63382, 0x8dc2eb02274d6ff4})},
        {false, -133, MType({0x86b57ea610c7db33, 0xaacde920361dd054})},
        {false, -133, MType({0x5f034a40e6a2f09d, 0xc81618eb15421bab})},
        {false, -133, MType({0x594a31b2c5cc891c, 0xe59c7e66c5fedb4b})},
        {false, -133, MType({0x221efda58221904b, 0xf477584f97b654de})},
        {false, -132, MType({0x68a0dc47567691c9, 0x892e821975106e09})},
        {false, -132, MType({0x10bc94f44d216b49, 0x9841c66e17dfe7da})},
        {false, -132, MType({0xe303ea7e23c9d6fb, 0x9fd7be3318306cc5})},
        {false, -132, MType({0xce697dbaa00d4c7d, 0xaf1cb35bf494a8dd})},
        {false, -132, MType({0x9c216079dcf0ea96, 0xbe8380a2fa7eba5a})},
        {false, -132, MType({0x75278940eecfc3a9, 0xc643c7758283a271})},
        {false, -132, MType({0x2d3467d253e2d1fc, 0xd5de75ec27e4fe68})},
        {false, -132, MType({0xead4055dcdec7b22, 0xddb904e8f1272a95})},
        {false, -132, MType({0xe1e0dda0b3d375a4, 0xed88f6bb355fa196})},
        {false, -132, MType({0x38dc40c4fe11e608, 0xf57e8281ade9d92d})},
        {false, -131, MType({0x3bcdcfe7b23976cd, 0x82c2941bb20bbe1f})},
        {false, -131, MType({0x456350b0bda452a6, 0x86cb36632807cdcd})},
        {false, -131, MType({0x78185dcc37fda01a, 0x8eeaa306458b760a})},
        {false, -131, MType({0x307643adbbbde1b3, 0x9301839512fc1168})},
        {false, -131, MType({0x6c449d409f883fe3, 0x9b3dd1d550c41443})},
        {false, -131, MType({0x8ea7b30c8b4ad886, 0x9f6356aa03c34389})},
        {false, -131, MType({0x961c6e690d8879b4, 0xa7bd56cdde5d76a2})},
        {false, -131, MType({0x42643ced81ec14a, 0xabf1ea3e1d7bd7cf})},
        {false, -131, MType({0x4742fb3d0b5cdd19, 0xb02b9af74c2f879e})},
        {false, -131, MType({0xf7e2ab36f09e9014, 0xb8ae8671b3d7dd6c})},
        {false, -131, MType({0x8d3fc63485e7ff13, 0xbcf7dabd87c01afc})},
        {false, -131, MType({0xf3edc49375fbc5a5, 0xc1467f694d10a581})},
        {false, -131, MType({0x5fcd7d0ce937375f, 0xc9f3ef07e1f3fc5e})},
        {false, -131, MType({0x58252dada9f06111, 0xce52d50b94fa253a})},
        {false, -131, MType({0x62f01e5ff43708ab, 0xd2b74192fae43777})},
        {false, -131, MType({0x481d9b3131f52639, 0xd72142a84ca85abd})},
        {false, -131, MType({0xb305ced1419fe924, 0xdb90e68b8abf14af})},
        {false, -131, MType({0x849266a85513dc6d, 0xe48150cf32888b9c})},
        {false, -131, MType({0x80ecf3266b4dcf4, 0xe90234c65a15e533})},
        {false, -131, MType({0xe1e0dda0b3d375a4, 0xed88f6bb355fa196})},
        {false, -131, MType({0xce3537a3a211b25b, 0xf215a60b6557943f})},
        {false, -131, MType({0x5dab68307fedefcd, 0xf6a852513757dfbd})},
        {false, -131, MType({0x1be2585c279c50a5, 0xffdfe15de3c01bac})},
        {false, -130, MType({0x18aa302171017dcb, 0x8242724a155219f3})},
        {false, -130, MType({0xabc7e698502d43c0, 0x849812d0ccbb5cbd})},
        {false, -130, MType({0xc339089a51663370, 0x86f0dab1ab5822b6})},
        {false, -130, MType({0x26f70b34ce5cf201, 0x894cd27d9f182c63})},
        {false, -130, MType({0x676f20a87ab433df, 0x8bac02e8ac3e09ac})},
        {false, -130, MType({0x6db4169cc4b83bc3, 0x8e0e74caae062e24})},
        {false, -130, MType({0xcd3fdb2fad0d1fd6, 0x907431201c7f651a})},
        {false, -130, MType({0x49d03e163250d1d4, 0x92dd410ad7bfe103})},
        {false, -130, MType({0x9ec7dc02d5e723b9, 0x9549add2f8a3c7e0})},
        {false, -130, MType({0x34698d03a5442573, 0x97b980e7a743d71c})},
        {false, -130, MType({0x522904d1e47f3de, 0x9a2cc3dff7548556})},
        {false, -130, MType({0x791a72646c87b976, 0x9ca3807bca9fe93f})},
        {false, -130, MType({0x3826f190d655d736, 0x9f1dc0a4b9cea286})},
        {false, -130, MType({0x544ab3e48199b299, 0xa19b8e6f03b60e45})},
        {false, -130, MType({0xbe775fa82961114e, 0xa41cf41a83643487})},
        {false, -130, MType({0x45798e5019e6c082, 0xa6a1fc13ad241953})},
        {false, -130, MType({0x91fb1ed0cdc4d1fb, 0xa92ab0f492b772bd})},
        {false, -130, MType({0x818b8b9cbbd17b72, 0xabb71d85ef05380d})},
        {false, -130, MType({0xa50c2fea60c5b3b2, 0xae474cc0397f0d4f})},
        {false, -130, MType({0x58ea34980ad8b720, 0xb0db49ccc1823c8e})},
        {false, -130, MType({0x4b5f71941be508a4, 0xb3732006d1fbbba5})},
        {false, -130, MType({0x9e405fb8bcb1ff1e, 0xb60edafcdd99ad1d})},
        {false, -130, MType({0x9e405fb8bcb1ff1e, 0xb60edafcdd99ad1d})},
        {false, -130, MType({0xf7e2ab36f09e9014, 0xb8ae8671b3d7dd6c})},
        {false, -130, MType({0xc669639640c305bb, 0xbb522e5dbf37f63b})},
        {false, -130, MType({0xa3dc9e464e98764b, 0xbdf9def04cf980ff})},
        {false, -130, MType({0xffd3256b59fa9c59, 0xc0a5a490dea95b5e})},
        {false, -130, MType({0xb0a2d48672a051a5, 0xc3558be085e3f4bc})},
        {false, -130, MType({0xb0a2d48672a051a5, 0xc3558be085e3f4bc})},
        {false, -130, MType({0xacb2ca5d4ca1c10e, 0xc609a1bb4aa98f59})},
        {false, -130, MType({0x43690b9e3cde0d02, 0xc8c1f3399ca7d33b})},
        {false, -130, MType({0x18b1fd60383f7e5a, 0xcb7e8db1cfe04827})},
        {false, -130, MType({0x248757e5f45af3d, 0xce3f7eb9a517c969})},
        {false, -130, MType({0x7c4acd605be48bc1, 0xd104d427de7fbcc4})},
        {false, -130, MType({0x7c4acd605be48bc1, 0xd104d427de7fbcc4})},
        {false, -130, MType({0x58ff63629a92652d, 0xd3ce9c15e10ec927})},
        {false, -130, MType({0x6b49be3bd8c89f10, 0xd69ce4e16303fcdd})},
        {false, -130, MType({0xe6dd603a881e9060, 0xd96fbd2e2814c9cc})},
        {false, -130, MType({0xe6dd603a881e9060, 0xd96fbd2e2814c9cc})},
        {false, -130, MType({0x89e281c98c1d705c, 0xdc4733e7cbcbfc8c})},
        {false, -130, MType({0xdc0db7cf0cce9f32, 0xdf2358439aa5dd12})},
        {false, -130, MType({0xfdf1c5b846db9deb, 0xe20439c27a7c01b8})},
        {false, -130, MType({0xfdf1c5b846db9deb, 0xe20439c27a7c01b8})},
        {false, -130, MType({0x3dd7eab48869c402, 0xe4e9e832e2da0c05})},
        {false, -130, MType({0x4e8fcc900b41daef, 0xe7d473b2e5db8f2a})},
        {false, -130, MType({0x4e8fcc900b41daef, 0xe7d473b2e5db8f2a})},
        {false, -130, MType({0x7593e1a9e917359a, 0xeac3ecb24a3ac7b4})},
        {false, -130, MType({0xe7741396b49e1ce5, 0xedb863f4b73f982d})},
        {false, -130, MType({0xe7741396b49e1ce5, 0xedb863f4b73f982d})},
        {false, -130, MType({0xc8ba4f8f47b85a5c, 0xf0b1ea93f34675a7})},
        {false, -130, MType({0x7007c1276821b705, 0xf3b09202359f9787})},
        {false, -130, MType({0x7007c1276821b705, 0xf3b09202359f9787})},
        {false, -130, MType({0x7ee19afe6db7e324, 0xf6b46c0c8c8fdea1})},
        {false, -130, MType({0xedf54f37f6d40420, 0xf9bd8add584687f0})},
        {false, -130, MType({0xedf54f37f6d40420, 0xf9bd8add584687f0})},
        {false, -130, MType({0xefe52ccf03e7dee1, 0xfccc00fedba4e6fb})},
        {false, -130, MType({0x1be2585c279c50a5, 0xffdfe15de3c01bac})},
        {false, -130, MType({0x1be2585c279c50a5, 0xffdfe15de3c01bac})},
        {false, -129, MType({0xe0b571f5c91b0446, 0x817c9fa643880404})},
        {false, -129, MType({0x7178594bef2def59, 0x830c17427ea55eca})},
        {false, -129, MType({0x7178594bef2def59, 0x830c17427ea55eca})},
        {false, -129, MType({0x9a741bb171158d2a, 0x849e6196487c1d1c})},
        {false, -129, MType({0x9a741bb171158d2a, 0x849e6196487c1d1c})},
        {false, -129, MType({0x1a618264446cb495, 0x863388eb55ebd295})},
        {false, -129, MType({0x71dbdbbec51d7657, 0x87cb97c3ff9eac18})},
        {false, -129, MType({0x71dbdbbec51d7657, 0x87cb97c3ff9eac18})},
        {false, -129, MType({0xabe0b522230f7d14, 0x896698dce4cff76c})},
        {false, -129, MType({0xabe0b522230f7d14, 0x896698dce4cff76c})},
        {false, -129, MType({0xd28e8adafea703b4, 0x8b04972e9d4d3011})},
        {false, -129, MType({0x208422d83be34b27, 0x8ca59def7b5cefc5})},
        {false, -129, MType({0x208422d83be34b27, 0x8ca59def7b5cefc5})},
        {false, -129, MType({0xc385cf49402af0e4, 0x8e49b8955e3ffb8a})},
        {false, -129, MType({0xc385cf49402af0e4, 0x8e49b8955e3ffb8a})},
        {false, -129, MType({0xda982a614e12c6dd, 0x8ff0f2d7960a075c})},
        {false, -129, MType({0xda982a614e12c6dd, 0x8ff0f2d7960a075c})},
        {false, -129, MType({0x38401fc1c1b5c2c, 0x919b58b0d999bbc8})},
        {false, -129, MType({0x38401fc1c1b5c2c, 0x919b58b0d999bbc8})},
        {false, -129, MType({0xa9b55d3f16da746a, 0x9348f6614f821394})},
        {false, -129, MType({0xa9b55d3f16da746a, 0x9348f6614f821394})},
        {false, -129, MType({0x88d2d1473d4f7f5, 0x94f9d870aac256a5})},
        {false, -129, MType({0x88d2d1473d4f7f5, 0x94f9d870aac256a5})},
        {false, -129, MType({0x7c1e117dea19e9e6, 0x96ae0bb05c35d5bd})},
        {false, -129, MType({0x7c1e117dea19e9e6, 0x96ae0bb05c35d5bd})},
        {false, -129, MType({0x336db0630f536fb9, 0x98659d3dd9b12532})},
        {false, 0, MType(0)},
    },
    // -log10(r) for the second step, generated by SageMath with:
    //
    // for i in range(-2^6, 2^7 + 1):
    //   r = 2^-16 * round( 2^16 / (1 + i*2^(-14)) );
    //   s, m, e = RealField(128)(r).log10().sign_mantissa_exponent();
    //   print("{false," if s == -1 else "{true,", e, ",
    //         MType({", hex(m % 2^64), ",", hex((m >> 64) % 2^64), "})},");
    /* .step_2 = */
    {
        {true, -137, MType({0x7f1ce002fa34131b, 0xdeca729013cd7c31})},
        {true, -137, MType({0x639afa085dd8b4c7, 0xdb5475b44946d986})},
        {true, -137, MType({0x5512632fe9a58cb, 0xd7de6b0e10cab7d2})},
        {true, -137, MType({0xb5380a9953117d07, 0xd468529cfc6fb395})},
        {true, -137, MType({0x70af2d7d53be1f31, 0xd0f22c609e474741})},
        {true, -137, MType({0xccd499c49b74cc2, 0xcd7bf858885dcae2})},
        {true, -137, MType({0x5b51ddc3987ebfb8, 0xca05b6844cba73cf})},
        {true, -137, MType({0x49375f5189b3782b, 0xc68f66e37d5f545a})},
        {true, -137, MType({0xf6e57738865c712f, 0xc3190975ac495b7a})},
        {true, -137, MType({0xca02b10a8c712acd, 0xbfa29e3a6b70547e})},
        {true, -137, MType({0x78e5038210208151, 0xbc2c25314cc6e6b6})},
        {true, -137, MType({0xfa099ecd71ee0ea, 0xb8b59e59e23a9524})},
        {true, -137, MType({0xeeb445ccb8fb09ed, 0xb53f09b3bdb3be28})},
        {true, -137, MType({0xc352fff18a1c02fb, 0xb1c8673e71159b33})},
        {true, -137, MType({0x7949e03ecf9b390b, 0xae51b6f98e3e406e})},
        {true, -137, MType({0x2681f33f30aadedc, 0xaadaf8e4a7069c6c})},
        {true, -137, MType({0xf01d5496eea213b3, 0xa7642cff4d4277d6})},
        {true, -137, MType({0xe92ef555ff1de975, 0xa3ed534912c0751d})},
        {true, -137, MType({0xeb0c7519b3e7c1e0, 0xa0766bc1894a1022})},
        {true, -137, MType({0xf60d204ff0fe5296, 0x9c21b6e91e7f03a3})},
        {true, -137, MType({0x125c19a4f057c18b, 0x98aab0491050bea8})},
        {true, -137, MType({0x7e9383ce1bdf9575, 0x95339bd64cd953e7})},
        {true, -137, MType({0xbf274f4d8f770253, 0x91bc799065cc57d6})},
        {true, -137, MType({0x656bd9b758fe44ba, 0x8e454976ecd836ad})},
        {true, -137, MType({0xbfdd2c7f388fc014, 0x8ace0b8973a63413})},
        {true, -137, MType({0x83fbf6ed936c493a, 0x8756bfc78bda6ad0})},
        {true, -137, MType({0x71bfa9a18bec01cc, 0x83df6630c713cc76})},
        {true, -137, MType({0xf09d19f56dbfef72, 0x8067fec4b6ec2111})},
        {true, -138, MType({0x4c422713b1642228, 0xf9e11305d9f00dad})},
        {true, -138, MType({0xc3c7c5699b7a0a4, 0xf2f20cd5f58de39a})},
        {true, -138, MType({0xb8db7c69e3fa0797, 0xec02eaf8e3c656ff})},
        {true, -138, MType({0xa083eb05506ff7ed, 0xe513ad6dc7a3a553})},
        {true, -138, MType({0xc21595e745f1fa15, 0xde245433c425b5c5})},
        {true, -138, MType({0xb9d5bcdbfe719389, 0xd734df49fc42189b})},
        {true, -138, MType({0xa17a1e85e93461f4, 0xd0454eaf92e4068b})},
        {true, -138, MType({0xe3537584da333fda, 0xc955a263aaec6016})},
        {true, -138, MType({0x963177f24682c2, 0xc265da656731ace5})},
        {true, -138, MType({0x4ac037347bcfc50e, 0xbb75f6b3ea801b1e})},
        {true, -138, MType({0x901a736a4364cdfd, 0xb485f74e57997ec6})},
        {true, -138, MType({0xbb550acc3b9d7247, 0xad95dc33d1355117})},
        {true, -138, MType({0x663cf2b27e8f1ffb, 0xa6a5a5637a00afdc})},
        {true, -138, MType({0x5f89bd08feb39952, 0x9fb552dc749e5cca})},
        {true, -138, MType({0x23c2623c73f494db, 0x98c4e49de3a6bcdd})},
        {true, -138, MType({0x4937d3b5485af61e, 0x91d45aa6e9a7d7b0})},
        {true, -138, MType({0xdf14214e7a6d8111, 0x8ae3b4f6a92556d9})},
        {true, -138, MType({0xbf7cfc14999fb4bc, 0x83f2f38c44988544})},
        {true, -139, MType({0xa990c0ee569a8d51, 0xfa042ccdbce09d15})},
        {true, -139, MType({0xa38463e9d941e1c2, 0xec223b0b32227c9e})},
        {true, -139, MType({0xba0324530edaa03f, 0xde4011cf2daaff31})},
        {true, -139, MType({0x5e997a02dad7ace7, 0xd05db117f419b857})},
        {true, -139, MType({0x4a14676d4d0f817e, 0xc27b18e3c9f977c7})},
        {true, -139, MType({0x857c002ee7a1e473, 0xb4984930f3c0481c})},
        {true, -139, MType({0x5923b2eb72d8012a, 0xa6b541fdb5cf6d89})},
        {true, -139, MType({0x21cde8f85ca1f9fd, 0x98d203485473648b})},
        {true, -139, MType({0xbe08e08b1d212d4, 0x8aee8d0f13e3e09e})},
        {true, -140, MType({0x695023998e6bd7b0, 0xfa15bea0708795e1})},
        {true, -140, MType({0x634cea6750617a92, 0xde4df4140b42822f})},
        {true, -140, MType({0xfbd7e970aef9dbb8, 0xc285ba757feb2781})},
        {true, -140, MType({0x9aedc1c1ba7d0695, 0xa6bd11c1564a8ace})},
        {true, -140, MType({0x8d306ba207233c44, 0x8af3f9f41600120a})},
        {true, -141, MType({0x856a0a3a00fcf3c1, 0xde54e6148d030322})},
        {true, -141, MType({0xb3a2c1407cf6d38d, 0xa6c0fa00de35f314})},
        {true, -142, MType({0xd791cf6a70c3a504, 0xde585f4c5bbbcd3d})},
        {true, -143, MType({0x10a633f2c4a8ea22, 0xde5a1bf627b1f68f})},
        {true, 0, MType({0x0, 0x0})},
        {false, -143, MType({0xed4a68e5e6e83ddf, 0xde5d95658a729eab})},
        {false, -142, MType({0x3281f1872cdbee94, 0xde5f522b21e3e25a})},
        {false, -141, MType({0xf1466edaa96e356e, 0xa6c8cb3b7e5bbbfd})},
        {false, -141, MType({0x8a607fd695dfc3d9, 0xde62cbd21e895473})},
        {false, -140, MType({0xc36b8713ceefe2de, 0x8afed57032bebc7c})},
        {false, -140, MType({0x5c2e76c953e3e3e6, 0xa6ccb436a3c72fa4})},
        {false, -140, MType({0x8e4950fa5c943bbf, 0xc29b023fdcb2dccf})},
        {false, -140, MType({0x20fa8a73c585f634, 0xde69bf8f58005dfc})},
        {false, -140, MType({0xaa106d9b0a9717a, 0xfa38ec28905810a3})},
        {false, -139, MType({0x85d70e032de41aec, 0x8b04440780460c2a})},
        {false, -139, MType({0xbeee21cbb82a9a78, 0x98ec49a311cc30ab})},
        {false, -139, MType({0xabd7b0fdd8efe6f6, 0xa6d486e8ba5151a0})},
        {false, -139, MType({0x3221c56e2c1aa912, 0xb4bcfbda377d31cc})},
        {false, -139, MType({0x57b795a36d9c5f19, 0xc2a5a879470c7c37})},
        {false, -139, MType({0x131ec142c053ac3b, 0xd08e8cc7a6d0c580})},
        {false, -139, MType({0x35e3298f4bb2aa0a, 0xde77a8c714b08d28})},
        {false, -139, MType({0x7133dafdfc44f160, 0xec60fc794ea73ee4})},
        {false, -139, MType({0x74b37d23121c59d5, 0xfa4a87e012c533eb})},
        {false, -138, MType({0x93bf5f4207da8a4c, 0x841a257e8f97da22})},
        {false, -138, MType({0xfdb5990ec6057f4e, 0x8b0f22e919107c0c})},
        {false, -138, MType({0x2d408a58b1b202fe, 0x92043c3084f41481})},
        {false, -138, MType({0x1759381b61dfbf01, 0x98f97155b274b1ab})},
        {false, -138, MType({0x41e90a054df4b9f1, 0x9feec25980cedbbe})},
        {false, -138, MType({0xa1e66c6203725d50, 0xa6e42f3ccf49959d})},
        {false, -138, MType({0x8693d36ab45bd7ce, 0xadd9b8007d365d83})},
        {false, -138, MType({0x91e25bb40ad3f098, 0xb4cf5ca569f12da9})},
        {false, -138, MType({0xbdf94392c4cc7f6c, 0xbbc51d2c74e07cf0})},
        {false, -138, MType({0x6fe37973354a82f9, 0xc2baf9967d753f89})},
        {false, -138, MType({0x97647b4267bfd801, 0xc9b0f1e4632ae79b})},
        {false, -138, MType({0xdbf5c32a454f7bdf, 0xd0a70617058765ee})},
        {false, -138, MType({0xd6edfe04c37ba916, 0xd79d362f441b2a92})},
        {false, -138, MType({0x5ad3480ccfbe9890, 0xde93822dfe812587})},
        {false, -138, MType({0xc7d9ac765be7e325, 0xe589ea14145ec764})},
        {false, -138, MType({0x6d8f24b9a3ca011b, 0xec806de265640204})},
        {false, -138, MType({0xf9b654807dcdd5b2, 0xf3770d99d14b4928})},
        {false, -138, MType({0xf4513f4745663028, 0xfa6dc93b37d99326})},
        {false, -137, MType({0xa46e9a72d80da75f, 0x80b25063bc6f2cc6})},
        {false, -137, MType({0xee60992b51ffac4b, 0x842dca1fba19cce6})},
        {false, -137, MType({0x1977fa1c786886b3, 0x87a951d204deeaf3})},
        {false, -137, MType({0xe5f7c52cdf119d5, 0x8b24e77b0cb60a84})},
        {false, -137, MType({0x3bf9d70da1021a10, 0x8ea08b1b419bf221})},
        {false, -137, MType({0xfd0406b07523b8e6, 0x921c3cb31392ab94})},
        {false, -137, MType({0x453ee32c020f2a8, 0x9597fc42f2a18441})},
        {false, -137, MType({0xcfb3ec22066bf7f6, 0x9913c9cb4ed50d72})},
        {false, -137, MType({0x215c025bd493ecf9, 0x9c8fa54c983f1cb8})},
        {false, -137, MType({0x39c116b7ee3a83ec, 0x9f2c93192e68232b})},
        {false, -137, MType({0xf41f4b3ede2782f0, 0xa2a8870f24ac5f66})},
        {false, -137, MType({0x61196927723eb75c, 0xa62488ff3c735799})},
        {false, -137, MType({0xe615e836cb1edab, 0xa9a098e9e5e2a432})},
        {false, -137, MType({0x6981331c5fc71cfc, 0xad1cb6cf91252372})},
        {false, -137, MType({0x5f6a4faa054f11fa, 0xb098e2b0ae6af9c2})},
        {false, -137, MType({0x2a68bc681a74c28, 0xb4151c8dade99205})},
        {false, -137, MType({0x382ba24d90566403, 0xb7916466ffdb9ded})},
        {false, -137, MType({0x6ad1abe51dd22e00, 0xbb0dba3d14811652})},
        {false, -137, MType({0x456d3f7f59b13960, 0xbe8a1e105c1f3b85})},
        {false, -137, MType({0x738dd8b7d66e9058, 0xc2068fe1470095a4})},
        {false, -137, MType({0x68e123fed7ff11c6, 0xc5830fb04574f4f1})},
        {false, -137, MType({0x2f3bd09780c3aa11, 0xc8ff9d7dc7d17225})},
        {false, -137, MType({0x3b48887f1ce36935, 0xcc7c394a3e706ec5})},
        {false, -137, MType({0x47ddae655ecc4633, 0xcff8e31619b19578})},
        {false, -137, MType({0x37fa81eef4819c88, 0xd3759ae1c9f9da5b})},
        {false, -137, MType({0xff6c4a8d747c65ed, 0xd6f260adbfb37b55})},
        {false, -137, MType({0x921c29493a33318c, 0xda6f347a6b4e0070})},
        {false, -137, MType({0xda0631eb65e731d8, 0xddec16483d3e3c27})},
        {false, -137, MType({0xb3da6c07d110babc, 0xe1690617a5fe4bc2})},
        {false, -137, MType({0xf2485c7868b8835a, 0xe4e603e9160d97a6})},
        {false, -137, MType({0x67f5b7ed01344055, 0xe8630fbcfdf0d3ae})},
        {false, -137, MType({0xf820df445b1d0622, 0xebe02993ce31ff7b})},
        {false, -137, MType({0xadefc674b7eca5cd, 0xef5d516df76066d0})},
        {false, -137, MType({0xda6be6dc057d3235, 0xf2da874bea10a1e0})},
        {false, -137, MType({0x392bdde152ab5ff5, 0xf657cb2e16dc95a9})},
        {false, -137, MType({0x1bab58e2ec99cf73, 0xf9d51d14ee637444})},
        {false, -137, MType({0x9b51ef7e3388d692, 0xfd527d00e149bd3e})},
        {false, -136, MType({0xe914c6a7f3f22fa2, 0x8067f579301c9ef6})},
        {false, -136, MType({0xd22862eb2081c94, 0x8226b374edf088e2})},
        {false, -136, MType({0x29ebd0b476cd8fd8, 0x83e57873e27ad153})},
        {false, -136, MType({0x98feddc2806d01ed, 0x85a44476461854a0})},
        {false, -136, MType({0x471bfc261a401854, 0x8763177c512896af})},
        {false, -136, MType({0xb6f89c19b4cd1acd, 0x88b23a5b61430a16})},
        {false, -136, MType({0xb39aaf34163fb099, 0x8a7119a85909ebe9})},
        {false, -136, MType({0x1665f0f821541c36, 0x8c2ffff99357e887})},
        {false, -136, MType({0xa5051754e049c1cb, 0x8deeed4f489679a6})},
        {false, -136, MType({0x8c5a9a1c57b2e986, 0x8fade1a9b131c159})},
        {false, -136, MType({0x1d8448438a26a9ae, 0x916cdd0905988a35})},
        {false, -136, MType({0x8e3a0913ecd2fd02, 0x932bdf6d7e3c477d})},
        {false, -136, MType({0xbc881a45f47f1d36, 0x94eae8d753911550})},
        {false, -136, MType({0xf5e51c05499b06d0, 0x96a9f946be0db8d0})},
        {false, -136, MType({0xc1a43be81a243fde, 0x986910bbf62ba04f})},
        {false, -136, MType({0xaec3cfebe971beb7, 0x9a282f373466e378})},
        {false, -136, MType({0x2518b29328614989, 0x9be754b8b13e437c})},
        {false, -136, MType({0x39d6b147cbe803a4, 0x9da68140a5332b3a})},
        {false, -136, MType({0x87765e3004ae428d, 0x9f65b4cf48c9af6d})},
        {false, -136, MType({0x8f896ab28245bac, 0xa124ef64d4888ed6})},
        {false, -136, MType({0xf8880fb5ca630c87, 0xa2e4310180f93263})},
        {false, -136, MType({0xb179397cf82e935c, 0xa4a379a586a7ad62})},
        {false, -136, MType({0x95a8cb717197ad81, 0xa662c9511e22bda3})},
        {false, -136, MType({0xf6394a34b7f9a4a4, 0xa82220047ffbcba8})},
        {false, -136, MType({0xffafd8c2b57884e8, 0xa9e17dbfe4c6ead0})},
        {false, -136, MType({0xa970a643b8a6ac2b, 0xaba0e283851ad980})},
        {false, -136, MType({0xa89b49fb749d47e0, 0xad604e4f9991014e})},
        {false, -136, MType({0x66475ed2ac983305, 0xaf1fc1245ac5772e})},
        {false, -136, MType({0xb4fd6209364bb36f, 0xb06f5be1bf1918e7})},
        {false, -136, MType({0x8b5ce79b0965962a, 0xb22edb0636da31d6})},
        {false, -136, MType({0x6724232b07396427, 0xb3ee6133f7149769})},
        {false, -136, MType({0x2f02b14dcad8a49c, 0xb5adee6b386e62ae})},
        {false, -136, MType({0xbd6443a81f792e07, 0xb76d82ac339058db})},
        {false, -136, MType({0xea1cd9625749939a, 0xb92d1df72125eb7c})},
        {false, -136, MType({0x97775e3142198913, 0xbaecc04c39dd389b})},
        {false, -136, MType({0xc2a701b809a2bc39, 0xbcac69abb6670aeb})},
        {false, -136, MType({0x979b990f39e662e3, 0xbe6c1a15cf76d9f6})},
        {false, -136, MType({0x88395c463ddd82b2, 0xc02bd18abdc2ca45})},
        {false, -136, MType({0x66f451bd9ba5ed05, 0xc1eb900aba03ad8d})},
        {false, -136, MType({0x84cfb9413f6437a6, 0xc3ab5595fcf502d9})},
        {false, -136, MType({0xd2c1c8d32943ca42, 0xc56b222cbf54f6b6})},
        {false, -136, MType({0x67c0d1fd95192e6, 0xc72af5cf39e4635f})},
        {false, -136, MType({0xc298bf9edb6441f2, 0xc8ead07da566d0e3})},
        {false, -136, MType({0xc22d646addde3910, 0xcaaab2383aa27559})},
        {false, -136, MType({0x7c301e5c7d1ca40, 0xcc6a9aff32603504})},
        {false, -136, MType({0xfb444464df02505, 0xce2a8ad2c56ba27f})},
        {false, -136, MType({0x5f1df3591ae898f, 0xcfea81b32c92feec})},
        {false, -136, MType({0xb43caf8e7b891066, 0xd13a7f7c07506f7d})},
        {false, -136, MType({0x597fb13f0d0fdf19, 0xd2fa82b36a610c4f})},
        {false, -136, MType({0x3c21f1c60a60b0d6, 0xd4ba8cf83dd2a06b})},
        {false, -136, MType({0x2b7455909a0428a4, 0xd67a9e4aba7d7ce5})},
        {false, -136, MType({0x1438b60573d2da10, 0xd83ab6ab193ca223})},
        {false, -136, MType({0x49f86400c5ab2b11, 0xd9fad61992edc008})},
        {false, -136, MType({0xd3c313d148a23c35, 0xdbbafc9660713620})},
        {false, -136, MType({0xbc56852355e0f0d5, 0xdd7b2a21baaa13cc})},
    },
    // -log10(r) for the third step, generated by SageMath with:
    //
    // for i in range(-80, 81):
    //   r = 2^-21 * round( 2^21 / (1 + i*2^(-21)) );
    //   s, m, e = RealField(128)(r).log10().sign_mantissa_exponent();
    //   print("{false," if (s == -1) else "{true,", e, ",
    //         MType({", hex(m % 2^64), ",", hex((m >> 64) % 2^64), "})},");
    /* .step_3 = */
    {
        {true, -143, MType({0x54d7e49898ca0093, 0x8af8b9b322ba8c7d})},
        {true, -143, MType({0xc321bbf16665f29c, 0x893c06529deffc3d})},
        {true, -143, MType({0x8246df7140c3e4ae, 0x877f52e433ac7ec4})},
        {true, -143, MType({0x1deaa9e85780e4c1, 0x85c29f67e3ef35bc})},
        {true, -143, MType({0xcd8a5121a9162d0, 0x8405ebddaeb742cf})},
        {true, -143, MType({0xb10486fa4644308d, 0x824938459403c7a7})},
        {true, -143, MType({0x578a2f61eedd4be8, 0x808c849f93d3e5f0})},
        {true, -144, MType({0x715b4a491790e8a7, 0xfd9fa1d75c4d7ea6})},
        {true, -144, MType({0xefb6273a04c71573, 0xfa263a53c5f6eaf4})},
        {true, -144, MType({0x474d901560c17807, 0xf6acd2b464a25420})},
        {true, -144, MType({0x6b9a5deceb80ec57, 0xf3336af9384dfd7c})},
        {true, -144, MType({0x2665a32f7cc64f79, 0xefba032240f82a5d})},
        {true, -144, MType({0x17c8a67316659363, 0xec409b2f7e9f1e16})},
        {true, -144, MType({0xb62cdd3ef5c8673d, 0xe8c73320f1411bfa})},
        {true, -144, MType({0x4e4be6d5a4a07422, 0xe54dcaf698dc675e})},
        {true, -144, MType({0x32f86ff08c92e22, 0xe1d462b0756f4394})},
        {true, -144, MType({0xce31a0d27359396f, 0xde5afa4e86f7f3ee})},
        {true, -144, MType({0x7efc3180aee36373, 0xdae191d0cd74bbc1})},
        {true, -144, MType({0xbb894b1e0ce72fc4, 0xd768293748e3de5e})},
        {true, -144, MType({0x230f6c7270f8be, 0xd3eec081f9439f19})},
        {true, -144, MType({0x9f63aaa563e9a399, 0xd07557b0de924142})},
        {true, -144, MType({0xc2354e441015e7eb, 0xccfbeec3f8ce082d})},
        {true, -144, MType({0x67d22bcf5a452a4c, 0xc98285bb47f5372c})},
        {true, -144, MType({0x65c46fa3e3afea18, 0xc6091c96cc061190})},
        {true, -144, MType({0x67e63bbe1405c20d, 0xc28fb35684fedaab})},
        {true, -144, MType({0xf061a284212afbad, 0xbf1649fa72ddd5ce})},
        {true, -144, MType({0x57b0a1901625b539, 0xbb9ce08295a1464c})},
        {true, -144, MType({0xcc9d1c79d93a9a1e, 0xb82376eeed476f74})},
        {true, -144, MType({0x5440d7a131392da8, 0xb4aa0d3f79ce9499})},
        {true, -144, MType({0xca0572f7c9f7a7de, 0xb130a3743b34f90a})},
        {true, -144, MType({0xdfa464cb37fe6455, 0xadb7398d3178e019})},
        {true, -144, MType({0x1d26f48efb62e2e0, 0xaa3dcf8a5c988d17})},
        {true, -144, MType({0xe0e635a681d259e2, 0xa6c4656bbc924352})},
        {true, -144, MType({0x5f8b022f27cbda35, 0xa34afb315164461d})},
        {true, -144, MType({0xa40df5ca390a0465, 0x9fd190db1b0cd8c6})},
        {true, -144, MType({0x8fb76866f01c4f2d, 0x9c582669198a3e9e})},
        {true, -144, MType({0xda1f690c752fdeff, 0x98debbdb4cdabaf4})},
        {true, -144, MType({0x112db8a3dc07ee78, 0x95655131b4fc9119})},
        {true, -144, MType({0x9919c4c22125c79e, 0x91ebe66c51ee045a})},
        {true, -144, MType({0xac6aa27226204db3, 0x8e727b8b23ad5808})},
        {true, -144, MType({0x5bf708fead2b1780, 0x8af9108e2a38cf72})},
        {true, -144, MType({0x8ee54cbc53cd19ed, 0x877fa575658eade6})},
        {true, -144, MType({0x2ab59d38cc6e2c5, 0x84063a40d5ad36b4})},
        {true, -144, MType({0x4b0eaf0a99286378, 0x808ccef07a92ad29})},
        {true, -145, MType({0xa448b11f012c975c, 0xfa26c708a87aa929})},
        {true, -145, MType({0xb0a1d584117de73b, 0xf333eff8c556e089})},
        {true, -145, MType({0xe890f9fb57fdabb6, 0xec4118b14bb6870e})},
        {true, -145, MType({0x261d48c71e693130, 0xe54e41323b962355})},
        {true, -145, MType({0xefecdd48ed894c32, 0xde5b697b94f23bf7})},
        {true, -145, MType({0x7944b9957598a88a, 0xd768918d57c75792})},
        {true, -145, MType({0xa208bc0875093645, 0xd075b9678411fcbf})},
        {true, -145, MType({0xf6bb94d89da8b432, 0xc982e10a19ceb219})},
        {true, -145, MType({0xb07ebbab782457b0, 0xc290087518f9fe3b})},
        {true, -145, MType({0xb512652945eb9165, 0xbb9d2fa8819067be})},
        {true, -145, MType({0x96d57890e171eea5, 0xb4aa56a4538e753c})},
        {true, -145, MType({0x94c5854b9cd01726, 0xadb77d688ef0ad4e})},
        {true, -145, MType({0x9a7eb8811ec3e6bb, 0xa6c4a3f533b3968d})},
        {true, -145, MType({0x403bd2ab3e0fa2d7, 0x9fd1ca4a41d3b792})},
        {true, -145, MType({0xcad61d29db384b6b, 0x98def067b94d96f4})},
        {true, -145, MType({0x2bc55fd6b8a306ec, 0x91ec164d9a1dbb4d})},
        {true, -145, MType({0x11fd6995111a927, 0x8af93bfbe440ab33})},
        {true, -145, MType({0x959a26faac7e5494, 0x8406617297b2ed3d})},
        {true, -146, MType({0xc10eab7266ac6bc0, 0xfa270d6368e21007})},
        {true, -146, MType({0xbb178b90026b2b2, 0xec41577274ef0439})},
        {true, -146, MType({0xac3bfd925e6b33e1, 0xde5ba1125385c43b})},
        {true, -146, MType({0x9d0a01a95b355319, 0xd075ea43049f5d3b})},
        {true, -146, MType({0x31b3b7b20a6a6496, 0xc29033048834dc64})},
        {true, -146, MType({0x170da891504620f4, 0xb4aa7b56de3f4ee0})},
        {true, -146, MType({0x53289e84744549cb, 0xa6c4c33a06b7c1d9})},
        {true, -146, MType({0x45519048b0ce7e7f, 0x98df0aae01974279})},
        {true, -146, MType({0xa6118c42bf99407e, 0x8af951b2ced6dde8})},
        {true, -147, MType({0xe5b474cc5a64cf6, 0xfa273090dcdf429f})},
        {true, -147, MType({0xa74dab3bd6067bc7, 0xde5bbcddc0b533aa})},
        {true, -147, MType({0x9f73f4e37357341b, 0xc290484c4921a941})},
        {true, -147, MType({0x31bf5d5f815220e7, 0xa6c4d2dc7616bdb0})},
        {true, -147, MType({0x4b987ca5fca242d7, 0x8af95c8e47868b41})},
        {true, -148, MType({0x19be3fabd93832c5, 0xde5bcac37ac6587d})},
        {true, -148, MType({0x8fd43f0c9ce444d3, 0xa6c4daadaf3d75e0})},
        {true, -149, MType({0x61cd853e796bc2c, 0xde5bd1b658ad4676})},
        {true, -150, MType({0x87d6afabfba0644f, 0xde5bd52fc7d8545f})},
        {false, 0, MType({0x0, 0x0})},
        {false, -150, MType({0xa9bf32001043629d, 0xde5bdc22a69d9e19})},
        {false, -149, MType({0x8014f0f360272d82, 0xde5bdf9c1637d9ef})},
        {false, -148, MType({0xfe94a02fc639c0e3, 0xa6c4ea5024795bd2})},
        {false, -148, MType({0xbee710a5ace7c8d4, 0xde5be68ef5db7f99})},
        {false, -147, MType({0x1a778d8100437e4f, 0x8af972453faf11e8})},
        {false, -147, MType({0x97d773f8992f7051, 0xa6c4f221608e89fe})},
        {false, -147, MType({0xc9ee5841a3afa95, 0xc29072dbdd9a0dd5})},
        {false, -147, MType({0x7b644b13993cf4ef, 0xde5bf474b6df8331})},
        {false, -147, MType({0x3448f66e2bd7a0ca, 0xfa2776ebec6ccfdb})},
        {false, -146, MType({0x6a7ca5f1a87a1a3c, 0x8af97d20bf27eccd})},
        {false, -146, MType({0x245675fe3061108f, 0x98df3f3ab64b431d})},
        {false, -146, MType({0x64136e97019d0a3b, 0xa6c501c3dba75dc2})},
        {false, -146, MType({0x6cdadac4d6925bd4, 0xb4aac4bc2f432fa3})},
        {false, -146, MType({0x2899e23791d29632, 0xc2908823b125aba7})},
        {false, -146, MType({0x28039e1f0323a4c1, 0xd0764bfa6155c4b5})},
        {false, -146, MType({0xa2912e03afc8cc28, 0xde5c10403fda6db5})},
        {false, -146, MType({0x7681cc9f9e0d89f9, 0xec41d4f54cba9991})},
        {false, -146, MType({0x28dae4b7241255e1, 0xfa279a1987fd3b32})},
        {false, -145, MType({0xf2b412f8dceda28e, 0x8406afd678d4a2c0})},
        {false, -145, MType({0xbf5dccd967504857, 0x8af992d7c4e2d5b5})},
        {false, -145, MType({0x3716dbf950b07f85, 0x91ec7610a82cafed})},
        {false, -145, MType({0x69eebe0b8e5b18e1, 0x98df598122b5aadd})},
        {false, -145, MType({0xbb583ce65af56beb, 0x9fd23d2934813ffc})},
        {false, -145, MType({0xe22978efa7a962a0, 0xa6c52108dd92e8c1})},
        {false, -145, MType({0xe89bf3898ef27836, 0xadb805201dee1ea3})},
        {false, -145, MType({0x2c4c997ec90bab0b, 0xb4aae96ef5965b1a})},
        {false, -145, MType({0x5e3bcd6f21fe6224, 0xbb9dcdf5648f179c})},
        {false, -145, MType({0x82cd723bf1524680, 0xc290b2b36adbcda2})},
        {false, -145, MType({0xf1c8f574935e109b, 0xc98397a9087ff6a4})},
        {false, -145, MType({0x565959c2e4394a59, 0xd0767cd63d7f0c1c})},
        {false, -145, MType({0xaf0d4157bc4f05be, 0xd769623b09dc8781})},
        {false, -145, MType({0x4dd6f8576e9188b8, 0xde5c47d76d9be24e})},
        {false, -145, MType({0xd80c7f46484eee3d, 0xe54f2dab68c095fb})},
        {false, -145, MType({0x4667957512a6bd26, 0xec4213b6fb4e1c04})},
        {false, -145, MType({0xe505c36d95a074fa, 0xf334f9fa2547ede1})},
        {false, -145, MType({0x5368655f1ce3110b, 0xfa27e074e6b1850f})},
        {false, -144, MType({0xc23a5ac57f06c112, 0x808d63939fc72d83})},
        {false, -144, MType({0xdf39eb5890580f93, 0x8406d70897f0f4a2})},
        {false, -144, MType({0xcd896f3e43f38669, 0x87804a995bd7d4a2})},
        {false, -144, MType({0x83b16ff7eecace8c, 0x8af9be45eb7d8a41})},
        {false, -144, MType({0x21ec7ae8ffa1531d, 0x8e73320e46e3d23d})},
        {false, -144, MType({0xf227268d464ae907, 0x91eca5f26e0c6953})},
        {false, -144, MType({0x680017af3bbaf2d3, 0x956619f260f90c44})},
        {false, -144, MType({0x20c8069e4ae400de, 0x98df8e0e1fab77cd})},
        {false, -144, MType({0xe381c4651a67ee13, 0x9c590245aa2568ac})},
        {false, -144, MType({0xa0e23fffd718794e, 0x9fd2769900689ba2})},
        {false, -144, MType({0x73508b927f485b97, 0xa34beb082276cd6d})},
        {false, -144, MType({0x9ee5e19f2eecdb55, 0xa6c55f931051bacc})},
        {false, -144, MType({0x916daa3c6c8fdc9d, 0xaa3ed439c9fb207f})},
        {false, -144, MType({0xe265804b77126ed3, 0xadb848fc4f74bb45})},
        {false, -144, MType({0x52fd36ae943fd7b4, 0xb131bddaa0c047df})},
        {false, -144, MType({0xce16dd7f60311bf6, 0xb4ab32d4bddf830b})},
        {false, -144, MType({0x6846c7451d8105ac, 0xb824a7eaa6d4298b})},
        {false, -144, MType({0x5fd38e2b0650a884, 0xbb9e1d1c5b9ff81e})},
        {false, -144, MType({0x1cb619369e1c641f, 0xbf179269dc44ab85})},
        {false, -144, MType({0x3099a17e0461648c, 0xc29107d328c40080})},
        {false, -144, MType({0x56dbb75e4813a12b, 0xc60a7d58411fb3d0})},
        {false, -144, MType({0x748c47b1bbe45a07, 0xc983f2f925598236})},
        {false, -144, MType({0x986da1064b5913e1, 0xccfd68b5d5732873})},
        {false, -144, MType({0xfaf478d3d0b31300, 0xd076de8e516e6348})},
        {false, -144, MType({0xfe47f0b26ba754ff, 0xd3f05482994cef77})},
        {false, -144, MType({0x2e419b90d8e709b7, 0xd769ca92ad1089c2})},
        {false, -144, MType({0x406d82eaca788b6f, 0xdae340be8cbaeee9})},
        {false, -144, MType({0x140a2bff40e0d670, 0xde5cb706384ddbaf})},
        {false, -144, MType({0xb2089d06e51d8034, 0xe1d62d69afcb0cd5})},
        {false, -144, MType({0x4d0c626a636f2e4f, 0xe54fa3e8f3343f1f})},
        {false, -144, MType({0x416b93f8c6f48d30, 0xe8c91a84028b2f4e})},
        {false, -144, MType({0x152eda1dd615c6f5, 0xec42913addd19a25})},
        {false, -144, MType({0x781173186fc07a66, 0xefbc080d85093c66})},
        {false, -144, MType({0x43813830e974324d, 0xf3357efbf833d2d5})},
        {false, -144, MType({0x7a9ea2ef6e1f5d41, 0xf6aef60637531a34})},
        {false, -144, MType({0x4a3cd2525dccc623, 0xfa286d2c4268cf47})},
        {false, -144, MType({0x8e19004ae218d5d, 0xfda1e46e1976aed1})},
        {false, -143, MType({0x9b62aaca25d5d18a, 0x808dade5de3f3aca})},
        {false, -143, MType({0xbee9a8d43e00613c, 0x824a69a295c0f02b})},
        {false, -143, MType({0xd8d4b69c2056f729, 0x8407256d334155ed})},
        {false, -143, MType({0xe7cc28605d7bb77e, 0x85c3e145b6c14a72})},
        {false, -143, MType({0xff51b4bdc834a8f1, 0x87809d2c2041ac1c})},
        {false, -143, MType({0x47c0774aa81c3561, 0x893d59206fc3594e})},
        {false, -143, MType({0xfe4cf331ecb9eb62, 0x8afa1522a5473068})},
    },
    // -log10(r) for the fourth step, generated by SageMath with:
    //
    // for i in range(-65, 65):
    //   r = 2^-28 * round( 2^28 / (1 + i*2^(-28)) );
    //   s, m, e = RealField(128)(r).log10().sign_mantissa_exponent();
    //   print("{false," if (s == -1) else "{true,", e, ",
    //         MType({", hex(m % 2^64), ",", hex((m >> 64) % 2^64), "})},");
    /* .step_4 = */
    {
        {true, -151, MType({0xe471a82bbedbe0ae, 0xe1d5464122cf95a4})},
        {true, -151, MType({0xaf6e93be8e4c1764, 0xde5bd6ec7f7bc110})},
        {true, -151, MType({0xe44848f0a5779499, 0xdae26797a490f80e})},
        {true, -151, MType({0x90205533f4e70566, 0xd768f842920f3a98})},
        {true, -151, MType({0xc01844ace3729e48, 0xd3ef88ed47f688a6})},
        {true, -151, MType({0x8151a2324e41c7c4, 0xd0761997c646e232})},
        {true, -151, MType({0xe0edf74d88cacafd, 0xccfcaa420d004734})},
        {true, -151, MType({0xec0ecc3a5cd27e58, 0xc9833aec1c22b7a6})},
        {true, -151, MType({0xafd5a7e70a6bf214, 0xc609cb95f3ae3381})},
        {true, -151, MType({0x39640ff447f81ceb, 0xc2905c3f93a2babe})},
        {true, -151, MType({0x95db88b5422588b1, 0xbf16ece8fc004d55})},
        {true, -151, MType({0xd25d952f9beffeec, 0xbb9d7d922cc6eb40})},
        {true, -151, MType({0xfc0bb71b6ea03578, 0xb8240e3b25f69478})},
        {true, -151, MType({0x20076ee349cb7b20, 0xb4aa9ee3e78f48f7})},
        {true, -151, MType({0x4b723ba43353643d, 0xb1312f8c719108b4})},
        {true, -151, MType({0x8b6d9b2da7657754, 0xadb7c034c3fbd3a9})},
        {true, -151, MType({0xed1b0a01987ad9b4, 0xaa3e50dcdecfa9cf})},
        {true, -151, MType({0x7d9c03546f57fc11, 0xa6c4e184c20c8b20})},
        {true, -151, MType({0x4a12010d0b0c4727, 0xa34b722c6db27794})},
        {true, -151, MType({0x5f9e7bc4c0f1c851, 0x9fd202d3e1c16f24})},
        {true, -151, MType({0xcb62eac75cacde29, 0x9c58937b1e3971c9})},
        {true, -151, MType({0x9a80c413202be52a, 0x98df2422231a7f7d})},
        {true, -151, MType({0xda197c58c3a6e445, 0x9565b4c8f0649838})},
        {true, -151, MType({0x974e86fb759f3988, 0x91ec456f8617bbf4})},
        {true, -151, MType({0xdf415610dadf46b3, 0x8e72d615e433eaa9})},
        {true, -151, MType({0xbf135a610e7a1ddc, 0x8af966bc0ab92451})},
        {true, -151, MType({0x43e60366a1cb2e09, 0x877ff761f9a768e5})},
        {true, -151, MType({0x7adabf4e9c75efce, 0x84068807b0feb85d})},
        {true, -151, MType({0x7112faf87c6591ee, 0x808d18ad30bf12b3})},
        {true, -152, MType({0x676043ec6b994be5, 0xfa2752a4f1d0efc0})},
        {true, -152, MType({0x9fa73d186649999d, 0xf33473ef12f5cfb9})},
        {true, -152, MType({0xa53db362aa5cc6f0, 0xec419538c4ecc544})},
        {true, -152, MType({0x9266761de5e05f13, 0xe54eb68207b5d053})},
        {true, -152, MType({0x81645201b36e17ba, 0xde5bd7cadb50f0d8})},
        {true, -152, MType({0x8c7a112a9a2b2a52, 0xd768f9133fbe26c5})},
        {true, -152, MType({0xcdea7b1a0dc7ad42, 0xd0761a5b34fd720c})},
        {true, -152, MType({0x5ff854b66e7ded1f, 0xc9833ba2bb0ed2a0})},
        {true, -152, MType({0x5ce6604b0911c5ed, 0xc2905ce9d1f24872})},
        {true, -152, MType({0xdef75d8816cffc59, 0xbb9d7e3079a7d374})},
        {true, -152, MType({0x6e0982bd8d96ef, 0xb4aa9f76b22f739a})},
        {true, -152, MType({0xdb8d1eb50fa7375c, 0xadb7c0bc7b8928d3})},
        {true, -152, MType({0x8a9754fe0c0073a7, 0xa6c4e201d5b4f314})},
        {true, -152, MType({0x27cf61a19e032f69, 0x9fd20346c0b2d24e})},
        {true, -152, MType({0xcd77f7489d9ef50b, 0x98df248b3c82c672})},
        {true, -152, MType({0x95d3c600cf484f03, 0x91ec45cf4924cf74})},
        {true, -152, MType({0x9b257b3ce3f82109, 0x8af96712e698ed45})},
        {true, -152, MType({0xf7afc1d4792b015a, 0x8406885614df1fd7})},
        {true, -153, MType({0x8b6a840831c123d8, 0xfa275331a7eece3b})},
        {true, -153, MType({0x3ef142da7335b35a, 0xec4195b647c38612})},
        {true, -153, MType({0x3e79062c7cbb3b7d, 0xde5bd83a093c6718})},
        {true, -153, MType({0xbe870ed4ed5b755b, 0xd0761abcec597131})},
        {true, -153, MType({0xf3a098743d20fb64, 0xc2905d3ef11aa442})},
        {true, -153, MType({0x124ad974bd15fbca, 0xb4aa9fc017800030})},
        {true, -153, MType({0x4f0b030a9742eb00, 0xa6c4e2405f8984dd})},
        {true, -153, MType({0xde664133cead362d, 0x98df24bfc937322e})},
        {true, -153, MType({0xf4e1bab83f55f5a1, 0x8af9673e54890808})},
        {true, -154, MType({0x8e0522533c713e98, 0xfa27537802fe0c9f})},
        {true, -154, MType({0x129bc1c6f293726e, 0xde5bd871a03259cf})},
        {true, -154, MType({0xe09182166eeb17eb, 0xc2905d6980aef768})},
        {true, -154, MType({0x60f08720313daa3f, 0xa6c4e25fa473e535})},
        {true, -154, MType({0xfcc2ea566b3af38b, 0x8af967540b8122fc})},
        {true, -155, MType({0x3a25757e00f4e3a0, 0xde5bd88d6bad6110})},
        {true, -155, MType({0x55d3f9e70cf177b8, 0xa6c4e26f46e91b3e})},
        {true, -156, MType({0x3d4aac85125398d0, 0xde5bd89b516ae82a})},
        {true, -157, MType({0x9ab5a849a06f400d, 0xde5bd8a24449ac95})},
        {false, 0, MType({0x0, 0x0})},
        {false, -157, MType({0xd3cc88fd4ef34c2, 0xde5bd8b02a073729})},
        {false, -156, MType({0x225916c2b3f33c90, 0xde5bd8b71ce5fd51})},
        {false, -155, MType({0x17847f98acf08d54, 0xa6c4e28e8bd3930a})},
        {false, -155, MType({0x44397830931fddd, 0xde5bd8c502a38b5e})},
        {false, -154, MType({0xc2ab385913176984, 0x8af9677f79717409})},
        {false, -154, MType({0xe454dec82bde52e5, 0xa6c4e29e2e48d4cc})},
        {false, -154, MType({0xfe1522b0470d7d7f, 0xc2905dbe9fd7e82f})},
        {false, -154, MType({0xa6e2721f2afc3cce, 0xde5bd8e0ce1eae6a})},
        {false, -154, MType({0x75b3458eec3c106c, 0xfa275404b91d27b4})},
        {false, -153, MType({0x80bf0ff2f6cd9f93, 0x8af967953069aa22})},
        {false, -153, MType({0xf09cc73b7013b906, 0x98df2528e2a09a29})},
        {false, -153, MType({0x55ee1480619827c4, 0xa6c4e2bd7333640c})},
        {false, -153, MType({0x7c2e48d772250b3c, 0xb4aaa052e22207e5})},
        {false, -153, MType({0x2ed8ba8c6fa81c98, 0xc2905de92f6c85d1})},
        {false, -153, MType({0x3968c5214f33fc4f, 0xd0761b805b12ddeb})},
        {false, -153, MType({0x6759c94e2d017fad, 0xde5bd9186515104f})},
        {false, -153, MType({0x84272d014c70fe58, 0xec4196b14d731d19})},
        {false, -153, MType({0x5b4c5b5f180b9fe1, 0xfa27544b142d0465})},
        {false, -152, MType({0x5c22626110c254a4, 0x840688f2dca16327})},
        {false, -152, MType({0xb345ef5d90dd6545, 0x8af967c09e5a3178})},
        {false, -152, MType({0x98ce92087c5cb614, 0x91ec468ecf40ed34})},
        {false, -152, MType({0xf27a0a6056dcfe57, 0x98df255d6f559668})},
        {false, -152, MType({0xa6061afeb7929f24, 0x9fd2042c7e982d23})},
        {false, -152, MType({0x99308918494a4a20, 0xa6c4e2fbfd08b172})},
        {false, -152, MType({0xb1b71c7cca69a844, 0xadb7c1cbeaa72363})},
        {false, -152, MType({0xd5579f970cf000a9, 0xb4aaa09c47738304})},
        {false, -152, MType({0xe9cfdf6cf676df42, 0xbb9d7f6d136dd063})},
        {false, -152, MType({0xd4ddab9f8032bbab, 0xc2905e3e4e960b8e})},
        {false, -152, MType({0x7c3ed66ab6f39fe9, 0xc9833d0ff8ec3493})},
        {false, -152, MType({0xc5b134a5bb25cf2e, 0xd0761be212704b7f})},
        {false, -152, MType({0x96f29dc2c0d26ca0, 0xd768fab49b225061})},
        {false, -152, MType({0xd5c0ebcf0fa0221e, 0xde5bd98793024346})},
        {false, -152, MType({0x67d9fb7302d3c705, 0xe54eb85afa10243d})},
        {false, -152, MType({0x32fbabf2095106f1, 0xec41972ed04bf353})},
        {false, -152, MType({0x1ce3df2aa59b0889, 0xf334760315b5b096})},
        {false, -152, MType({0xb5079966dd5143e, 0xfa2754d7ca4d5c14})},
        {false, -151, MType({0x71ffb12505e19d89, 0x808d19d677097aed})},
        {false, -151, MType({0x4657417a9e657eae, 0x8406894140833efc})},
        {false, -151, MType({0x758de3f168f9f8c9, 0x877ff8ac4193fa3d})},
        {false, -151, MType({0xf2828ffc57f43581, 0x8af968177a3bacb7})},
        {false, -151, MType({0xb0143e5be77b1053, 0x8e72d782ea7a5672})},
        {false, -151, MType({0xa121e91e1d8769ef, 0x91ec46ee924ff774})},
        {false, -151, MType({0xb88a8b9e89e47b9c, 0x9565b65a71bc8fc4})},
        {false, -151, MType({0xe92d228646302a9c, 0x98df25c688c01f69})},
        {false, -151, MType({0x25e8abcbf5db5b8c, 0x9c589532d75aa66b})},
        {false, -151, MType({0x619c26b3c62a45c8, 0x9fd2049f5d8c24cf})},
        {false, -151, MType({0x8f2693cf6e34c6cc, 0xa34b740c1b549a9d})},
        {false, -151, MType({0xa166f4fe2ee6b59a, 0xa6c4e37910b407dc})},
        {false, -151, MType({0x8b3c4d6cd3003616, 0xaa3e52e63daa6c93})},
        {false, -151, MType({0x3f85a195af160c71, 0xadb7c253a237c8c9})},
        {false, -151, MType({0xb121f740a191f084, 0xb13131c13e5c1c84})},
        {false, -151, MType({0xd2f0558312b2e136, 0xb4aaa12f121767cc})},
        {false, -151, MType({0x97cfc4bff48d77de, 0xb824109d1d69aaa8})},
        {false, -151, MType({0xf29f4ea7c30c3ba5, 0xbb9d800b6052e51e})},
        {false, -151, MType({0xd63dfe3883eff4e9, 0xbf16ef79dad31736})},
        {false, -151, MType({0x358adfbdc6d0009f, 0xc2905ee88cea40f7})},
        {false, -151, MType({0x36500d0a51aa3b6, 0xc609ce5776986267})},
        {false, -151, MType({0x32ab7057c2155e78, 0xc9833dc697dd7b8d})},
        {false, -151, MType({0xb63d3e874add3ff0, 0xccfcad35f0b98c70})},
        {false, -151, MType({0x80f97ce0f6673948, 0xd0761ca5812c9518})},
        {false, -151, MType({0x85bf3e340580712d, 0xd3ef8c154936958b})},
        {false, -151, MType({0xb76d969d42ce9734, 0xd768fb8548d78dd0})},
        {false, -151, MType({0x8e39b8702d0373a, 0xdae26af5800f7def})},
        {false, -151, MType({0x6d0063a923dd0cc6, 0xde5bda65eede65ed})},
    }};

// > P = fpminimax(log10(1 + x)/x, 3, [|128...|], [-0x1.0002143p-29 , 0x1p-29]);
// > P;
// > dirtyinfnorm(log10(1 + x)/x - P, [-0x1.0002143p-29 , 0x1p-29]);
// 0x1.64fb8...p-123
const Float128 BIG_COEFFS[4]{
    {true, -131, MType({0x6903c4ce1582517d, 0xde5bd8a9373f89a7})},
    {false, -130, MType({0xb8a21791624e2e8a, 0x943d3b1b7a1af679})},
    {true, -130, MType({0x355baaafabc25990, 0xde5bd8a937287195})},
    {false, -129, MType({0x355baaafad33dbd9, 0xde5bd8a937287195})},
};

// Reuse the output of the fast pass range reduction.
// -2^-8 <= m_x < 2^-7
double log10_accurate(int e_x, int index, double m_x) {

  Float128 e_x_f128(static_cast<float>(e_x));
  Float128 sum = fputil::quick_mul(LOG10_2, e_x_f128);
  sum = fputil::quick_add(sum, LOG10_TABLE.step_1[index]);

  Float128 v_f128 = log_range_reduction(m_x, LOG10_TABLE, sum);

  // Polynomial approximation
  Float128 p = fputil::quick_mul(v_f128, BIG_COEFFS[0]);
  p = fputil::quick_mul(v_f128, fputil::quick_add(p, BIG_COEFFS[1]));
  p = fputil::quick_mul(v_f128, fputil::quick_add(p, BIG_COEFFS[2]));
  p = fputil::quick_mul(v_f128, fputil::quick_add(p, BIG_COEFFS[3]));

  Float128 r = fputil::quick_add(sum, p);

  return static_cast<double>(r);
}

} // namespace

LLVM_LIBC_FUNCTION(double, log10, (double x)) {
  using FPBits_t = typename fputil::FPBits<double>;
  FPBits_t xbits(x);
  uint64_t x_u = xbits.uintval();

  int x_e = -FPBits_t::EXPONENT_BIAS;

  if (LIBC_UNLIKELY(x_u == 0x3FF0'0000'0000'0000ULL)) {
    // log10(1.0) = +0.0
    return 0.0;
  }

  if (LIBC_UNLIKELY(xbits.uintval() < FPBits_t::MIN_NORMAL ||
                    xbits.uintval() > FPBits_t::MAX_NORMAL)) {
    if (xbits.is_zero()) {
      // return -Inf and raise FE_DIVBYZERO.
      fputil::set_errno_if_required(ERANGE);
      fputil::raise_except_if_required(FE_DIVBYZERO);
      return static_cast<double>(FPBits_t::neg_inf());
    }
    if (xbits.get_sign() && !xbits.is_nan()) {
      fputil::set_errno_if_required(EDOM);
      fputil::raise_except_if_required(FE_INVALID);
      return FPBits_t::build_quiet_nan(0);
    }
    if (xbits.is_inf_or_nan()) {
      return x;
    }
    // Normalize denormal inputs.
    xbits.set_val(x * 0x1.0p52);
    x_e -= 52;
    x_u = xbits.uintval();
  }

  // log10(x) = log10(2^x_e * x_m)
  //          = x_e * log10(2) + log10(x_m)

  // Range reduction for log10(x_m):
  // For each x_m, we would like to find r such that:
  //   -2^-8 <= r * x_m - 1 < 2^-7
  int shifted = x_u >> 45;
  int index = shifted & 0x7F;
  double r = RD[index];

  // Add unbiased exponent. Add an extra 1 if the 8 leading fractional bits are
  // all 1's.
  x_e += static_cast<int>((x_u + (1ULL << 45)) >> 52);
  double e_x = static_cast<double>(x_e);

  // hi is exact
  double hi = fputil::multiply_add(e_x, LOG_2_HI, LOG_R_DD[index].hi);
  // lo errors ~ e_x * LSB(LOG_2_LO) + LSB(LOG_R[index].lo) + rounding err
  //           <= 2 * (e_x * LSB(LOG_2_LO) + LSB(LOG_R[index].lo))
  double lo = fputil::multiply_add(e_x, LOG_2_LO, LOG_R_DD[index].lo);

  // Set m = 1.mantissa.
  uint64_t x_m = (x_u & 0x000F'FFFF'FFFF'FFFFULL) | 0x3FF0'0000'0000'0000ULL;
  double m = FPBits_t(x_m).get_val();

  double u, u_sq, err;
  fputil::DoubleDouble r1;

  // Perform exact range reduction
#ifdef LIBC_TARGET_CPU_HAS_FMA
  u = fputil::multiply_add(r, m, -1.0); // exact
#else
  uint64_t c_m = x_m & 0x3FFF'E000'0000'0000ULL;
  double c = FPBits_t(c_m).get_val();
  u = fputil::multiply_add(r, m - c, CD[index]); // exact
#endif // LIBC_TARGET_CPU_HAS_FMA

  // Error of u_sq = ulp(u^2);
  u_sq = u * u;
  // Degree-7 minimax polynomial
  double p0 = fputil::multiply_add(u, LOG_COEFFS[1], LOG_COEFFS[0]);
  double p1 = fputil::multiply_add(u, LOG_COEFFS[3], LOG_COEFFS[2]);
  double p2 = fputil::multiply_add(u, LOG_COEFFS[5], LOG_COEFFS[4]);
  double p = fputil::polyeval(u_sq, lo, p0, p1, p2);

  // Exact sum:
  //   r1.hi + r1.lo = e_x * log(2)_hi - log(r)_hi + u
  r1 = fputil::exact_add(hi, u);
  r1.lo += p;

  // Quick double-double multiplication:
  //   r2.hi + r2.lo ~ r1 * log10(e),
  // with error bounded by:
  //   4*ulp( ulp(r2.hi) )
  fputil::DoubleDouble r2 = fputil::quick_mult(r1, LOG10_E);

  // Technicallly error of r1.lo is bounded by:
  //    |hi|*ulp(log(2)_lo) + C*ulp(u^2)
  // To simplify the error computation a bit, we replace |hi|*ulp(log(2)_lo)
  // with the upper bound: 2^11 * ulp(log(2)_lo) = 2^-85.
  // Total error is bounded by ~ C * ulp(u^2) + 2^-85.
  err = fputil::multiply_add(u_sq, P_ERR, HI_ERR);

  // Lower bound from the result
  double left = r2.hi + (r2.lo - err);
  // Upper bound from the result
  double right = r2.hi + (r2.lo + err);

  // Ziv's test if fast pass is accurate enough.
  if (left == right)
    return left;

  // Exact cases:
  if (LIBC_UNLIKELY((x_u & 0x3FFFFF) == 0)) {
    switch (x_u) {
    case 0x4024000000000000: // x = 10.0
      return 1.0;
    case 0x4059000000000000: // x = 10^2
      return 2.0;
    case 0x408f400000000000: // x = 10^3
      return 3.0;
    case 0x40c3880000000000: // x = 10^4
      return 4.0;
    case 0x40f86a0000000000: // x = 10^5
      return 5.0;
    case 0x412e848000000000: // x = 10^6
      return 6.0;
    case 0x416312d000000000: // x = 10^7
      return 7.0;
    case 0x4197d78400000000: // x = 10^8
      return 8.0;
    case 0x41cdcd6500000000: // x = 10^9
      return 9.0;
    case 0x4202a05f20000000: // x = 10^10
      return 10.0;
    case 0x42374876e8000000: // x = 10^11
      return 11.0;
    case 0x426d1a94a2000000: // x = 10^12
      return 12.0;
    case 0x42a2309ce5400000: // x = 10^13
      return 13.0;
    }
  } else {
    switch (x_u) {
    case 0x42d6bcc41e900000: // x = 10^14
      return 14.0;
    case 0x430c6bf526340000: // x = 10^15
      return 15.0;
    case 0x4341c37937e08000: // x = 10^16
      return 16.0;
    case 0x4376345785d8a000: // x = 10^17
      return 17.0;
    case 0x43abc16d674ec800: // x = 10^18
      return 18.0;
    case 0x43e158e460913d00: // x = 10^19
      return 19.0;
    case 0x4415af1d78b58c40: // x = 10^20
      return 20.0;
    case 0x444b1ae4d6e2ef50: // x = 10^21
      return 21.0;
    case 0x4480f0cf064dd592: // x = 10^22
      return 22.0;
    }
  }

  return log10_accurate(x_e, index, u);
}

} // namespace __llvm_libc
