/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcas_gsi_utils.c
    \brief  This file contains utility functions to retrieve info from gsi
            credentials/contexts (at the gssapi level)
    \author Martijn Steenbakkers for EGEE

    This file contains utility functions to retrieve info from gsi
    credentials/contexts (at the gssapi level)
    Contains modified globus code.
*/

#ifndef LCAS_GSI_UTILS_C
#define LCAS_GSI_UTILS_C

#define _XOPEN_SOURCE 500
 
/******************************************************************************
                             Include header files
******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* For X509 and STACK_OF(X509) structs (output) */
#include <openssl/x509.h>

/* For the gss_cred_id_t struct (input) */
#include <gssapi.h>

/* Internal globus header:
 * defines gss_cred_id_desc (= internal structure of gss_cred_id_t)
 */
/* #include "gssapi_openssl.h" */

/* Defines globus_gsi_cred_handle_t (GSI (globus) credential,
 * part of gss_cred_id_desc)
 * and interface functions to globus_gsi_cred_handle_t:
 * globus_gsi_cred_get_cert() and globus_gsi_cred_get_cert_chain()
 */
/* #include "globus_gsi_credential.h" */

/* Internal globus header:
 * describes internal structure of globus_gsi_cred_get_cert
 * not needed if interface functions to globus_gsi_cred_handle_t are used
 */
//#include "globus_i_gsi_credential.h"

/* LCAS includes */
#include "_lcas_globus_internal.h"
#include "_lcas_gsi_utils.h"

/******************************************************************************
Function:       lcas_gss_cred_to_dn() (copied from GLOBUS gatekeeper.c)
Description:    Get the globusid from gssapi
Parameters:
                globus_cred: globus credential
Returns:        globusid string (which could be freeed)
******************************************************************************/
char * lcas_gss_cred_to_dn(
        gss_cred_id_t globus_cred
)
{
    char*                         globusid = NULL;
    char*                         globusid_tmp = NULL;
    gss_name_t                    globus_name = GSS_C_NO_NAME;
    gss_buffer_desc               globus_buffer_desc = GSS_C_EMPTY_BUFFER;
    gss_buffer_t                  globus_buffer = &globus_buffer_desc;
    OM_uint32                     major_status = 0;
    OM_uint32                     minor_status = 0;
    OM_uint32                     minor_status2 = 0;

    if ((major_status = gss_inquire_cred(&minor_status,
                                         globus_cred,
                                         &globus_name,
                                         NULL, NULL, NULL)) == GSS_S_COMPLETE)
    {
        major_status = gss_display_name(&minor_status,
                                        globus_name, globus_buffer, NULL);
        gss_release_name(&minor_status2, &globus_name);
    }
    /*
     * The gssapi_cleartext does not implement gss_inquire_cred,
     * so fall back to using environment variable.
     */
    if (major_status == GSS_S_COMPLETE)
    {
        globusid = globus_buffer_desc.value;
    }
    else
    {
        globusid = getenv("GLOBUSID");
        globusid = (globusid ? globusid : "GLOBUSID");
    }
    globusid_tmp = strdup(globusid);

    if (globus_buffer_desc.value)
    {
        gss_release_buffer(&minor_status2, globus_buffer);
    }
    return globusid_tmp;
}

/******************************************************************************
Function:       lcas_cred_to_x509()
Description:    Return the pointer to X509 structure from gss credential
Parameters:
                cred: globus credential
Returns:        pointer to X509 struct or NULL
******************************************************************************/
/*!
    \fn lcas_cred_to_x509(
        gss_cred_id_t cred
        )
    \brief Return the pointer to X509 structure from gss credential

    This function takes a gss credential as input and returns the corresponding
    X509 structure, which is allocated for this purpose (should be freed)
    
    \param cred the gss credential

    \return a pointer to a X509 struct or NULL
*/
X509 * lcas_cred_to_x509(
        gss_cred_id_t cred
)
{
    /* Internally a gss_cred_id_t type is a pointer to a gss_cred_id_desc */
    lcas_gss_cred_id_desc *       cred_desc = NULL;
    globus_gsi_cred_handle_t gsi_cred_handle;
    X509 * px509=NULL;

    /* cast to gss_cred_id_desc */
    if (cred != GSS_C_NO_CREDENTIAL)
    {
        cred_desc = (lcas_gss_cred_id_desc *) cred;
        if (globus_module_activate(GLOBUS_GSI_CREDENTIAL_MODULE) ==GLOBUS_SUCCESS)
        {
            gsi_cred_handle = cred_desc->cred_handle;
            if (globus_gsi_cred_get_cert(gsi_cred_handle, &px509) == GLOBUS_SUCCESS)
            {
                globus_module_deactivate(GLOBUS_GSI_CREDENTIAL_MODULE);
                return px509;
            }
            else
            {
                globus_module_deactivate(GLOBUS_GSI_CREDENTIAL_MODULE);
                return NULL;
            }
        }
        else
        {
            globus_module_deactivate(GLOBUS_GSI_CREDENTIAL_MODULE);
            return NULL;
        }
    }
    else
    {
        return NULL;
    }
}


/******************************************************************************
Function:       lcas_cred_to_x509_chain()
Description:    Return the pointer to X509 chain from gss credential
Parameters:
                cred: globus credential
Returns:        pointer to X509 chain or NULL
******************************************************************************/
/*!
    \fn lcas_cred_to_x509_chain(
        gss_cred_id_t cred
        )
    \brief Return the pointer to X509 structure from gss credential

    This function takes a gss credential as input and returns the corresponding
    X509 chain, which is allocated for this purpose (should be freed)
    
    \param cred the gss credential

    \return a pointer to a X509 chain or NULL
*/
STACK_OF(X509) * lcas_cred_to_x509_chain(
        gss_cred_id_t cred
)
{
    /* Internally a gss_cred_id_t type is a pointer to a gss_cred_id_desc */
    lcas_gss_cred_id_desc *       cred_desc = NULL;
    globus_gsi_cred_handle_t gsi_cred_handle;
    STACK_OF(X509) * px509_chain=NULL;

    /* cast to gss_cred_id_desc */
    if (cred != GSS_C_NO_CREDENTIAL)
    {
        cred_desc = (lcas_gss_cred_id_desc *) cred;
        if (globus_module_activate(GLOBUS_GSI_CREDENTIAL_MODULE) ==GLOBUS_SUCCESS)
        {
            gsi_cred_handle = cred_desc->cred_handle;
            if (globus_gsi_cred_get_cert_chain(gsi_cred_handle, &px509_chain) == GLOBUS_SUCCESS)
            {
                globus_module_deactivate(GLOBUS_GSI_CREDENTIAL_MODULE);
                return px509_chain;
            }
            else
            {
                globus_module_deactivate(GLOBUS_GSI_CREDENTIAL_MODULE);
                return NULL;
            }
        }
        else
        {
            globus_module_deactivate(GLOBUS_GSI_CREDENTIAL_MODULE);
            return NULL;
        }
    }
    else
    {
        return NULL;
    }
}

#endif /* LCAS_GSI_UTILS_C */

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcas/src/grid_credential_handling/gsi_handling/lcas_gsi_utils.c,v $
    $Date: 2010-05-03 10:42:50 $
    $Revision: 1.2 $
    $Author: okoeroo $
******************************************************************************/
