import QtQuick 2.0
import Ubuntu.Components 0.1
import Friends 0.2

// Field to enter a new messages
// Used by the PostPage and the ReplyArea
Item {
    id: entryField

    onVisibleChanged: {
        if(visible)
            entryArea.forceActiveFocus()
    }

    property var lastCursorPosition
    property bool hideable

    property int queueSize: 0

    // Send done will be emited when all messages have been send successfully
    signal sendDone

    property bool failed: false

    onFailedChanged: {
        if (failed) {
            PopupUtils.open(errorPopover, sendButton);
        }
    }

    property var accountIds
    property var replyToId

    function setText(text) {
        entryArea.text = text + " "
        entryArea.cursorPosition = text.length + 1
    }

    TextArea {
        id: entryArea
        anchors {
            top: parent.top
            bottom: parent.bottom
            left: parent.left
            leftMargin: units.gu(1)
            right: sendButton.left
            rightMargin: units.gu(1)
        }

        //FIXME this is should depend on the service
        property int maxMessageLength: 140

        // send message if return (without shift) was pressed
        // otherwise we enter a newline
        Keys.onReturnPressed: {
            if(event.modifiers & Qt.ShiftModifier)
                entryArea.insert(cursorPosition, "\n")
            else
                sendButton.clicked(null)
        }

        Keys.onEscapePressed: text = ""
        height: units.gu(4)
        placeholderText: i18n.tr("Reply")
        autoSize: true
        maximumLineCount: 0
        color: text.length < maxMessageLength ? "gray" : "red"
        textFormat: TextEdit.PlainText
        contentWidth: width - units.gu(5)

        onTextChanged: {
            var enteredText = text.substring(entryField.lastCursorPosition, cursorPosition);
            if (enteredText.substring(0,4) == "http") {
                var shortUrl = friends.urlShorten(enteredText);
                if (shortUrl.length > 4) {
                    text = text.replace (enteredText, shortUrl);
                    cursorPosition = text.length;
                }
            }
            entryField.lastCursorPosition = cursorPosition;
        }

        Item {
            id: counter
            anchors {
                right: entryArea.right
                top: entryArea.top
                rightMargin: units.gu(1)
            }
            height: units.gu(4)
            width: childrenRect.width

            visible: (entryArea.text.length > 0) ? true : false

            Text {
                text: entryArea.maxMessageLength - entryArea.text.length
                font.bold: true
                opacity: 0.3
            }
        }
    }

    Button {
        id: sendButton

        anchors {
            right: parent.right
            rightMargin: units.gu(1)
            top: parent.top
        }

        width: units.gu(9)
        height: units.gu(4)
        text: i18n.tr("Send")
        color: main.headerColor
        onClicked: {
            if(replyToId == null)
            {
                for(var i = 0; i < accountIds.length; ++i)
                {
                    console.log ("Posted new message from " + accountIds[i] + ": " + entryArea.text);

                    sendSpinner.visible = true
                    queueSize++;
                    friends.sendForAccountAsync(accountIds[i], entryArea.text);
                }
            }
            else
            {
                console.log ("Posted new reply to " + replyToId + " from " + accountIds[0] + ": " + entryArea.text);

                sendSpinner.visible = true
                queueSize = 1

                // Reply currenlty does only make sense for one account at a time
                friends.sendReplyAsync(accountId[0], replyToId, entryArea.text);
            }
        }
    }
    Button {
        id: clearButton

        visible: entryField.hideable

        anchors {
            right: parent.right
            top: sendButton.bottom
            rightMargin: units.gu(1)
            topMargin: units.gu(1)
        }

        height: sendButton.height
        width: sendButton.width
        iconSource: "images/clear-search.png"
        color: main.headerColor

        onClicked: {
            if(!entryField.hideable)
                return

            entryArea.text = ""
            entryField.visible = false
        }
    }

    ActivityIndicator {
        id: sendSpinner
        anchors.centerIn: entryArea
        visible: false
        running: visible
    }

    FriendsDispatcher {
        id: friends

        onSendComplete: {
            if (success) {
                console.log ("Send completed successfully");
            } else {
                console.log ("Send failed: " + errorMessage.split("str: str:")[1]);
                failed = true;
            }
            queueSize--;

            if (queueSize == 0 && failed == false) {
                sendSpinner.visible = false;
                entryArea.text = "";

                sendDone()
            }
        }
    }
}
