/****************************************************************************
 *    app/options.cpp - This file is part of coala							*
 *																			*
 *    Copyright (C) 2009-2010  Torsten Grote								*
 *																			*
 *    This program is free software; you can redistribute it and/or modify	*
 *    it under the terms of the GNU General Public License as published by	*
 *    the Free Software Foundation; either version 3 of the License, or		*
 *    (at your option) any later version.									*
 *																			*
 *    This program is distributed in the hope that it will be useful,		*
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of		*
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *    GNU General Public License for more details.							*
 *																			*
 *    You should have received a copy of the GNU General Public License		*
 *    along with this program; if not, see http://www.gnu.org/licenses		*
 ****************************************************************************/

#include "options.h"
#include "program_opts/value.h"  

using namespace ProgramOptions;
using namespace std;

namespace Coala {

Options::Options() { }

Options::~Options() { }

void Options::setDefaults() {
	language		= "c";
	fake_class_neg	= false;
	debug			= 0;
	no_direct_enc	= false;
	incremental		= false;
	reverse			= false;
	no_where_check	= false;
	version			= false;
	help			= false;
}

void Options::initOptions(ProgramOptions::OptionGroup& allOpts, ProgramOptions::OptionGroup& hidden) {
	allOpts.addOptions()
		("language,l"		, value<string>(&language),		"the action language (c, c_taid, b or m) to be used as input (default: c)", "<LANG>")
		("incremental,i"	, bool_switch(&incremental),	"output incremental encoding (does not work with -l c_taid)")
		("reverse,r"		, bool_switch(&reverse),		"output reverse encoding (does not work with -l c_taid)")
		("debug,d"			, value<int>(&debug),			"debug program by showing more atoms (default: 0)", "<num>")
		("no-direct-enc,e"	, bool_switch(&no_direct_enc),	"don't output direct encoding to ASP (does not work with -l c_taid)")
		("no-where,w"		, bool_switch(&no_where_check),	"do not check for correct <where> clauses")
		("neg,n"			, bool_switch(&fake_class_neg),	"simulate classical negation instead of using the built-in one")
		("version,v"		, bool_switch(&version),		"print version information and exit")
		("help,h"			, bool_switch(&help),			"print help information and exit")
	;
	hidden.addOptions()
		("file,f"		, value<vector<string> >(&input)->setComposing(),	"Read from <files> instead of stdin", "<files>")
	;
}

bool Options::parse(int argc, char** argv, std::ostream& os) {
	setDefaults();
	OptionValues values;
	error_.clear();
	try {
		OptionGroup allOpts, visible, hidden;
		initOptions(visible, hidden);
		allOpts.addOptions(visible).addOptions(hidden);
		warning_.clear();
		error_.clear();
		values.store(parseCommandLine(argc, argv, allOpts, false, "file", &input));
		if (help) { 
			printHelp(visible, os);
			return true;
		} else if (version) {
			printVersion(visible, os);
			return true;
		}
		checkCommonOptions(values);
		os << error_;
		return true;
	}
	catch(const std::exception& e) {
		error_ = e.what();
		return false;
	}
	return true;
}

void Options::checkCommonOptions(const OptionValues& vm) {
	// set language "al" as synonym for "b"
	if(language == "al") language = "b";

	if(incremental && language == "c_taid") {
		throw std::runtime_error("The incremental encoding does not work with action language C_taid.");
	}
	
	if(reverse && (language == "b" || language == "c_taid")) {
		throw std::runtime_error("The reverse encoding does not work with this action language");
	}

	if(fake_class_neg && no_direct_enc && language != "b") {
		warning_ += "Hint: Classical negation does not need to be simulated with non-direct encoding.\n";
		fake_class_neg = false;
	}
	
	if(reverse && !incremental) {
		warning_ += "Reverse encoding needs to be incremental. The -i parameter was activated.\n";
		incremental = true;
	}
	
	if(no_direct_enc && (language == "m" || language == "c_taid")) {
		throw std::runtime_error("The non-direct encoding does not work with this action language for now.");
	}
	
	if(reverse && !no_direct_enc) {
		throw std::runtime_error("The reverse encoding can not be direct for now.");
	}
}

void Options::printHelp(const OptionGroup& opts, std::ostream& os) const {
	os << "coala " << VERSION << " Copyright (C) 2007-2010  Torsten Grote\n";
	os << "This program comes with ABSOLUTELY NO WARRANTY. It is free software, and you\n";
	os << "are welcome to redistribute it under certain conditions.\n";
	os << "\n";
	os << "Usage: coala.bin [-l|--language=<LANG>] [-i|--incremental] [-r|--reverse] [-d|--debug=<num>]\n";
	os << "                 [-e|--no-direct-enc] [-w|--no-where] [-n|--neg] [--help] [<files>]\n";
	os << "\n";
	os << opts;
	os << "\n";
}

void Options::printVersion(const OptionGroup& opts, std::ostream& os) const {
	os << "coala " << VERSION << std::endl;
	os << std::endl;
	os << "Copyright (C) 2007-2010  Torsten Grote" << std::endl;
	os << "This program comes with ABSOLUTELY NO WARRANTY. It is free software, and you" << std::endl;
	os << "are welcome to redistribute it under certain conditions." << std::endl;
}

}
