/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2025 Univ. Grenoble Alpes, CNRS, Grenoble INP - UGA, TIMC, 38000 Grenoble, France
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef __HOTPLUG_ACTION__
#define __HOTPLUG_ACTION__

#include "Action.h"

class VariantDataModel;

#include "HotPlugActionExtension.h"
namespace camitk {

/**
 * @brief An Action that can be created on the fly
 *
 */
class CAMITK_API HotPlugAction : public Action {
    Q_OBJECT

public:
    /// Default Constructor
    HotPlugAction(HotPlugActionExtension* actionExtension, const VariantDataModel& data);

    /// Default Destructor
    virtual ~HotPlugAction() = default;

    /// calls user-defined init()
    /// @return true if the initialization went well
    virtual bool init() = 0;

    /// Calls user-defined targetDefined() and getUI()
    virtual QWidget* getWidget() override = 0;

    /// manage property change immediately
    virtual bool event(QEvent* e) override;

    /// if this method returns true, update() should be called.
    /// To be reimplemented in subclass for language (i.e., C++) that
    /// needs to watch the user source code
    /// @return false unless reimplemented
    virtual bool needsUpdate();

    /// update the action from the source code.
    /// To be reimplemented in subclass for language (i.e., C++) that
    /// needs to watch the user source code and rebuild the code
    /// By default does nothing and returns true
    virtual bool update();

public slots:
    /** this method is automatically called when the action is triggered.
      * Call getTargets() method to get the list of components to use.
      * \note getTargets() is automatically filtered so that it only contains compatible components,
      * i.e., instances of "$componentClass$" (or a subclass).
      */
    virtual Action::ApplyStatus apply() override = 0;

protected:
    /// called when a parameter was changed by an event
    /// reimplemented in inherited class
    virtual void parameterChangedEvent(QString parameterName) = 0;

    /// where the action is managed
    HotPlugActionExtension* hotPlugExtension;

private:
    /// do not manage event during initialization
    bool initializationPending;

    /// create all the parameters from the data model
    /// @param parameters is the "parameters" of the action
    void createParameters(VariantDataModel parameters);

    /// returns a new Property* (action parameter) corresponding to the given data model
    /// @param parameter is the dictionary with all the information about one parameter
    Property* createParameter(VariantDataModel parameterData);

    /// use the parameterData to set attributes of the given parameter
    /// managed attributes are: group, minimum, maximum, singleStep, decimals
    /// regExp, readOnly and enumValues
    /// Note that the parameter type is not checked (apart for the attribute enumValues,
    /// a valid data model is assumed)
    void setParameterAttributes(Property* parameter, VariantDataModel parameterData);
};

} // namespace camitk

#endif // __HOTPLUG_ACTION__
