/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2016 Univ. Grenoble Alpes, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef MESHCOMPONENT_H
#define MESHCOMPONENT_H

#include "Component.h"
#include "Geometry.h"
#include "MeshDataModel.h"
#include "MeshDataView.h"
#include "MeshSelectionModel.h"
#include "MeshSelectionView.h"

class vtkPointSet;
class vtkSelection;
class vtkSelectionNode;

class QTableView;
class QComboBox;

namespace camitk {

/**
 * @ingroup group_sdk_libraries_core_component_mesh
 *
 * @brief
 * Basic component to manage any kind of mesh.
 */
class CAMITK_API MeshComponent : public Component {
    Q_OBJECT
public:

    /** Creates a top-level MeshComponent from a file.
     *  \note this is only to be used from a Component Extension open(...) or from an Action that creates data from a filter or transformation of a vtkPointSet.
     *
     *  Please consider using MeshComponent(vtkSmartPointer<vtkPointSet> , const QString &)
     */
    MeshComponent ( const QString & file ) throw ( AbortException );

    /** Creates a top-level MeshComponent from a vtkPointSet (and instanciate its 3D representation).
     *
     * \note the filename is set to the empty string "".
     *
     * \note if aPointSet is NULL, the representation is not initialized, it is the responsability of the subclass to initialize it later
     *
     * @param aPointSet point set of the new MeshComponent
     * @param name name to be given to the Component
     */
    MeshComponent ( vtkSmartPointer<vtkPointSet> aPointSet, const QString &name );

    /** Creates a MeshComponent as a sub component of another Component using a vtkPointSet (and instanciate its 3D representation).
     *
     * \note if aPointSet is NULL, the representation is not initialized, it is the responsability of the subclass to initialize it later
     *
     * @param parentComponent the parent component of the new MeshComponent
     * @param aPointSet point set of the new MeshComponent
     * @param name name to be given to the new MeshComponent
     */
    MeshComponent ( Component *parentComponent, vtkSmartPointer<vtkPointSet> aPointSet, const QString &name );

    /// Destructor
    virtual ~MeshComponent();

    /// reimplemented to save the last pick point id
    void pointPicked ( vtkIdType pointId, bool );

    /// reimplemented to save the last pick point id
    void cellPicked ( vtkIdType cellId, bool );

    /// get the last pick point id, @return -1 if no point where picked
    vtkIdType getPickedPointId();

    /// get the last pick point id, @return -1 if no point where picked
    vtkIdType getPickedCellId();

    /// update property: no specific properties to manage, this is needed to avoid console warnings.
    void updateProperty ( QString , QVariant ) {}

    /// there is more than one property widgets (to view as tabs in the property explorer)
    virtual unsigned int getNumberOfPropertyWidget();

    /// the proprety widgets are: default property widget, selection view and data view
    virtual QWidget * getPropertyWidgetAt ( unsigned int i, QWidget* parent = 0 );

    /// Get the pixmap that will be displayed for this node
    virtual QPixmap getIcon();

    /**
     * @name Selection management
     * @see http://www.kitware.com/media/html/SelectionsInVTK.html
     *
     * The selection are stored using a list of vtkSelectionNode.
     * Since this list is managed through the a Qt model/view,
     * please do not modify directly the vtkSelectionNode and use
     * the methods dedicated to selections.
     *
     * @{
     */

    /**
     * @brief Get the selection list.
     *
     * The selection list contains vtkSelectionNode.
     *
     * \note TODO This method should be const, in case a subcomponent needs to add extra/specific behavior.
     * And then:
     * - To loop over selections, use getNumberOfSelections() and getSelectionAt().
     * - To remove selection, use removeSelection().
     * - To add selections, use addSelection() or addToSelectedSelection().
     *
     * @return the selection list
     */
    QList< vtkSmartPointer<vtkSelectionNode> >& getSelections();

    /**
     * @brief Get the number of selections.
     *
     * @return the number of selections.
     */
    unsigned int getNumberOfSelections() const;

    /**
     * @brief Get active selections
     *
     * @return active selections
     */
    vtkSmartPointer<vtkSelection> getActiveSelection() const;

    /**
     * @brief Get a selection from its name.
     *
     * @TODO mettre en const pour pas de modeif externe
     *
     * @param name name of the selection
     * @return the selection node
     */
    vtkSmartPointer<vtkSelectionNode> getSelection ( const QString& name ) const;

    /**
     * @brief Get a selection from its index in the list.
     *
     * @TODO mettre en const pour pas de modeif externe
     *
     * @param index index of the selection
     * @return the selection node
     */
    vtkSmartPointer<vtkSelectionNode> getSelectionAt ( unsigned int index ) const;

    /**
     * @brief Get the selection index in the list from its name.
     *
     * @param name name of the selection
     * @return the selection index or -1 if there is no selection of that name
     */
    int getSelectionIndex ( const QString& name ) const;

    /**
     * @brief Add a selection.
     *
     * If the name of the selection already exists, the data of the existing
     * selection are updated according to the SelectionPolicy flag.
     *
     * @param name name of the selection
     * @param fieldType field type of the selection (one of vtkSelectionNode::SelectionField)
     * @param contentType content type (one of vtkSelectionNode::SelectionContent)
     * @param array array of the selection
     * @param policy policy to update the existing selection
     * @return the index of the added selection in the selection list
     */
    virtual int addSelection ( const QString& name, int fieldType, int contentType,  vtkSmartPointer< vtkAbstractArray > array, MeshSelectionModel::InsertionPolicy policy = MeshSelectionModel::REPLACE );

    /**
     * @brief Add a selection to the currently selected selection.
     *
     * If there is no selected selection, one is created with the name "Picked Selection".
     *
     * @param fieldType field type of the selection (one of vtkSelectionNode::SelectionField)
     * @param contentType content type (one of vtkSelectionNode::SelectionContent)
     * @param array array of the selection
     * @param policy policy to updated the selection selection
     * @return the index of the selected selection in the selection list
     */
    virtual int addToSelectedSelection ( int fieldType, int contentType,  vtkSmartPointer< vtkAbstractArray > array, MeshSelectionModel::InsertionPolicy policy = MeshSelectionModel::REPLACE );

    ///@cond
    /// TODO CAMITK_DEPRECATED. This section list all the methods marked as deprecated. They are to be removed in CamiTK 4.0
    /// This method was only available in CamiTK 3.4.0
    /// Please use the getAllSelections method instead.
    /**
      * Get the selection list
      */
    const QMap< QString, vtkSmartPointer<vtkSelectionNode> >& getSelectionList() const;
    ///@endcond


    /**
     * @}
     */

    /**
     * @name Data management
     * @{
     */

    /**
     * @brief Get the data array of specified field type and name.
     *
     * @param fieldType field type (@see MeshDataModel::FieldType)
     * @param arrayName array name
     *
     * @return data array
     */
    vtkSmartPointer<vtkDataArray> getDataArray(MeshDataModel::FieldType fieldType, const QString& arrayName);

    /**
     * @brief Get the data array of specified field type and index.
     *
     * @param fieldType field type (@see MeshDataModel::FieldType)
     * @param index index
     *
     * @return data array
     */
    vtkSmartPointer<vtkDataArray> getDataArray(MeshDataModel::FieldType fieldType, int index);

    /**
     * @brief Add a data array.
     *
     * @param fieldType field type (@see MeshDataModel::FieldType)
     * @param arrayName array name
     * @param data data array
     */
    void addDataArray ( MeshDataModel::FieldType fieldType, const QString& name, vtkSmartPointer<vtkDataArray> data );


    /**
     * @brief Remove a data array.
     *
     * @param fieldType field type (@see MeshDataModel::FieldType)
     * @param arrayName array name
     */
    void removeDataArray ( MeshDataModel::FieldType fieldType, const QString& name );

    /**
     * @brief Add a data array linked to the points.
     *
     * @param name name
     * @param data data array
     */
    void addPointData ( const QString& name, vtkSmartPointer<vtkDataArray> data );

    /**
     * @brief Add a data array linked to the cells.
     *
     * @param name name
     * @param data data array
     */
    void addCellData ( const QString& name, vtkSmartPointer<vtkDataArray> data );

    /**
     * @brief Get the prop associed to the data.
     *
     * @param fieldType field type (@see MeshDataModel::FieldType)
     * @param name data array name
     *
     * @return prop
     */
    vtkSmartPointer<vtkProp> getDataProp ( MeshDataModel::FieldType fieldType, const QString& name );

    /** Set the given named scalar data array corresponding to the given field type as the currently active data.
     *
     * This will set the current active data. If the data has one dimension (scalar),
     * it will also show the corresponding values using the
     * default color scale and display the color scale in the 3D viewer.
     *
     * \note if fieldType is POINTS and name is equal to NULL, the color scale is removed from the 3D viewer
     *
     * @param fieldType field type (@see MeshDataModel::FieldType)
     * @param name data array name
     */
    void setActiveData(MeshDataModel::FieldType fieldType, const char *name);

    /// get the current data view model (model as the M in Qt MVC design pattern)
    MeshDataModel* getDataModel() {
        return dataModel;
    }
    /**
     * @}
     */

protected:

    /// build the instance of Geometry from the given vtkPointSet
    virtual void initRepresentation ( vtkSmartPointer<vtkPointSet> );

    /// initialize selections
    virtual void initSelection();

    /// initialize data
    virtual void initData();

    /// create and initialize dynamic properties
    virtual void initDynamicProperties();

protected slots:

    /// called when the selection is modified
    void changeSelectedSelection ( const QItemSelection &selected, const QItemSelection &deselected );

    /// remove the selected selection
    void removeSelectedSelections();

    /// remove the selected selection
    void removeSelectedData();

private:

    /// the concrete building of the 3D objects (Slice/Geometry): none in this case, everything is done by initRepresentation(vtkPointSet)
    virtual void initRepresentation() {}

    /// initialisation of the mesh component members
    void init();

    /// the last picked point
    vtkIdType pickedPointId;

    /// the last picked cell
    vtkIdType pickedCellId;

    /// manages current selection using vtk
    vtkSmartPointer<vtkSelection> currentSelection;

    /// list of selections
    QList< vtkSmartPointer<vtkSelectionNode> > selectionList;

    ///@cond
    /// TODO CAMITK_DEPRECATED. This section list all the methods marked as deprecated. They are to be removed in CamiTK 4.0
    /// When the method getSelectionList is removed (CamiTK 4.0), please remove this attribute.
    ///
    /// map of selections
    QMap< QString, vtkSmartPointer<vtkSelectionNode> > selectionMap;
    ///@endcond

    /// selection model (model as the M in Qt MVC design pattern)
    MeshSelectionModel* selectionModel;

    /// selection GUI View (view as the V in Qt MVC design pattern)
    MeshSelectionView* selectionView;

    /// selection widget
    QWidget* selectionWidget;

    /// action to remove selections
    QAction* removeSelections;

    /// action to merge selections
    QAction* mergeSelection;

    /// action to inspect selection
    QAction* inspectSelection;

    /// combo box to select the selection insertion policy
    QComboBox* policyBox;

    /// data model (model as the M in Qt MVC design pattern)
    MeshDataModel* dataModel;

    /// data GUI View (view as the V in Qt MVC design pattern)
    MeshDataView* dataView;

    /// selection widget
    QWidget* dataWidget;

    /// action to remove data
    QAction* removeData;

    /// action to inspect data
    QAction* inspectData;

};

}

#endif // MESHCOMPONENT_H
