
/*
 Copyright 2013, Kovid Goyal <kovid at kovidgoyal.net>
 Released under the GPLv3 License
*/


(function() {
  var PreviewIntegration, find_containing_block, in_table, is_block, is_hidden, log,
    __bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  if (typeof window !== "undefined" && window !== null ? window.calibre_utils : void 0) {
    log = window.calibre_utils.log;
  }

  is_hidden = function(elem) {
    while (elem) {
      if (elem.style && (elem.style.visibility === 'hidden' || elem.style.display === 'none')) {
        return true;
      }
      elem = elem.parentNode;
    }
    return false;
  };

  is_block = function(elem) {
    var style, _ref;
    style = window.getComputedStyle(elem);
    return (_ref = style.display) === 'block' || _ref === 'flex-box' || _ref === 'box';
  };

  in_table = function(elem) {
    var _ref;
    while (elem) {
      if (((_ref = elem.tagName) != null ? _ref.toLowerCase() : void 0) === 'table') {
        return true;
      }
      elem = elem.parentNode;
    }
    return false;
  };

  find_containing_block = function(elem) {
    while (elem && elem.getAttribute('data-is-block') !== '1') {
      elem = elem.parentNode;
    }
    return elem;
  };

  PreviewIntegration = (function() {

    PreviewIntegration.name = 'PreviewIntegration';

    /*
        # Namespace to expose all the functions used for integration with the Tweak
        # Book Preview Panel.
    */


    function PreviewIntegration() {
      this.go_to_anchor = __bind(this.go_to_anchor, this);

      this.onclick = __bind(this.onclick, this);

      this.onload = __bind(this.onload, this);

      this.report_split = __bind(this.report_split, this);

      this.split_mode = __bind(this.split_mode, this);

      this.find_blocks = __bind(this.find_blocks, this);

      this.line_numbers = __bind(this.line_numbers, this);

      this.go_to_line = __bind(this.go_to_line, this);
      if (!this instanceof arguments.callee) {
        throw new Error('PreviewIntegration constructor called as function');
      }
      this.blocks_found = false;
      this.in_split_mode = false;
    }

    PreviewIntegration.prototype.go_to_line = function(lnum) {
      var node, _i, _len, _ref;
      _ref = document.querySelectorAll('[data-lnum="' + lnum + '"]');
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        node = _ref[_i];
        if (is_hidden(node)) {
          continue;
        }
        if (node === document.body) {
          window.scrollTo(0, 0);
        } else {
          node.scrollIntoView();
        }
        return;
      }
    };

    PreviewIntegration.prototype.line_numbers = function() {
      var ans, found_body, node, _i, _len, _ref;
      found_body = false;
      ans = [];
      _ref = document.getElementsByTagName('*');
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        node = _ref[_i];
        if (!found_body && node.tagName.toLowerCase() === "body") {
          found_body = true;
        }
        if (found_body) {
          ans.push(node.getAttribute("data-lnum"));
        }
      }
      return ans;
    };

    PreviewIntegration.prototype.find_blocks = function() {
      var elem, _i, _len, _ref;
      if (this.blocks_found) {
        return;
      }
      _ref = document.body.getElementsByTagName('*');
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        elem = _ref[_i];
        if (is_block(elem) && !in_table(elem)) {
          elem.setAttribute('data-is-block', '1');
        }
      }
      return this.blocks_found = true;
    };

    PreviewIntegration.prototype.split_mode = function(enabled) {
      this.in_split_mode = enabled;
      document.body.setAttribute('data-in-split-mode', enabled ? '1' : '0');
      if (enabled) {
        return this.find_blocks();
      }
    };

    PreviewIntegration.prototype.report_split = function(node) {
      var loc, num, parent, sibling, totals;
      loc = [];
      totals = [];
      parent = find_containing_block(node);
      while (parent && parent.tagName.toLowerCase() !== 'body') {
        totals.push(parent.parentNode.children.length);
        num = 0;
        sibling = parent.previousElementSibling;
        while (sibling) {
          num += 1;
          sibling = sibling.previousElementSibling;
        }
        loc.push(num);
        parent = parent.parentNode;
      }
      loc.reverse();
      totals.reverse();
      return window.py_bridge.request_split(JSON.stringify(loc), JSON.stringify(totals));
    };

    PreviewIntegration.prototype.onload = function() {
      return window.document.body.addEventListener('click', this.onclick, true);
    };

    PreviewIntegration.prototype.onclick = function(event) {
      var e, href, lnum, tn, _ref;
      event.preventDefault();
      if (this.in_split_mode) {
        this.report_split(event.target);
      } else {
        e = event.target;
        lnum = e.getAttribute('data-lnum');
        href = tn = '';
        while (e && e !== document.body && e !== document && (tn !== 'a' || !href)) {
          tn = (_ref = e.tagName) != null ? _ref.toLowerCase() : void 0;
          href = e.getAttribute('href');
          e = e.parentNode;
        }
        window.py_bridge.request_sync(tn, href, lnum);
      }
      return false;
    };

    PreviewIntegration.prototype.go_to_anchor = function(anchor, lnum) {
      var elem;
      elem = document.getElementById(anchor);
      if (!elem) {
        elem = document.querySelector('[name="' + anchor + '"]');
      }
      if (elem) {
        elem.scrollIntoView();
        lnum = elem.getAttribute('data-lnum');
      }
      return window.py_bridge.request_sync('', '', lnum);
    };

    return PreviewIntegration;

  })();

  window.calibre_preview_integration = new PreviewIntegration();

  window.onload = window.calibre_preview_integration.onload;

}).call(this);

