// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/ErrorCollector.h"
#include <string.h>

namespace Puma {


ErrorCollector::Msg::Msg (const ErrorSeverity &s, Location l, const char *m) :
  _sev (s), _loc (l), _have_loc (true) {
  _msg = new char[strlen (m) + 1];
  strcpy (_msg, m);
}
   
ErrorCollector::Msg::Msg (const ErrorSeverity &s, const char *m) :
  _sev (s), _have_loc (false) {
  _msg = new char[strlen (m) + 1];
  strcpy (_msg, m);
}

ErrorCollector::Msg::~Msg () {
  delete[] _msg; 
}

bool ErrorCollector::Msg::operator == (const Msg& msg) const {
  if (_have_loc != msg._have_loc)
    return false;
  if (_have_loc)
    if (_loc != msg._loc)
      return false;
  if (_sev != msg._sev)
    return false;
  if (strcmp (_msg, msg._msg) != 0)
    return false;
  return true;
}

void ErrorCollector::Msg::Shift (ErrorSink &e) {
  if (_have_loc)
    e << _loc;
  e << _sev << _msg << Puma::endMessage;
}

ErrorCollector& ErrorCollector::operator << (const char *v) {
  _msg << v;
  return *this;
}
 
ErrorCollector& ErrorCollector::operator << (long v) {
  _msg << v;
  return *this;
}
 
ErrorCollector& ErrorCollector::operator << (unsigned v) {
  _msg << v;
  return *this;
}
 
ErrorCollector& ErrorCollector::operator << (int v) {
  _msg << v;
  return *this;
}
 
ErrorCollector& ErrorCollector::operator << (short v) {
  _msg << v;
  return *this;
}
 
ErrorCollector& ErrorCollector::operator << (char v) {
  _msg << v;
  return *this;
}
 
ErrorCollector& ErrorCollector::operator << (double v) {
  _msg << v;
  return *this;
}

ErrorCollector& ErrorCollector::operator << (const ErrorSeverity& sev) {
  if (_sev < sev)
    _sev = sev;
  return *this;
}
 
ErrorCollector& ErrorCollector::operator << (Location v) {
  _loc = v;
  _have_loc = true;
  return *this;
}
 
ErrorCollector& ErrorCollector::operator << (const Printable& v) {
  _msg << v;
  return *this;
}
 
ErrorCollector& ErrorCollector::operator << (void (*f)(ErrorSink&)) {
  (*f) (*(ErrorSink*)this);
  return *this;
}
 
void ErrorCollector::endMessage () {
  Msg *msg;
  if (_have_loc)
    msg = new Msg (_sev, _loc, _msg.str ().c_str ());
  else
    msg = new Msg (_sev, _msg.str ().c_str ());
    
  bool found = false;
  for (int m = 0; m < _msgs.length (); m++)
    if (*_msgs.lookup (m) == *msg)
      found = true;
  
  if (! found)
    _msgs.append (msg);
  else
    delete msg;
  
  clear ();
}

void ErrorCollector::clear () {
  _sev = sev_none;
  _have_loc = false;
  _msg.str ("");
  _msg.clear ();
}

int ErrorCollector::index () {
  return (int)_msgs.length ();
}

void ErrorCollector::index (int n) {
  while (_msgs.length () > n) {
    delete _msgs.lookup (_msgs.length () - 1);
    _msgs.remove (_msgs.length () - 1);
  }
}

void ErrorCollector::Shift (ErrorSink &e) {
  for (int m = 0; m < _msgs.length (); m++)
    _msgs.lookup (m)->Shift (e);
  index (0);
}

const ErrorSeverity &ErrorCollector::severity () const { 
  const ErrorSeverity *max_sev = &sev_none;
  for (int m = 0; m < _msgs.length (); m++) 
    if (*max_sev < _msgs.lookup (m)->severity ())
      max_sev = &_msgs.lookup (m)->severity ();
  return *max_sev; 
}


} // namespace Puma
