#
# Copyright 2001 by Object Craft P/L, Melbourne, Australia.
#
# LICENCE - see LICENCE file distributed with this software for details.
#
# Stuff shared by all albatross modules - must not import any other
# albatross modules, or cycles will result.

# Base exception class used by all Albatross exceptions
class AlbatrossError(Exception):
    pass


# Generic causes of exceptions:
class UserError(AlbatrossError):
    pass


class ApplicationError(AlbatrossError):
    pass


class InternalError(AlbatrossError):
    pass


class ServerError(AlbatrossError):
    pass


# Specific exceptions:
class SecurityError(UserError):
    pass


class TemplateLoadError(ApplicationError):
    pass

class SessionExpired(UserError):
    pass

# HTTP/1.0 Status definitions from RFC1945
HTTP_OK                           = 200
HTTP_CREATED                      = 201
HTTP_ACCEPTED                     = 202
HTTP_NO_CONTENT                   = 204
HTTP_MULTIPLE_CHOICES             = 300
HTTP_MOVED_PERMANENTLY            = 301
HTTP_MOVED_TEMPORARILY            = 302
HTTP_NOT_MODIFIED                 = 304
HTTP_BAD_REQUEST                  = 400
HTTP_UNAUTHORIZED                 = 401
HTTP_FORBIDDEN                    = 403
HTTP_NOT_FOUND                    = 404
HTTP_INTERNAL_SERVER_ERROR        = 500
HTTP_NOT_IMPLEMENTED              = 501
HTTP_BAD_GATEWAY                  = 502
HTTP_SERVICE_UNAVAILABLE          = 503

def http_status_informational(status):
    return str(status).startswith('1')

def http_status_successful(status):
    return str(status).startswith('2')

def http_status_redirection(status):
    return str(status).startswith('3')

def http_status_client_error(status):
    return str(status).startswith('4')

def http_status_server_error(status):
    return str(status).startswith('5')

http_status_phrases = {
    HTTP_OK                       : "Ok",
    HTTP_CREATED                  : "Created",
    HTTP_ACCEPTED                 : "Accepted",
    HTTP_NO_CONTENT               : "No Content",
    HTTP_MULTIPLE_CHOICES         : "Multiple Choices",
    HTTP_MOVED_PERMANENTLY        : "Moved Permanently",
    HTTP_MOVED_TEMPORARILY        : "Moved Temporarily",
    HTTP_NOT_MODIFIED             : "Not Modified",
    HTTP_BAD_REQUEST              : "Bad Request",
    HTTP_UNAUTHORIZED             : "Unauthorized",
    HTTP_FORBIDDEN                : "Forbidden",
    HTTP_NOT_FOUND                : "Not Found",
    HTTP_INTERNAL_SERVER_ERROR    : "Internal Server Error",
    HTTP_NOT_IMPLEMENTED          : "Not Implemented",
    HTTP_BAD_GATEWAY              : "Bad Gateway",
    HTTP_SERVICE_UNAVAILABLE      : "Service Unavailable",
}

def http_status_string(status):
    try:
        return '%s %s' % (status, http_status_phrases[status])
    except KeyError:
        return '%s' % status

def commasplit(text):
    """ split a string into comma separated fields """
    return [f.strip() for f in text.split(',')]
