/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2025 Univ. Grenoble Alpes, CNRS, Grenoble INP - UGA, TIMC, 38000 Grenoble, France
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef ACTIONWIDGET_H
#define ACTIONWIDGET_H

// -- Core stuff
#include "CamiTKAPI.h"
#include "Action.h"

//-- Qt
#include <QFrame>
#include <QPushButton>

namespace camitk {

// forward declaration
class ObjectController;

/**
 * @ingroup group_sdk_libraries_core_action
 *
 * @brief
 * Build a default widget for a given action using its Qt properties.
*  It should be good enough in most of the case.

*  The default widget contains:
*  - a title with the action icon, name and an information button
*  - the parameter ObjectController that allows you to edit/modify all the action parameters
*  - apply and revert buttons (the button frame)
*
*  The information button shows a dialog with the action classification,
*  a description, the list of parameters and their properties, and the current target component names
*  (inputs),
*
*  The button frame contains an "Apply" and a "Revert" button. The Apply button updates
*  the action properties using the value entered by the user in the GUI. I.e., its clicked()
*  signal is connected the the action apply() slot.
*  The revert button resets the values in GUI to the initial values (all modification in the GUI
*  are cancelled).
*  You can easily show or hide the button frame using setButtonVisibility().
*  You can also customize the "Apply" button text using setApplyButtonText().
*
*  update() updates the widget using the actual/current properties and target of the action.
*  This method should be called by an action in the overriden getWidget method.
*  A typical getWidget() method of an action should use the lazy instantiation pattern to instantiate
*  ActionWidget the first time it is called, and call the ActionWidget instance update() method
*  for any subsequent calls.
*
*  If you do not want to wait for the user to click on the apply button (or if this button is hidden)
*  and you want the value of the action's property/parameter to be updated everytime
*  there is a change in the GUI, call setAutoUpdateProperty(true)
*  (default is false);
*
*/
class CAMITK_API ActionWidget : public QFrame {
public:
    /// create a default action widget for the given action
    ActionWidget(Action*);

    ~ActionWidget() override;

    /// update display (target and property controller)
    void update();

    /// if false then the apply/revert buttons are shown
    void setButtonVisibility(bool);

    /// modify the "Apply" button text
    void setApplyButtonText(QString);

    /// automatically update the action properties whenever there is a change in the GUI
    void setAutoUpdateProperty(bool autoUpdate);

public slots:

    /// show help dialog
    void showInfo();

protected:
    /// return a text with the target component names followed by their class names
    QString getTargetLabel();

private:
    /// the managed action
    Action* myAction;

    /// the object controller with the properties of the default widget
    ObjectController* objectController;

    /// this frame contains the Apply/Revert buttons and can be hide using setButtonVisibility()
    QFrame* buttonFrame;

    /// The apply button text may be modified using setApplyButtonText()
    QPushButton* applyButton;
};

}
#endif // ACTIONWIDGET_H
